/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.s3a;

import com.amazonaws.AmazonClientException;
import com.amazonaws.services.s3.model.S3ObjectSummary;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.ListIterator;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Set;
import javax.annotation.Nullable;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathFilter;
import org.apache.hadoop.fs.RemoteIterator;
import org.apache.hadoop.fs.s3a.S3AFileStatus;
import org.apache.hadoop.fs.s3a.S3AFileSystem;
import org.apache.hadoop.fs.s3a.S3ALocatedFileStatus;
import org.apache.hadoop.fs.s3a.S3AUtils;
import org.apache.hadoop.fs.s3a.S3ListRequest;
import org.apache.hadoop.fs.s3a.S3ListResult;
import org.apache.hadoop.fs.s3a.Tristate;
import org.slf4j.Logger;

@InterfaceAudience.Private
public class Listing {
    private final S3AFileSystem owner;
    private static final Logger LOG = S3AFileSystem.LOG;
    static final FileStatusAcceptor ACCEPT_ALL_BUT_S3N = new AcceptAllButS3nDirs();

    public Listing(S3AFileSystem owner) {
        this.owner = owner;
    }

    ProvidedFileStatusIterator createProvidedFileStatusIterator(S3AFileStatus[] fileStatuses, PathFilter filter, FileStatusAcceptor acceptor) {
        return new ProvidedFileStatusIterator(fileStatuses, filter, acceptor);
    }

    public FileStatusListingIterator createFileStatusListingIterator(Path listPath, S3ListRequest request, PathFilter filter, FileStatusAcceptor acceptor) throws IOException {
        return this.createFileStatusListingIterator(listPath, request, filter, acceptor, null);
    }

    public FileStatusListingIterator createFileStatusListingIterator(Path listPath, S3ListRequest request, PathFilter filter, FileStatusAcceptor acceptor, RemoteIterator<S3AFileStatus> providedStatus) throws IOException {
        return new FileStatusListingIterator(new ObjectListingIterator(listPath, request), filter, acceptor, providedStatus);
    }

    @VisibleForTesting
    public LocatedFileStatusIterator createLocatedFileStatusIterator(RemoteIterator<S3AFileStatus> statusIterator) {
        return new LocatedFileStatusIterator(statusIterator);
    }

    @VisibleForTesting
    TombstoneReconcilingIterator createTombstoneReconcilingIterator(RemoteIterator<S3ALocatedFileStatus> iterator, Set<Path> tombstones) {
        return new TombstoneReconcilingIterator(iterator, tombstones);
    }

    public static class AcceptAllButSelfAndS3nDirs
    implements FileStatusAcceptor {
        private final Path qualifiedPath;

        public AcceptAllButSelfAndS3nDirs(Path qualifiedPath) {
            this.qualifiedPath = qualifiedPath;
        }

        @Override
        public boolean accept(Path keyPath, S3ObjectSummary summary) {
            return !keyPath.equals((Object)this.qualifiedPath) && !summary.getKey().endsWith("_$folder$");
        }

        @Override
        public boolean accept(Path keyPath, String prefix) {
            return !keyPath.equals((Object)this.qualifiedPath);
        }

        @Override
        public boolean accept(FileStatus status) {
            return status != null && !status.getPath().equals((Object)this.qualifiedPath);
        }
    }

    static class AcceptAllButS3nDirs
    implements FileStatusAcceptor {
        AcceptAllButS3nDirs() {
        }

        @Override
        public boolean accept(Path keyPath, S3ObjectSummary summary) {
            return !summary.getKey().endsWith("_$folder$");
        }

        @Override
        public boolean accept(Path keyPath, String prefix) {
            return !keyPath.toString().endsWith("_$folder$");
        }

        @Override
        public boolean accept(FileStatus status) {
            return !status.getPath().toString().endsWith("_$folder$");
        }
    }

    static class TombstoneReconcilingIterator
    implements RemoteIterator<S3ALocatedFileStatus> {
        private S3ALocatedFileStatus next = null;
        private final RemoteIterator<S3ALocatedFileStatus> iterator;
        private final Set<Path> tombstones;

        TombstoneReconcilingIterator(RemoteIterator<S3ALocatedFileStatus> iterator, Set<Path> tombstones) {
            this.iterator = iterator;
            this.tombstones = tombstones != null ? tombstones : Collections.emptySet();
        }

        private boolean fetch() throws IOException {
            while (this.next == null && this.iterator.hasNext()) {
                S3ALocatedFileStatus candidate = (S3ALocatedFileStatus)((Object)this.iterator.next());
                if (this.tombstones.contains(candidate.getPath())) continue;
                this.next = candidate;
                return true;
            }
            return false;
        }

        public boolean hasNext() throws IOException {
            if (this.next != null) {
                return true;
            }
            return this.fetch();
        }

        public S3ALocatedFileStatus next() throws IOException {
            if (this.hasNext()) {
                S3ALocatedFileStatus result = this.next;
                this.next = null;
                this.fetch();
                return result;
            }
            throw new NoSuchElementException();
        }
    }

    class LocatedFileStatusIterator
    implements RemoteIterator<S3ALocatedFileStatus> {
        private final RemoteIterator<S3AFileStatus> statusIterator;

        LocatedFileStatusIterator(RemoteIterator<S3AFileStatus> statusIterator) {
            this.statusIterator = statusIterator;
        }

        public boolean hasNext() throws IOException {
            return this.statusIterator.hasNext();
        }

        public S3ALocatedFileStatus next() throws IOException {
            return Listing.this.owner.toLocatedFileStatus((S3AFileStatus)((Object)this.statusIterator.next()));
        }
    }

    static class AcceptFilesOnly
    implements FileStatusAcceptor {
        private final Path qualifiedPath;

        public AcceptFilesOnly(Path qualifiedPath) {
            this.qualifiedPath = qualifiedPath;
        }

        @Override
        public boolean accept(Path keyPath, S3ObjectSummary summary) {
            return !keyPath.equals((Object)this.qualifiedPath) && !summary.getKey().endsWith("_$folder$") && !S3AUtils.objectRepresentsDirectory(summary.getKey(), summary.getSize());
        }

        @Override
        public boolean accept(Path keyPath, String prefix) {
            return false;
        }

        @Override
        public boolean accept(FileStatus status) {
            return status != null && status.isFile();
        }
    }

    class ObjectListingIterator
    implements RemoteIterator<S3ListResult> {
        private final Path listPath;
        private S3ListResult objects;
        private S3ListRequest request;
        private boolean firstListing = true;
        private int listingCount = 1;
        private int maxKeys;

        ObjectListingIterator(Path listPath, S3ListRequest request) throws IOException {
            this.listPath = listPath;
            this.maxKeys = Listing.this.owner.getMaxKeys();
            this.objects = Listing.this.owner.listObjects(request);
            this.request = request;
        }

        public boolean hasNext() throws IOException {
            return this.firstListing || this.objects.isTruncated();
        }

        public S3ListResult next() throws IOException {
            if (this.firstListing) {
                this.firstListing = false;
            } else {
                try {
                    if (!this.objects.isTruncated()) {
                        throw new NoSuchElementException("No more results in listing of " + this.listPath);
                    }
                    LOG.debug("[{}], Requesting next {} objects under {}", new Object[]{this.listingCount, this.maxKeys, this.listPath});
                    this.objects = Listing.this.owner.continueListObjects(this.request, this.objects);
                    ++this.listingCount;
                    LOG.debug("New listing status: {}", (Object)this);
                }
                catch (AmazonClientException e) {
                    throw S3AUtils.translateException("listObjects()", this.listPath, e);
                }
            }
            return this.objects;
        }

        public String toString() {
            return "Object listing iterator against " + this.listPath + "; listing count " + this.listingCount + "; isTruncated=" + this.objects.isTruncated();
        }

        public Path getListPath() {
            return this.listPath;
        }

        public int getListingCount() {
            return this.listingCount;
        }
    }

    class FileStatusListingIterator
    implements RemoteIterator<S3AFileStatus> {
        private final ObjectListingIterator source;
        private final PathFilter filter;
        private final FileStatusAcceptor acceptor;
        private int batchSize;
        private ListIterator<S3AFileStatus> statusBatchIterator;
        private final Map<Path, S3AFileStatus> providedStatus;
        private Iterator<S3AFileStatus> providedStatusIterator;

        FileStatusListingIterator(ObjectListingIterator source, PathFilter filter, @Nullable FileStatusAcceptor acceptor, RemoteIterator<S3AFileStatus> providedStatus) throws IOException {
            this.source = source;
            this.filter = filter;
            this.acceptor = acceptor;
            this.providedStatus = new HashMap<Path, S3AFileStatus>();
            while (providedStatus != null && providedStatus.hasNext()) {
                S3AFileStatus status = (S3AFileStatus)((Object)providedStatus.next());
                Path path = status.getPath();
                if (!filter.accept(path) || !acceptor.accept(status)) continue;
                this.providedStatus.put(path, status);
            }
            this.requestNextBatch();
        }

        public boolean hasNext() throws IOException {
            return this.sourceHasNext() || this.providedStatusIterator.hasNext();
        }

        private boolean sourceHasNext() throws IOException {
            if (this.statusBatchIterator.hasNext() || this.requestNextBatch()) {
                return true;
            }
            if (this.providedStatusIterator == null) {
                LOG.debug("Start iterating the provided status.");
                this.providedStatusIterator = this.providedStatus.values().iterator();
            }
            return false;
        }

        public S3AFileStatus next() throws IOException {
            S3AFileStatus status;
            if (this.sourceHasNext()) {
                status = this.statusBatchIterator.next();
                S3AFileStatus provided = this.providedStatus.remove(status.getPath());
                if (provided != null) {
                    LOG.debug("Removed and returned the status from provided file status {}", (Object)status);
                    return provided;
                }
            } else if (this.providedStatusIterator.hasNext()) {
                status = this.providedStatusIterator.next();
                LOG.debug("Returning provided file status {}", (Object)status);
            } else {
                throw new NoSuchElementException();
            }
            return status;
        }

        private boolean requestNextBatch() throws IOException {
            while (this.source.hasNext()) {
                if (this.buildNextStatusBatch(this.source.next())) {
                    return true;
                }
                LOG.debug("All entries in batch were filtered...continuing");
            }
            return false;
        }

        private boolean buildNextStatusBatch(S3ListResult objects) {
            int added = 0;
            int ignored = 0;
            ArrayList<S3AFileStatus> stats = new ArrayList<S3AFileStatus>(objects.getObjectSummaries().size() + objects.getCommonPrefixes().size());
            for (S3ObjectSummary summary : objects.getObjectSummaries()) {
                String key = summary.getKey();
                Path keyPath = Listing.this.owner.keyToQualifiedPath(key);
                if (LOG.isDebugEnabled()) {
                    LOG.debug("{}: {}", (Object)keyPath, (Object)S3AUtils.stringify(summary));
                }
                if (this.acceptor.accept(keyPath, summary) && this.filter.accept(keyPath)) {
                    S3AFileStatus status = S3AUtils.createFileStatus(keyPath, summary, Listing.this.owner.getDefaultBlockSize(keyPath), Listing.this.owner.getUsername(), summary.getETag(), null);
                    LOG.debug("Adding: {}", (Object)status);
                    stats.add(status);
                    ++added;
                    continue;
                }
                LOG.debug("Ignoring: {}", (Object)keyPath);
                ++ignored;
            }
            for (String prefix : objects.getCommonPrefixes()) {
                Path keyPath = Listing.this.owner.keyToQualifiedPath(prefix);
                if (this.acceptor.accept(keyPath, prefix) && this.filter.accept(keyPath)) {
                    S3AFileStatus status = new S3AFileStatus(Tristate.FALSE, keyPath, Listing.this.owner.getUsername());
                    LOG.debug("Adding directory: {}", (Object)status);
                    ++added;
                    stats.add(status);
                    continue;
                }
                LOG.debug("Ignoring directory: {}", (Object)keyPath);
                ++ignored;
            }
            this.batchSize = stats.size();
            this.statusBatchIterator = stats.listIterator();
            boolean hasNext = this.statusBatchIterator.hasNext();
            LOG.debug("Added {} entries; ignored {}; hasNext={}; hasMoreObjects={}", new Object[]{added, ignored, hasNext, objects.isTruncated()});
            return hasNext;
        }

        public int getBatchSize() {
            return this.batchSize;
        }
    }

    static class ProvidedFileStatusIterator
    implements RemoteIterator<S3AFileStatus> {
        private final ArrayList<S3AFileStatus> filteredStatusList;
        private int index = 0;

        ProvidedFileStatusIterator(S3AFileStatus[] fileStatuses, PathFilter filter, FileStatusAcceptor acceptor) {
            Preconditions.checkArgument((fileStatuses != null ? 1 : 0) != 0, (Object)"Null status list!");
            this.filteredStatusList = new ArrayList(fileStatuses.length);
            for (S3AFileStatus status : fileStatuses) {
                if (!filter.accept(status.getPath()) || !acceptor.accept(status)) continue;
                this.filteredStatusList.add(status);
            }
            this.filteredStatusList.trimToSize();
        }

        public boolean hasNext() throws IOException {
            return this.index < this.filteredStatusList.size();
        }

        public S3AFileStatus next() throws IOException {
            if (!this.hasNext()) {
                throw new NoSuchElementException();
            }
            return this.filteredStatusList.get(this.index++);
        }
    }

    static final class SingleStatusRemoteIterator
    implements RemoteIterator<S3ALocatedFileStatus> {
        private S3ALocatedFileStatus status;

        SingleStatusRemoteIterator(S3ALocatedFileStatus status) {
            this.status = status;
        }

        public boolean hasNext() throws IOException {
            return this.status != null;
        }

        public S3ALocatedFileStatus next() throws IOException {
            if (this.hasNext()) {
                S3ALocatedFileStatus s = this.status;
                this.status = null;
                return s;
            }
            throw new NoSuchElementException();
        }
    }

    static interface FileStatusAcceptor {
        public boolean accept(Path var1, S3ObjectSummary var2);

        public boolean accept(Path var1, String var2);

        public boolean accept(FileStatus var1);
    }
}

