/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.s3a.s3guard;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.Closeable;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.URI;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;
import javax.annotation.Nullable;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.RemoteIterator;
import org.apache.hadoop.fs.s3a.Constants;
import org.apache.hadoop.fs.s3a.S3AFileStatus;
import org.apache.hadoop.fs.s3a.S3AFileSystem;
import org.apache.hadoop.fs.s3a.S3AUtils;
import org.apache.hadoop.fs.s3a.s3guard.BulkOperationState;
import org.apache.hadoop.fs.s3a.s3guard.DirListingMetadata;
import org.apache.hadoop.fs.s3a.s3guard.DynamoDBClientFactory;
import org.apache.hadoop.fs.s3a.s3guard.ITtlTimeProvider;
import org.apache.hadoop.fs.s3a.s3guard.MetadataStore;
import org.apache.hadoop.fs.s3a.s3guard.NullMetadataStore;
import org.apache.hadoop.fs.s3a.s3guard.PathMetadata;
import org.apache.hadoop.fs.s3a.s3guard.PathMetadataDynamoDBTranslation;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.util.ExitUtil;
import org.apache.hadoop.util.ReflectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
@InterfaceStability.Unstable
public final class S3Guard {
    private static final Logger LOG = LoggerFactory.getLogger(S3Guard.class);
    @InterfaceAudience.Private
    @InterfaceStability.Unstable
    @VisibleForTesting
    public static final String S3GUARD_DDB_CLIENT_FACTORY_IMPL = "fs.s3a.s3guard.ddb.client.factory.impl";
    static final Class<? extends DynamoDBClientFactory> S3GUARD_DDB_CLIENT_FACTORY_IMPL_DEFAULT = DynamoDBClientFactory.DefaultDynamoDBClientFactory.class;
    private static final S3AFileStatus[] EMPTY_LISTING = new S3AFileStatus[0];
    @VisibleForTesting
    public static final boolean DIR_MERGE_UPDATES_ALL_RECORDS_NONAUTH = false;
    @VisibleForTesting
    public static final String DISABLED_LOG_MSG = "S3Guard is disabled on this bucket: %s";
    public static final String UNKNOWN_WARN_LEVEL = "Unknown fs.s3a.s3guard.disabled.warn.level value: ";

    private S3Guard() {
    }

    public static MetadataStore getMetadataStore(FileSystem fs, ITtlTimeProvider ttlTimeProvider) throws IOException {
        Preconditions.checkNotNull((Object)fs);
        Configuration conf = fs.getConf();
        Preconditions.checkNotNull((Object)conf);
        try {
            Class<? extends MetadataStore> msClass = S3Guard.getMetadataStoreClass(conf);
            MetadataStore msInstance = (MetadataStore)ReflectionUtils.newInstance(msClass, (Configuration)conf);
            LOG.debug("Using {} metadata store for {} filesystem", (Object)msClass.getSimpleName(), (Object)fs.getScheme());
            msInstance.initialize(fs, ttlTimeProvider);
            return msInstance;
        }
        catch (FileNotFoundException e) {
            throw e;
        }
        catch (IOException | RuntimeException e) {
            String message = "Failed to instantiate metadata store " + conf.get("fs.s3a.metadatastore.impl") + " defined in " + "fs.s3a.metadatastore.impl" + ": " + e;
            LOG.error(message, (Throwable)e);
            if (e instanceof IOException) {
                throw e;
            }
            throw new IOException(message, e);
        }
    }

    static Class<? extends MetadataStore> getMetadataStoreClass(Configuration conf) {
        if (conf == null) {
            return NullMetadataStore.class;
        }
        if (conf.get("fs.s3a.metadatastore.impl") != null && LOG.isDebugEnabled()) {
            LOG.debug("Metastore option source {}", (Object)conf.getPropertySources("fs.s3a.metadatastore.impl"));
        }
        Class aClass = conf.getClass("fs.s3a.metadatastore.impl", NullMetadataStore.class, MetadataStore.class);
        return aClass;
    }

    public static S3AFileStatus putAndReturn(MetadataStore ms, S3AFileStatus status, ITtlTimeProvider timeProvider) throws IOException {
        return S3Guard.putAndReturn(ms, status, timeProvider, null);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static S3AFileStatus putAndReturn(MetadataStore ms, S3AFileStatus status, ITtlTimeProvider timeProvider, @Nullable BulkOperationState operationState) throws IOException {
        long startTimeNano = System.nanoTime();
        try {
            S3Guard.putWithTtl(ms, new PathMetadata(status), timeProvider, operationState);
        }
        finally {
            ms.getInstrumentation().entryAdded(System.nanoTime() - startTimeNano);
        }
        return status;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void putAuthDirectoryMarker(MetadataStore ms, S3AFileStatus status, ITtlTimeProvider timeProvider, @Nullable BulkOperationState operationState) throws IOException {
        long startTimeNano = System.nanoTime();
        try {
            PathMetadata fileMeta = PathMetadataDynamoDBTranslation.authoritativeEmptyDirectoryMarker(status);
            S3Guard.putWithTtl(ms, fileMeta, timeProvider, operationState);
        }
        finally {
            ms.getInstrumentation().directoryMarkedAuthoritative();
            ms.getInstrumentation().entryAdded(System.nanoTime() - startTimeNano);
        }
    }

    public static BulkOperationState initiateBulkWrite(@Nullable MetadataStore metastore, BulkOperationState.OperationType operation, Path path) throws IOException {
        Preconditions.checkArgument((operation != BulkOperationState.OperationType.Rename ? 1 : 0) != 0, (Object)"Rename operations cannot be started through initiateBulkWrite");
        if (metastore == null || S3Guard.isNullMetadataStore(metastore)) {
            return null;
        }
        return metastore.initiateBulkWrite(operation, path);
    }

    public static S3AFileStatus[] iteratorToStatuses(RemoteIterator<S3AFileStatus> iterator, Set<Path> tombstones) throws IOException {
        ArrayList<S3AFileStatus> statuses = new ArrayList<S3AFileStatus>();
        while (iterator.hasNext()) {
            S3AFileStatus status = (S3AFileStatus)((Object)iterator.next());
            if (tombstones.contains(status.getPath())) continue;
            statuses.add(status);
        }
        return statuses.toArray(new S3AFileStatus[0]);
    }

    public static S3AFileStatus[] dirMetaToStatuses(DirListingMetadata dirMeta) {
        if (dirMeta == null) {
            return EMPTY_LISTING;
        }
        Collection<PathMetadata> listing = dirMeta.getListing();
        ArrayList<S3AFileStatus> statuses = new ArrayList<S3AFileStatus>();
        for (PathMetadata pm : listing) {
            if (pm.isDeleted()) continue;
            statuses.add(pm.getFileStatus());
        }
        return statuses.toArray(new S3AFileStatus[0]);
    }

    public static FileStatus[] dirListingUnion(MetadataStore ms, Path path, List<S3AFileStatus> backingStatuses, DirListingMetadata dirMeta, boolean isAuthoritative, ITtlTimeProvider timeProvider) throws IOException {
        if (S3Guard.isNullMetadataStore(ms)) {
            return backingStatuses.toArray(new FileStatus[backingStatuses.size()]);
        }
        S3Guard.assertQualified(path);
        if (dirMeta == null) {
            dirMeta = new DirListingMetadata(path, DirListingMetadata.EMPTY_DIR, false);
        }
        BulkOperationState operationState = ms.initiateBulkWrite(BulkOperationState.OperationType.Listing, path);
        if (isAuthoritative) {
            S3Guard.authoritativeUnion(ms, path, backingStatuses, dirMeta, timeProvider, operationState);
        } else {
            S3Guard.nonAuthoritativeUnion(ms, path, backingStatuses, dirMeta, timeProvider, operationState);
        }
        IOUtils.cleanupWithLogger((Logger)LOG, (Closeable[])new Closeable[]{operationState});
        return S3Guard.dirMetaToStatuses(dirMeta);
    }

    private static void authoritativeUnion(MetadataStore ms, Path path, List<S3AFileStatus> backingStatuses, DirListingMetadata dirMeta, ITtlTimeProvider timeProvider, BulkOperationState operationState) throws IOException {
        ArrayList<Path> unchangedEntries = new ArrayList<Path>(dirMeta.getListing().size());
        boolean changed = !dirMeta.isAuthoritative();
        Set<Path> deleted = dirMeta.listTombstones();
        Map<Path, PathMetadata> dirMetaMap = dirMeta.getListing().stream().collect(Collectors.toMap(pm -> pm.getFileStatus().getPath(), pm -> pm));
        for (S3AFileStatus s : backingStatuses) {
            Path statusPath = s.getPath();
            if (deleted.contains(statusPath)) continue;
            PathMetadata pathMetadata = dirMetaMap.get(statusPath);
            if (pathMetadata == null) {
                pathMetadata = new PathMetadata(s);
            } else {
                unchangedEntries.add(statusPath);
            }
            changed |= dirMeta.put(pathMetadata);
        }
        if (changed) {
            LOG.debug("Marking the directory {} as authoritative", (Object)path);
            ms.getInstrumentation().directoryMarkedAuthoritative();
            dirMeta.setAuthoritative(true);
            S3Guard.putWithTtl(ms, dirMeta, unchangedEntries, timeProvider, operationState);
        }
    }

    private static void nonAuthoritativeUnion(MetadataStore ms, Path path, List<S3AFileStatus> backingStatuses, DirListingMetadata dirMeta, ITtlTimeProvider timeProvider, BulkOperationState operationState) throws IOException {
        ArrayList<PathMetadata> entriesToAdd = new ArrayList<PathMetadata>(backingStatuses.size());
        Set<Path> deleted = dirMeta.listTombstones();
        Map<Path, PathMetadata> dirMetaMap = dirMeta.getListing().stream().collect(Collectors.toMap(pm -> pm.getFileStatus().getPath(), pm -> pm));
        for (S3AFileStatus s : backingStatuses) {
            boolean shouldUpdate;
            Path statusPath = s.getPath();
            if (deleted.contains(statusPath)) continue;
            PathMetadata pathMetadata = dirMetaMap.get(statusPath);
            if (pathMetadata != null) {
                shouldUpdate = s.getModificationTime() > pathMetadata.getFileStatus().getModificationTime();
                pathMetadata = new PathMetadata(s);
            } else {
                pathMetadata = new PathMetadata(s);
                shouldUpdate = false;
            }
            if (shouldUpdate) {
                LOG.debug("Update ms with newer metadata of: {}", (Object)s);
                entriesToAdd.add(pathMetadata);
            }
            dirMeta.put(pathMetadata);
        }
        if (!entriesToAdd.isEmpty()) {
            LOG.debug("Adding {} entries under directory {}", (Object)entriesToAdd.size(), (Object)path);
            S3Guard.putWithTtl(ms, entriesToAdd, timeProvider, operationState);
        }
    }

    public static boolean isNullMetadataStore(MetadataStore ms) {
        return ms instanceof NullMetadataStore;
    }

    @Deprecated
    public static void makeDirsOrdered(MetadataStore ms, List<Path> dirs, String owner, boolean authoritative, ITtlTimeProvider timeProvider) {
        if (dirs == null) {
            return;
        }
        S3AFileStatus prevStatus = null;
        ArrayList<PathMetadata> pathMetas = new ArrayList<PathMetadata>(dirs.size());
        try {
            for (int i = 0; i < dirs.size(); ++i) {
                boolean isLeaf = prevStatus == null;
                Path f = dirs.get(i);
                S3Guard.assertQualified(f);
                S3AFileStatus status = S3AUtils.createUploadFileStatus(f, true, 0L, 0L, owner, null, null);
                DirListingMetadata dirMeta = null;
                if (authoritative) {
                    Collection<PathMetadata> children;
                    if (isLeaf) {
                        children = DirListingMetadata.EMPTY_DIR;
                    } else {
                        children = new ArrayList<PathMetadata>(1);
                        children.add(new PathMetadata(prevStatus));
                    }
                    dirMeta = new DirListingMetadata(f, children, authoritative);
                    S3Guard.putWithTtl(ms, dirMeta, Collections.emptyList(), timeProvider, null);
                }
                pathMetas.add(new PathMetadata(status));
                prevStatus = status;
            }
            S3Guard.putWithTtl(ms, pathMetas, timeProvider, null);
        }
        catch (IOException ioe) {
            LOG.error("MetadataStore#put() failure:", (Throwable)ioe);
        }
    }

    public static void addMoveDir(MetadataStore ms, Collection<Path> srcPaths, Collection<PathMetadata> dstMetas, Path srcPath, Path dstPath, String owner) {
        if (S3Guard.isNullMetadataStore(ms)) {
            return;
        }
        S3Guard.assertQualified(srcPath, dstPath);
        S3AFileStatus dstStatus = S3AUtils.createUploadFileStatus(dstPath, true, 0L, 0L, owner, null, null);
        S3Guard.addMoveStatus(srcPaths, dstMetas, srcPath, dstStatus);
    }

    public static void addMoveFile(MetadataStore ms, Collection<Path> srcPaths, Collection<PathMetadata> dstMetas, Path srcPath, Path dstPath, long size, long blockSize, String owner, String eTag, String versionId) {
        if (S3Guard.isNullMetadataStore(ms)) {
            return;
        }
        S3Guard.assertQualified(srcPath, dstPath);
        S3AFileStatus dstStatus = S3AUtils.createUploadFileStatus(dstPath, false, size, blockSize, owner, eTag, versionId);
        S3Guard.addMoveStatus(srcPaths, dstMetas, srcPath, dstStatus);
    }

    public static void addMoveAncestors(MetadataStore ms, Collection<Path> srcPaths, Collection<PathMetadata> dstMetas, Path srcRoot, Path srcPath, Path dstPath, String owner) {
        if (S3Guard.isNullMetadataStore(ms)) {
            return;
        }
        S3Guard.assertQualified(srcRoot, srcPath, dstPath);
        if (srcPath.equals((Object)srcRoot)) {
            LOG.debug("Skip moving ancestors of source root directory {}", (Object)srcRoot);
            return;
        }
        Path parentSrc = srcPath.getParent();
        Path parentDst = dstPath.getParent();
        while (!(parentSrc == null || parentSrc.isRoot() || parentSrc.equals((Object)srcRoot) || srcPaths.contains(parentSrc))) {
            LOG.debug("Renaming non-listed parent {} to {}", (Object)parentSrc, (Object)parentDst);
            S3Guard.addMoveDir(ms, srcPaths, dstMetas, parentSrc, parentDst, owner);
            parentSrc = parentSrc.getParent();
            parentDst = parentDst.getParent();
        }
    }

    public static void addAncestors(MetadataStore metadataStore, Path qualifiedPath, ITtlTimeProvider timeProvider, @Nullable BulkOperationState operationState) throws IOException {
        metadataStore.addAncestors(qualifiedPath, operationState);
    }

    private static void addMoveStatus(Collection<Path> srcPaths, Collection<PathMetadata> dstMetas, Path srcPath, S3AFileStatus dstStatus) {
        srcPaths.add(srcPath);
        dstMetas.add(new PathMetadata(dstStatus));
    }

    public static void assertQualified(Path p) {
        URI uri = p.toUri();
        Preconditions.checkNotNull((Object)uri.getHost(), (Object)("Null host in " + uri));
        Preconditions.checkNotNull((Object)uri.getScheme(), (Object)("Null scheme in " + uri));
    }

    public static void assertQualified(Path ... paths) {
        for (Path path : paths) {
            S3Guard.assertQualified(path);
        }
    }

    public static void putWithTtl(MetadataStore ms, DirListingMetadata dirMeta, List<Path> unchangedEntries, ITtlTimeProvider timeProvider, @Nullable BulkOperationState operationState) throws IOException {
        long now = timeProvider.getNow();
        dirMeta.setLastUpdated(now);
        dirMeta.getListing().forEach(pm -> pm.setLastUpdated(now));
        ms.put(dirMeta, unchangedEntries, operationState);
    }

    public static void putWithTtl(MetadataStore ms, PathMetadata fileMeta, @Nullable ITtlTimeProvider timeProvider, @Nullable BulkOperationState operationState) throws IOException {
        if (timeProvider != null) {
            fileMeta.setLastUpdated(timeProvider.getNow());
        } else {
            LOG.debug("timeProvider is null, put {} without setting last_updated", (Object)fileMeta);
        }
        ms.put(fileMeta, operationState);
    }

    public static void putWithTtl(MetadataStore ms, Collection<? extends PathMetadata> fileMetas, @Nullable ITtlTimeProvider timeProvider, @Nullable BulkOperationState operationState) throws IOException {
        S3Guard.patchLastUpdated(fileMetas, timeProvider);
        ms.put(fileMetas, operationState);
    }

    static void patchLastUpdated(Collection<? extends PathMetadata> fileMetas, @Nullable ITtlTimeProvider timeProvider) {
        if (timeProvider != null) {
            long now = timeProvider.getNow();
            fileMetas.forEach(fileMeta -> fileMeta.setLastUpdated(now));
        } else {
            LOG.debug("timeProvider is null, put {} without setting last_updated", fileMetas);
        }
    }

    public static PathMetadata getWithTtl(MetadataStore ms, Path path, @Nullable ITtlTimeProvider timeProvider, boolean needEmptyDirectoryFlag, boolean allowAuthoritative) throws IOException {
        PathMetadata pathMetadata = ms.get(path, needEmptyDirectoryFlag);
        if (timeProvider == null) {
            LOG.debug("timeProvider is null, returning pathMetadata as is");
            return pathMetadata;
        }
        if (allowAuthoritative) {
            LOG.debug("allowAuthoritative is true, returning pathMetadata as is");
            return pathMetadata;
        }
        long ttl = timeProvider.getMetadataTtl();
        if (pathMetadata != null) {
            if (pathMetadata.getLastUpdated() == 0L) {
                LOG.debug("PathMetadata TTL for {} is 0, so it will be returned as not expired.", (Object)path);
                return pathMetadata;
            }
            if (!pathMetadata.isExpired(ttl, timeProvider.getNow())) {
                return pathMetadata;
            }
            LOG.debug("PathMetadata TTl for {} is expired in metadata store.", (Object)path);
            return null;
        }
        return null;
    }

    public static DirListingMetadata listChildrenWithTtl(MetadataStore ms, Path path, @Nullable ITtlTimeProvider timeProvider, boolean allowAuthoritative) throws IOException {
        DirListingMetadata dlm = ms.listChildren(path);
        if (timeProvider == null) {
            LOG.debug("timeProvider is null, returning DirListingMetadata as is");
            return dlm;
        }
        if (allowAuthoritative) {
            LOG.debug("allowAuthoritative is true, returning pathMetadata as is");
            return dlm;
        }
        if (dlm != null) {
            dlm.removeExpiredEntriesFromListing(timeProvider.getMetadataTtl(), timeProvider.getNow());
        }
        return dlm;
    }

    public static Collection<String> getAuthoritativePaths(S3AFileSystem fs) {
        return S3Guard.getAuthoritativePaths(fs.getUri(), fs.getConf(), p -> fs.maybeAddTrailingSlash(fs.qualify((Path)p).toString()));
    }

    @VisibleForTesting
    static Collection<String> getAuthoritativePaths(URI uri, Configuration conf, Function<Path, String> qualifyToDir) {
        String[] rawAuthoritativePaths = conf.getTrimmedStrings("fs.s3a.authoritative.path", Constants.DEFAULT_AUTHORITATIVE_PATH);
        ArrayList<String> authoritativePaths = new ArrayList<String>();
        if (rawAuthoritativePaths.length > 0) {
            for (int i = 0; i < rawAuthoritativePaths.length; ++i) {
                Path path = new Path(rawAuthoritativePaths[i]);
                URI pathURI = path.toUri();
                if (pathURI.getAuthority() != null && !pathURI.getAuthority().equals(uri.getAuthority()) || pathURI.getScheme() != null && !pathURI.getScheme().equals(uri.getScheme())) continue;
                authoritativePaths.add(qualifyToDir.apply(path));
            }
        }
        return authoritativePaths;
    }

    public static boolean allowAuthoritative(Path p, S3AFileSystem fs, boolean authMetadataStore, Collection<String> authPaths) {
        String haystack = fs.maybeAddTrailingSlash(fs.qualify(p).toString());
        if (authMetadataStore) {
            return true;
        }
        if (!authPaths.isEmpty()) {
            for (String needle : authPaths) {
                if (!haystack.startsWith(needle)) continue;
                return true;
            }
        }
        return false;
    }

    public static void logS3GuardDisabled(Logger logger, String warnLevelStr, String bucket) throws ExitUtil.ExitException, IllegalArgumentException {
        DisabledWarnLevel warnLevel;
        try {
            warnLevel = DisabledWarnLevel.valueOf(warnLevelStr.toUpperCase(Locale.US));
        }
        catch (IllegalArgumentException e) {
            throw new IllegalArgumentException(UNKNOWN_WARN_LEVEL + warnLevelStr, e);
        }
        String text = String.format(DISABLED_LOG_MSG, bucket);
        switch (warnLevel) {
            case SILENT: {
                logger.debug(text);
                break;
            }
            case INFORM: {
                logger.info(text);
                break;
            }
            case WARN: {
                logger.warn(text);
                break;
            }
            case FAIL: {
                logger.error(text);
                throw new ExitUtil.ExitException(49, text);
            }
            default: {
                throw new IllegalArgumentException(UNKNOWN_WARN_LEVEL + warnLevelStr);
            }
        }
    }

    public static enum DisabledWarnLevel {
        SILENT,
        INFORM,
        WARN,
        FAIL;

    }

    public static class TtlTimeProvider
    implements ITtlTimeProvider {
        private long authoritativeDirTtl;

        public TtlTimeProvider(long authoritativeDirTtl) {
            this.authoritativeDirTtl = authoritativeDirTtl;
        }

        public TtlTimeProvider(Configuration conf) {
            this.authoritativeDirTtl = conf.getTimeDuration("fs.s3a.metadatastore.metadata.ttl", Constants.DEFAULT_METADATASTORE_METADATA_TTL, TimeUnit.MILLISECONDS);
        }

        @Override
        public long getNow() {
            return System.currentTimeMillis();
        }

        @Override
        public long getMetadataTtl() {
            return this.authoritativeDirTtl;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            TtlTimeProvider that = (TtlTimeProvider)o;
            return this.authoritativeDirTtl == that.authoritativeDirTtl;
        }

        public int hashCode() {
            return Objects.hash(this.authoritativeDirTtl);
        }

        public String toString() {
            StringBuilder sb = new StringBuilder("TtlTimeProvider{");
            sb.append("authoritativeDirTtl=").append(this.authoritativeDirTtl);
            sb.append(" millis}");
            return sb.toString();
        }
    }
}

