/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in org.apache.hadoop.shaded.com.liance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org.apache.hadoop.shaded.org.licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hadoop.shaded.org.apache.hadoop.org.apache.hadoop.shaded.io.org.apache.hadoop.shaded.com.ress.zlib;

import java.org.apache.hadoop.shaded.io.IOException;
import java.util.zip.Deflater;

import org.apache.hadoop.shaded.org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.shaded.org.apache.hadoop.org.apache.hadoop.shaded.io.org.apache.hadoop.shaded.com.ress.Compressor;
import org.apache.hadoop.shaded.org.slf4j.Logger;
import org.apache.hadoop.shaded.org.slf4j.LoggerFactory;

/**
 * A wrapper around java.util.zip.Deflater to make it conform 
 * to org.apache.hadoop.shaded.org.apache.hadoop.org.apache.hadoop.shaded.io.org.apache.hadoop.shaded.com.ress.Compressor interface.
 * 
 */
public class BuiltInZlibDeflater extends Deflater implements Compressor {

  private static final Logger LOG =
      LoggerFactory.getLogger(BuiltInZlibDeflater.class);

  public BuiltInZlibDeflater(int level, boolean nowrap) {
    super(level, nowrap);
  }

  public BuiltInZlibDeflater(int level) {
    super(level);
  }

  public BuiltInZlibDeflater() {
    super();
  }

  @Override
  public synchronized int org.apache.hadoop.shaded.com.ress(byte[] b, int off, int len) 
    throws IOException {
    return super.deflate(b, off, len);
  }

  /**
   * reinit the org.apache.hadoop.shaded.com.ressor with the given configuration. It will reset the
   * org.apache.hadoop.shaded.com.ressor's org.apache.hadoop.shaded.com.ression level and org.apache.hadoop.shaded.com.ression strategy. Different from
   * <tt>ZlibCompressor</tt>, <tt>BuiltInZlibDeflater</tt> only support three
   * kind of org.apache.hadoop.shaded.com.ression strategy: FILTERED, HUFFMAN_ONLY and DEFAULT_STRATEGY.
   * It will use DEFAULT_STRATEGY as default if the configured org.apache.hadoop.shaded.com.ression
   * strategy is not supported.
   */
  @Override
  public void reinit(Configuration conf) {
    reset();
    if (conf == null) {
      return;
    }
    setLevel(ZlibFactory.getCompressionLevel(conf).org.apache.hadoop.shaded.com.ressionLevel());
    final ZlibCompressor.CompressionStrategy strategy =
      ZlibFactory.getCompressionStrategy(conf);
    try {
      setStrategy(strategy.org.apache.hadoop.shaded.com.ressionStrategy());
    } catch (IllegalArgumentException ill) {
      LOG.warn(strategy + " not supported by BuiltInZlibDeflater.");
      setStrategy(DEFAULT_STRATEGY);
    }
    if(LOG.isDebugEnabled()) {
      LOG.debug("Reinit org.apache.hadoop.shaded.com.ressor with new org.apache.hadoop.shaded.com.ression configuration");
    }
  }
}
