/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.namenode.snapshot;

import java.io.ByteArrayOutputStream;
import java.io.Closeable;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.PrintStream;
import java.security.PrivilegedAction;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.FsShell;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hdfs.DFSTestUtil;
import org.apache.hadoop.hdfs.DFSUtil;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.MiniDFSNNTopology;
import org.apache.hadoop.hdfs.protocol.HdfsConstants;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockInfo;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockManager;
import org.apache.hadoop.hdfs.server.namenode.FSDirectory;
import org.apache.hadoop.hdfs.server.namenode.FSNamesystem;
import org.apache.hadoop.hdfs.server.namenode.INode;
import org.apache.hadoop.hdfs.server.namenode.INodeDirectory;
import org.apache.hadoop.hdfs.server.namenode.INodeFile;
import org.apache.hadoop.hdfs.server.namenode.NameNode;
import org.apache.hadoop.hdfs.server.namenode.NameNodeAdapter;
import org.apache.hadoop.hdfs.server.namenode.QuotaCounts;
import org.apache.hadoop.hdfs.server.namenode.ha.HATestUtil;
import org.apache.hadoop.hdfs.server.namenode.snapshot.DirectoryWithSnapshotFeature;
import org.apache.hadoop.hdfs.server.namenode.snapshot.Snapshot;
import org.apache.hadoop.hdfs.server.namenode.snapshot.SnapshotTestHelper;
import org.apache.hadoop.hdfs.server.namenode.snapshot.TestSnapshotBlocksMap;
import org.apache.hadoop.hdfs.util.ReadOnlyList;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.ipc.RemoteException;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.test.GenericTestUtils;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;

public class TestSnapshotDeletion {
    protected static final long seed = 0L;
    protected static final short REPLICATION = 3;
    protected static final short REPLICATION_1 = 2;
    protected static final long BLOCKSIZE = 1024L;
    private final Path dir = new Path("/TestSnapshot");
    private final Path sub = new Path(this.dir, "sub1");
    private final Path subsub = new Path(this.sub, "subsub1");
    protected Configuration conf;
    protected MiniDFSCluster cluster;
    protected FSNamesystem fsn;
    protected FSDirectory fsdir;
    protected BlockManager blockmanager;
    protected DistributedFileSystem hdfs;
    @Rule
    public ExpectedException exception = ExpectedException.none();

    @Before
    public void setUp() throws Exception {
        this.conf = new Configuration();
        this.cluster = new MiniDFSCluster.Builder(this.conf).numDataNodes(3).format(true).build();
        this.cluster.waitActive();
        this.fsn = this.cluster.getNamesystem();
        this.fsdir = this.fsn.getFSDirectory();
        this.blockmanager = this.fsn.getBlockManager();
        this.hdfs = this.cluster.getFileSystem();
    }

    @After
    public void tearDown() throws Exception {
        if (this.cluster != null) {
            this.cluster.shutdown();
            this.cluster = null;
        }
    }

    @Test(timeout=300000L)
    public void testDeleteDirectoryWithSnapshot() throws Exception {
        Path file0 = new Path(this.sub, "file0");
        Path file1 = new Path(this.sub, "file1");
        DFSTestUtil.createFile((FileSystem)this.hdfs, file0, 1024L, (short)3, 0L);
        DFSTestUtil.createFile((FileSystem)this.hdfs, file1, 1024L, (short)3, 0L);
        this.hdfs.allowSnapshot(this.sub);
        this.hdfs.createSnapshot(this.sub, "s1");
        this.exception.expect(RemoteException.class);
        String error = "The directory " + this.sub.toString() + " cannot be deleted since " + this.sub.toString() + " is snapshottable and already has snapshots";
        this.exception.expectMessage(error);
        this.hdfs.delete(this.sub, true);
    }

    @Test(timeout=300000L)
    public void testApplyEditLogForDeletion() throws Exception {
        Path foo = new Path("/foo");
        Path bar1 = new Path(foo, "bar1");
        Path bar2 = new Path(foo, "bar2");
        this.hdfs.mkdirs(bar1);
        this.hdfs.mkdirs(bar2);
        this.hdfs.allowSnapshot(bar1);
        this.hdfs.allowSnapshot(bar2);
        Assert.assertEquals((long)2L, (long)this.cluster.getNamesystem().getSnapshotManager().getNumSnapshottableDirs());
        Assert.assertEquals((long)2L, (long)this.cluster.getNamesystem().getSnapshotManager().getSnapshottableDirs().length);
        this.hdfs.delete(foo, true);
        this.cluster.restartNameNode(0);
        Assert.assertEquals((long)0L, (long)this.cluster.getNamesystem().getSnapshotManager().getNumSnapshottableDirs());
        Assert.assertEquals((long)0L, (long)this.cluster.getNamesystem().getSnapshotManager().getSnapshottableDirs().length);
        this.hdfs.setSafeMode(HdfsConstants.SafeModeAction.SAFEMODE_ENTER);
        this.hdfs.saveNamespace();
        this.hdfs.setSafeMode(HdfsConstants.SafeModeAction.SAFEMODE_LEAVE);
        this.cluster.restartNameNode(0);
    }

    @Test(timeout=300000L)
    public void testDeleteDirectoryWithSnapshot2() throws Exception {
        Path file0 = new Path(this.sub, "file0");
        Path file1 = new Path(this.sub, "file1");
        DFSTestUtil.createFile((FileSystem)this.hdfs, file0, 1024L, (short)3, 0L);
        DFSTestUtil.createFile((FileSystem)this.hdfs, file1, 1024L, (short)3, 0L);
        Path subfile1 = new Path(this.subsub, "file0");
        Path subfile2 = new Path(this.subsub, "file1");
        DFSTestUtil.createFile((FileSystem)this.hdfs, subfile1, 1024L, (short)3, 0L);
        DFSTestUtil.createFile((FileSystem)this.hdfs, subfile2, 1024L, (short)3, 0L);
        this.hdfs.allowSnapshot(this.subsub);
        this.hdfs.createSnapshot(this.subsub, "s1");
        this.exception.expect(RemoteException.class);
        String error = this.subsub.toString() + " is snapshottable and already has snapshots";
        this.exception.expectMessage(error);
        this.hdfs.delete(this.dir, true);
    }

    private static INodeDirectory getDir(FSDirectory fsdir, Path dir) throws IOException {
        String dirStr = dir.toString();
        return INodeDirectory.valueOf((INode)fsdir.getINode(dirStr), (Object)dirStr);
    }

    private void checkQuotaUsageComputation(Path dirPath, long expectedNs, long expectedDs) throws IOException {
        INodeDirectory dirNode = TestSnapshotDeletion.getDir(this.fsdir, dirPath);
        Assert.assertTrue((boolean)dirNode.isQuotaSet());
        QuotaCounts q = dirNode.getDirectoryWithQuotaFeature().getSpaceConsumed();
        Assert.assertEquals((String)dirNode.dumpTreeRecursively().toString(), (long)expectedNs, (long)q.getNameSpace());
        Assert.assertEquals((String)dirNode.dumpTreeRecursively().toString(), (long)expectedDs, (long)q.getStorageSpace());
        QuotaCounts counts = dirNode.computeQuotaUsage(this.fsdir.getBlockStoragePolicySuite(), false);
        Assert.assertEquals((String)dirNode.dumpTreeRecursively().toString(), (long)expectedNs, (long)counts.getNameSpace());
        Assert.assertEquals((String)dirNode.dumpTreeRecursively().toString(), (long)expectedDs, (long)counts.getStorageSpace());
    }

    @Test(timeout=300000L)
    public void testDeleteCurrentFileDirectory() throws Exception {
        Path deleteDir = new Path(this.subsub, "deleteDir");
        Path deleteFile = new Path(deleteDir, "deleteFile");
        Path noChangeDirParent = new Path(this.sub, "noChangeDirParent");
        Path noChangeDir = new Path(noChangeDirParent, "noChangeDir");
        Path noChangeFile = new Path(noChangeDir, "noChangeFile");
        DFSTestUtil.createFile((FileSystem)this.hdfs, deleteFile, 1024L, (short)3, 0L);
        DFSTestUtil.createFile((FileSystem)this.hdfs, noChangeFile, 1024L, (short)3, 0L);
        Path metaChangeFile1 = new Path(this.subsub, "metaChangeFile1");
        DFSTestUtil.createFile((FileSystem)this.hdfs, metaChangeFile1, 1024L, (short)3, 0L);
        Path metaChangeFile2 = new Path(noChangeDir, "metaChangeFile2");
        DFSTestUtil.createFile((FileSystem)this.hdfs, metaChangeFile2, 1024L, (short)3, 0L);
        this.hdfs.setQuota(this.dir, 0x7FFFFFFFFFFFFFFEL, 0x7FFFFFFFFFFFFFFEL);
        this.checkQuotaUsageComputation(this.dir, 10L, 12288L);
        this.hdfs.delete(deleteDir, true);
        this.checkQuotaUsageComputation(this.dir, 8L, 9216L);
        SnapshotTestHelper.createSnapshot(this.hdfs, this.dir, "s0");
        Path tempDir = new Path(this.dir, "tempdir");
        Path tempFile = new Path(tempDir, "tempfile");
        DFSTestUtil.createFile((FileSystem)this.hdfs, tempFile, 1024L, (short)3, 0L);
        INodeFile temp = TestSnapshotBlocksMap.assertBlockCollection(tempFile.toString(), 1, this.fsdir, this.blockmanager);
        BlockInfo[] blocks = temp.getBlocks();
        this.hdfs.delete(tempDir, true);
        this.checkQuotaUsageComputation(this.dir, 8L, 9216L);
        for (BlockInfo b : blocks) {
            Assert.assertEquals((long)-1L, (long)b.getBlockCollectionId());
        }
        Path newFileAfterS0 = new Path(this.subsub, "newFile");
        DFSTestUtil.createFile((FileSystem)this.hdfs, newFileAfterS0, 1024L, (short)3, 0L);
        this.hdfs.setReplication(metaChangeFile1, (short)2);
        this.hdfs.setReplication(metaChangeFile2, (short)2);
        SnapshotTestHelper.createSnapshot(this.hdfs, this.dir, "s1");
        this.checkQuotaUsageComputation(this.dir, 9L, 12288L);
        Snapshot snapshot0 = this.fsdir.getINode(this.dir.toString()).asDirectory().getSnapshot(DFSUtil.string2Bytes((String)"s0"));
        Snapshot snapshot1 = this.fsdir.getINode(this.dir.toString()).asDirectory().getSnapshot(DFSUtil.string2Bytes((String)"s1"));
        this.hdfs.delete(noChangeDirParent, true);
        this.checkQuotaUsageComputation(this.dir, 9L, 12288L);
        Path snapshotNoChangeDir = SnapshotTestHelper.getSnapshotPath(this.dir, "s1", this.sub.getName() + "/" + noChangeDirParent.getName() + "/" + noChangeDir.getName());
        INodeDirectory snapshotNode = (INodeDirectory)this.fsdir.getINode(snapshotNoChangeDir.toString());
        Assert.assertEquals(INodeDirectory.class, snapshotNode.getClass());
        ReadOnlyList children = snapshotNode.getChildrenList(0x7FFFFFFE);
        Assert.assertEquals((long)2L, (long)children.size());
        INode noChangeFileSCopy = (INode)children.get(1);
        Assert.assertEquals((Object)noChangeFile.getName(), (Object)noChangeFileSCopy.getLocalName());
        Assert.assertEquals(INodeFile.class, noChangeFileSCopy.getClass());
        TestSnapshotBlocksMap.assertBlockCollection(new Path(snapshotNoChangeDir, noChangeFileSCopy.getLocalName()).toString(), 1, this.fsdir, this.blockmanager);
        INodeFile metaChangeFile2SCopy = ((INode)children.get(0)).asFile();
        Assert.assertEquals((Object)metaChangeFile2.getName(), (Object)metaChangeFile2SCopy.getLocalName());
        Assert.assertTrue((boolean)metaChangeFile2SCopy.isWithSnapshot());
        Assert.assertFalse((boolean)metaChangeFile2SCopy.isUnderConstruction());
        TestSnapshotBlocksMap.assertBlockCollection(new Path(snapshotNoChangeDir, metaChangeFile2SCopy.getLocalName()).toString(), 1, this.fsdir, this.blockmanager);
        Assert.assertEquals((long)2L, (long)metaChangeFile2SCopy.getFileReplication(0x7FFFFFFE));
        Assert.assertEquals((long)2L, (long)metaChangeFile2SCopy.getFileReplication(snapshot1.getId()));
        Assert.assertEquals((long)3L, (long)metaChangeFile2SCopy.getFileReplication(snapshot0.getId()));
        Path newFile = new Path(this.sub, "newFile");
        DFSTestUtil.createFile((FileSystem)this.hdfs, newFile, 1024L, (short)3, 0L);
        INodeFile newFileNode = TestSnapshotBlocksMap.assertBlockCollection(newFile.toString(), 1, this.fsdir, this.blockmanager);
        blocks = newFileNode.getBlocks();
        this.checkQuotaUsageComputation(this.dir, 10L, 15360L);
        this.hdfs.delete(this.sub, true);
        this.checkQuotaUsageComputation(this.dir, 9L, 12288L);
        for (BlockInfo b : blocks) {
            Assert.assertEquals((long)-1L, (long)b.getBlockCollectionId());
        }
        Path snapshotSub = SnapshotTestHelper.getSnapshotPath(this.dir, "s1", this.sub.getName());
        INodeDirectory snapshotNode4Sub = this.fsdir.getINode(snapshotSub.toString()).asDirectory();
        Assert.assertTrue((boolean)snapshotNode4Sub.isWithSnapshot());
        Assert.assertEquals((long)1L, (long)snapshotNode4Sub.getChildrenList(0x7FFFFFFE).size());
        Assert.assertEquals((long)2L, (long)snapshotNode4Sub.getChildrenList(snapshot1.getId()).size());
        INode snapshotNode4Subsub = (INode)snapshotNode4Sub.getChildrenList(0x7FFFFFFE).get(0);
        Assert.assertTrue((boolean)snapshotNode4Subsub.asDirectory().isWithSnapshot());
        Assert.assertTrue((snapshotNode4Sub == snapshotNode4Subsub.getParent() ? 1 : 0) != 0);
        INodeDirectory snapshotSubsubDir = (INodeDirectory)snapshotNode4Subsub;
        children = snapshotSubsubDir.getChildrenList(0x7FFFFFFE);
        Assert.assertEquals((long)2L, (long)children.size());
        Assert.assertEquals((Object)((INode)children.get(0)).getLocalName(), (Object)metaChangeFile1.getName());
        Assert.assertEquals((Object)((INode)children.get(1)).getLocalName(), (Object)newFileAfterS0.getName());
        children = snapshotSubsubDir.getChildrenList(snapshot0.getId());
        Assert.assertEquals((long)1L, (long)children.size());
        INode child = (INode)children.get(0);
        Assert.assertEquals((Object)child.getLocalName(), (Object)metaChangeFile1.getName());
        INodeFile metaChangeFile1SCopy = child.asFile();
        Assert.assertTrue((boolean)metaChangeFile1SCopy.isWithSnapshot());
        Assert.assertFalse((boolean)metaChangeFile1SCopy.isUnderConstruction());
        Assert.assertEquals((long)2L, (long)metaChangeFile1SCopy.getFileReplication(0x7FFFFFFE));
        Assert.assertEquals((long)2L, (long)metaChangeFile1SCopy.getFileReplication(snapshot1.getId()));
        Assert.assertEquals((long)3L, (long)metaChangeFile1SCopy.getFileReplication(snapshot0.getId()));
    }

    @Test(timeout=300000L)
    public void testDeleteEarliestSnapshot1() throws Exception {
        Path file0 = new Path(this.sub, "file0");
        Path file1 = new Path(this.sub, "file1");
        DFSTestUtil.createFile((FileSystem)this.hdfs, file0, 1024L, (short)3, 0L);
        DFSTestUtil.createFile((FileSystem)this.hdfs, file1, 1024L, (short)3, 0L);
        String snapshotName = "s1";
        try {
            this.hdfs.deleteSnapshot(this.sub, snapshotName);
            Assert.fail((String)("SnapshotException expected: " + this.sub.toString() + " is not snapshottable yet"));
        }
        catch (Exception e) {
            GenericTestUtils.assertExceptionContains((String)("Directory is not a snapshottable directory: " + this.sub), (Throwable)e);
        }
        this.hdfs.allowSnapshot(this.sub);
        try {
            this.hdfs.deleteSnapshot(this.sub, snapshotName);
            Assert.fail((String)("SnapshotException expected: snapshot " + snapshotName + " does not exist for " + this.sub.toString()));
        }
        catch (Exception e) {
            GenericTestUtils.assertExceptionContains((String)("Cannot delete snapshot " + snapshotName + " from path " + this.sub.toString() + ": the snapshot does not exist."), (Throwable)e);
        }
        SnapshotTestHelper.createSnapshot(this.hdfs, this.sub, snapshotName);
        this.checkQuotaUsageComputation(this.sub, 3L, 6144L);
        this.hdfs.deleteSnapshot(this.sub, snapshotName);
        this.checkQuotaUsageComputation(this.sub, 3L, 6144L);
        this.hdfs.createSnapshot(this.sub, snapshotName);
        this.checkQuotaUsageComputation(this.sub, 3L, 6144L);
        Path newFile = new Path(this.sub, "newFile");
        DFSTestUtil.createFile((FileSystem)this.hdfs, newFile, 1024L, (short)3, 0L);
        String snapshotName2 = "s2";
        this.hdfs.createSnapshot(this.sub, snapshotName2);
        this.checkQuotaUsageComputation(this.sub, 4L, 9216L);
        Path ss = SnapshotTestHelper.getSnapshotPath(this.sub, snapshotName2, "newFile");
        FileStatus statusBeforeDeletion = this.hdfs.getFileStatus(ss);
        this.hdfs.deleteSnapshot(this.sub, snapshotName);
        this.checkQuotaUsageComputation(this.sub, 4L, 9216L);
        FileStatus statusAfterDeletion = this.hdfs.getFileStatus(ss);
        System.out.println("Before deletion: " + statusBeforeDeletion.toString() + "\n" + "After deletion: " + statusAfterDeletion.toString());
        Assert.assertEquals((Object)statusBeforeDeletion.toString(), (Object)statusAfterDeletion.toString());
    }

    @Test(timeout=300000L)
    public void testDeleteEarliestSnapshot2() throws Exception {
        Path noChangeDir = new Path(this.sub, "noChangeDir");
        Path noChangeFile = new Path(noChangeDir, "noChangeFile");
        Path metaChangeFile = new Path(noChangeDir, "metaChangeFile");
        Path metaChangeDir = new Path(noChangeDir, "metaChangeDir");
        Path toDeleteFile = new Path(metaChangeDir, "toDeleteFile");
        DFSTestUtil.createFile((FileSystem)this.hdfs, noChangeFile, 1024L, (short)3, 0L);
        DFSTestUtil.createFile((FileSystem)this.hdfs, metaChangeFile, 1024L, (short)3, 0L);
        DFSTestUtil.createFile((FileSystem)this.hdfs, toDeleteFile, 1024L, (short)3, 0L);
        INodeFile toDeleteFileNode = TestSnapshotBlocksMap.assertBlockCollection(toDeleteFile.toString(), 1, this.fsdir, this.blockmanager);
        BlockInfo[] blocks = toDeleteFileNode.getBlocks();
        SnapshotTestHelper.createSnapshot(this.hdfs, this.dir, "s0");
        this.checkQuotaUsageComputation(this.dir, 7L, 9216L);
        this.hdfs.delete(toDeleteFile, true);
        this.checkQuotaUsageComputation(this.dir, 7L, 9216L);
        this.hdfs.setReplication(metaChangeFile, (short)2);
        this.hdfs.setOwner(metaChangeDir, "unknown", "unknown");
        this.checkQuotaUsageComputation(this.dir, 7L, 9216L);
        this.hdfs.createSnapshot(this.dir, "s1");
        this.checkQuotaUsageComputation(this.dir, 7L, 9216L);
        this.hdfs.deleteSnapshot(this.dir, "s0");
        this.checkQuotaUsageComputation(this.dir, 6L, 5120L);
        for (BlockInfo b : blocks) {
            Assert.assertEquals((long)-1L, (long)b.getBlockCollectionId());
        }
        INodeDirectory dirNode = this.fsdir.getINode(this.dir.toString()).asDirectory();
        Snapshot snapshot0 = dirNode.getSnapshot(DFSUtil.string2Bytes((String)"s0"));
        Assert.assertNull((Object)snapshot0);
        Snapshot snapshot1 = dirNode.getSnapshot(DFSUtil.string2Bytes((String)"s1"));
        DirectoryWithSnapshotFeature.DirectoryDiffList diffList = dirNode.getDiffs();
        Assert.assertEquals((long)1L, (long)diffList.asList().size());
        Assert.assertEquals((long)snapshot1.getId(), (long)((DirectoryWithSnapshotFeature.DirectoryDiff)diffList.getLast()).getSnapshotId());
        diffList = this.fsdir.getINode(metaChangeDir.toString()).asDirectory().getDiffs();
        Assert.assertEquals((long)0L, (long)diffList.asList().size());
        INodeDirectory noChangeDirNode = (INodeDirectory)this.fsdir.getINode(noChangeDir.toString());
        Assert.assertEquals(INodeDirectory.class, noChangeDirNode.getClass());
        INodeFile noChangeFileNode = (INodeFile)this.fsdir.getINode(noChangeFile.toString());
        Assert.assertEquals(INodeFile.class, noChangeFileNode.getClass());
        TestSnapshotBlocksMap.assertBlockCollection(noChangeFile.toString(), 1, this.fsdir, this.blockmanager);
        FileStatus status = this.hdfs.getFileStatus(metaChangeDir);
        Assert.assertEquals((Object)"unknown", (Object)status.getOwner());
        Assert.assertEquals((Object)"unknown", (Object)status.getGroup());
        status = this.hdfs.getFileStatus(metaChangeFile);
        Assert.assertEquals((long)2L, (long)status.getReplication());
        TestSnapshotBlocksMap.assertBlockCollection(metaChangeFile.toString(), 1, this.fsdir, this.blockmanager);
        try {
            this.hdfs.getFileStatus(toDeleteFile);
            Assert.fail((String)"should throw FileNotFoundException");
        }
        catch (FileNotFoundException e) {
            GenericTestUtils.assertExceptionContains((String)("File does not exist: " + toDeleteFile.toString()), (Throwable)e);
        }
        Path toDeleteFileInSnapshot = SnapshotTestHelper.getSnapshotPath(this.dir, "s0", toDeleteFile.toString().substring(this.dir.toString().length()));
        try {
            this.hdfs.getFileStatus(toDeleteFileInSnapshot);
            Assert.fail((String)"should throw FileNotFoundException");
        }
        catch (FileNotFoundException e) {
            GenericTestUtils.assertExceptionContains((String)("File does not exist: " + toDeleteFileInSnapshot.toString()), (Throwable)e);
        }
    }

    @Test(timeout=60000L)
    public void testDeleteSnapshot1() throws Exception {
        Path root = new Path("/");
        Path dir = new Path("/dir1");
        Path file1 = new Path(dir, "file1");
        DFSTestUtil.createFile((FileSystem)this.hdfs, file1, 1024L, (short)3, 0L);
        this.hdfs.allowSnapshot(root);
        this.hdfs.createSnapshot(root, "s1");
        Path file2 = new Path(dir, "file2");
        DFSTestUtil.createFile((FileSystem)this.hdfs, file2, 1024L, (short)3, 0L);
        this.hdfs.createSnapshot(root, "s2");
        this.hdfs.delete(file1, true);
        this.hdfs.delete(file2, true);
        Assert.assertTrue((boolean)this.hdfs.delete(dir, false));
        this.hdfs.deleteSnapshot(root, "s2");
        NameNodeAdapter.enterSafeMode(this.cluster.getNameNode(), false);
        NameNodeAdapter.saveNamespace(this.cluster.getNameNode());
        this.cluster.restartNameNodes();
    }

    @Test(timeout=60000L)
    public void testDeleteSnapshot2() throws Exception {
        Path root = new Path("/");
        Path dir = new Path("/dir1");
        Path file1 = new Path(dir, "file1");
        DFSTestUtil.createFile((FileSystem)this.hdfs, file1, 1024L, (short)3, 0L);
        this.hdfs.allowSnapshot(root);
        this.hdfs.createSnapshot(root, "s1");
        Path file2 = new Path(dir, "file2");
        DFSTestUtil.createFile((FileSystem)this.hdfs, file2, 1024L, (short)3, 0L);
        INodeFile file2Node = this.fsdir.getINode(file2.toString()).asFile();
        long file2NodeId = file2Node.getId();
        this.hdfs.createSnapshot(root, "s2");
        Assert.assertTrue((boolean)this.hdfs.delete(dir, true));
        Assert.assertNotNull((Object)this.fsdir.getInode(file2NodeId));
        this.hdfs.deleteSnapshot(root, "s2");
        Assert.assertTrue((this.fsdir.getInode(file2NodeId) == null ? 1 : 0) != 0);
        NameNodeAdapter.enterSafeMode(this.cluster.getNameNode(), false);
        NameNodeAdapter.saveNamespace(this.cluster.getNameNode());
        this.cluster.restartNameNodes();
    }

    @Test(timeout=300000L)
    public void testCombineSnapshotDiff1() throws Exception {
        this.testCombineSnapshotDiffImpl(this.sub, "", 1);
    }

    @Test(timeout=300000L)
    public void testCombineSnapshotDiff2() throws Exception {
        this.testCombineSnapshotDiffImpl(this.sub, "subsub1/subsubsub1/", 3);
    }

    @Test(timeout=300000L)
    public void testCombineSnapshotDiff3() throws Exception {
        Path dir = new Path("/dir");
        Path subDir1 = new Path(dir, "subdir1");
        Path subDir2 = new Path(dir, "subdir2");
        this.hdfs.mkdirs(subDir2);
        Path subsubDir = new Path(subDir1, "subsubdir");
        this.hdfs.mkdirs(subsubDir);
        SnapshotTestHelper.createSnapshot(this.hdfs, dir, "s1");
        Path newDir = new Path(subsubDir, "newdir");
        Path newFile = new Path(newDir, "newfile");
        DFSTestUtil.createFile((FileSystem)this.hdfs, newFile, 1024L, (short)3, 0L);
        Path newFile2 = new Path(subDir2, "newfile");
        DFSTestUtil.createFile((FileSystem)this.hdfs, newFile2, 1024L, (short)3, 0L);
        SnapshotTestHelper.createSnapshot(this.hdfs, dir, "s2");
        this.checkQuotaUsageComputation(dir, 7L, 6144L);
        this.hdfs.delete(subsubDir, true);
        this.hdfs.delete(subDir2, true);
        this.checkQuotaUsageComputation(dir, 7L, 6144L);
        this.hdfs.deleteSnapshot(dir, "s2");
        this.checkQuotaUsageComputation(dir, 4L, 0L);
        Path subdir1_s1 = SnapshotTestHelper.getSnapshotPath(dir, "s1", subDir1.getName());
        Path subdir1_s2 = SnapshotTestHelper.getSnapshotPath(dir, "s2", subDir1.getName());
        Assert.assertTrue((boolean)this.hdfs.exists(subdir1_s1));
        Assert.assertFalse((boolean)this.hdfs.exists(subdir1_s2));
    }

    private void testCombineSnapshotDiffImpl(Path snapshotRoot, String modDirStr, int dirNodeNum) throws Exception {
        Path modDir = modDirStr.isEmpty() ? snapshotRoot : new Path(snapshotRoot, modDirStr);
        Path file10 = new Path(modDir, "file10");
        Path file11 = new Path(modDir, "file11");
        Path file12 = new Path(modDir, "file12");
        Path file13 = new Path(modDir, "file13");
        Path file14 = new Path(modDir, "file14");
        Path file15 = new Path(modDir, "file15");
        DFSTestUtil.createFile((FileSystem)this.hdfs, file10, 1024L, (short)2, 0L);
        DFSTestUtil.createFile((FileSystem)this.hdfs, file11, 1024L, (short)2, 0L);
        DFSTestUtil.createFile((FileSystem)this.hdfs, file12, 1024L, (short)2, 0L);
        DFSTestUtil.createFile((FileSystem)this.hdfs, file13, 1024L, (short)2, 0L);
        SnapshotTestHelper.createSnapshot(this.hdfs, snapshotRoot, "s1");
        this.checkQuotaUsageComputation(snapshotRoot, dirNodeNum + 4, 8192L);
        this.hdfs.delete(file11, true);
        this.checkQuotaUsageComputation(snapshotRoot, dirNodeNum + 4, 8192L);
        this.hdfs.setReplication(file12, (short)3);
        this.checkQuotaUsageComputation(snapshotRoot, dirNodeNum + 4, 9216L);
        this.hdfs.setReplication(file13, (short)3);
        this.checkQuotaUsageComputation(snapshotRoot, dirNodeNum + 4, 10240L);
        DFSTestUtil.createFile((FileSystem)this.hdfs, file14, 1024L, (short)3, 0L);
        this.checkQuotaUsageComputation(snapshotRoot, dirNodeNum + 5, 13312L);
        DFSTestUtil.createFile((FileSystem)this.hdfs, file15, 1024L, (short)3, 0L);
        this.checkQuotaUsageComputation(snapshotRoot, dirNodeNum + 6, 16384L);
        this.hdfs.createSnapshot(snapshotRoot, "s2");
        this.checkQuotaUsageComputation(snapshotRoot, dirNodeNum + 6, 16384L);
        DFSTestUtil.createFile((FileSystem)this.hdfs, file11, 1024L, (short)3, 0L);
        this.checkQuotaUsageComputation(snapshotRoot, dirNodeNum + 7, 19456L);
        this.hdfs.delete(file12, true);
        this.checkQuotaUsageComputation(snapshotRoot, dirNodeNum + 7, 19456L);
        this.hdfs.setReplication(file13, (short)1);
        this.checkQuotaUsageComputation(snapshotRoot, dirNodeNum + 7, 19456L);
        this.hdfs.delete(file14, true);
        this.checkQuotaUsageComputation(snapshotRoot, dirNodeNum + 7, 19456L);
        this.hdfs.setReplication(file15, (short)2);
        this.checkQuotaUsageComputation(snapshotRoot, dirNodeNum + 7, 19456L);
        this.hdfs.createSnapshot(snapshotRoot, "s3");
        this.checkQuotaUsageComputation(snapshotRoot, dirNodeNum + 7, 19456L);
        this.hdfs.setReplication(file10, (short)3);
        this.checkQuotaUsageComputation(snapshotRoot, dirNodeNum + 7, 20480L);
        Path file10_s1 = SnapshotTestHelper.getSnapshotPath(snapshotRoot, "s1", modDirStr + "file10");
        Path file11_s1 = SnapshotTestHelper.getSnapshotPath(snapshotRoot, "s1", modDirStr + "file11");
        Path file12_s1 = SnapshotTestHelper.getSnapshotPath(snapshotRoot, "s1", modDirStr + "file12");
        Path file13_s1 = SnapshotTestHelper.getSnapshotPath(snapshotRoot, "s1", modDirStr + "file13");
        Path file14_s2 = SnapshotTestHelper.getSnapshotPath(snapshotRoot, "s2", modDirStr + "file14");
        Path file15_s2 = SnapshotTestHelper.getSnapshotPath(snapshotRoot, "s2", modDirStr + "file15");
        FileStatus statusBeforeDeletion10 = this.hdfs.getFileStatus(file10_s1);
        FileStatus statusBeforeDeletion11 = this.hdfs.getFileStatus(file11_s1);
        FileStatus statusBeforeDeletion12 = this.hdfs.getFileStatus(file12_s1);
        FileStatus statusBeforeDeletion13 = this.hdfs.getFileStatus(file13_s1);
        INodeFile file14Node = TestSnapshotBlocksMap.assertBlockCollection(file14_s2.toString(), 1, this.fsdir, this.blockmanager);
        BlockInfo[] blocks_14 = file14Node.getBlocks();
        TestSnapshotBlocksMap.assertBlockCollection(file15_s2.toString(), 1, this.fsdir, this.blockmanager);
        this.hdfs.deleteSnapshot(snapshotRoot, "s2");
        this.checkQuotaUsageComputation(snapshotRoot, dirNodeNum + 6, 14336L);
        FileStatus statusAfterDeletion10 = this.hdfs.getFileStatus(file10_s1);
        FileStatus statusAfterDeletion11 = this.hdfs.getFileStatus(file11_s1);
        FileStatus statusAfterDeletion12 = this.hdfs.getFileStatus(file12_s1);
        FileStatus statusAfterDeletion13 = this.hdfs.getFileStatus(file13_s1);
        Assert.assertEquals((Object)statusBeforeDeletion10.toString(), (Object)statusAfterDeletion10.toString());
        Assert.assertEquals((Object)statusBeforeDeletion11.toString(), (Object)statusAfterDeletion11.toString());
        Assert.assertEquals((Object)statusBeforeDeletion12.toString(), (Object)statusAfterDeletion12.toString());
        Assert.assertEquals((Object)statusBeforeDeletion13.toString(), (Object)statusAfterDeletion13.toString());
        TestSnapshotBlocksMap.assertBlockCollection(file10_s1.toString(), 1, this.fsdir, this.blockmanager);
        TestSnapshotBlocksMap.assertBlockCollection(file11_s1.toString(), 1, this.fsdir, this.blockmanager);
        TestSnapshotBlocksMap.assertBlockCollection(file12_s1.toString(), 1, this.fsdir, this.blockmanager);
        TestSnapshotBlocksMap.assertBlockCollection(file13_s1.toString(), 1, this.fsdir, this.blockmanager);
        Path file14_s1 = SnapshotTestHelper.getSnapshotPath(snapshotRoot, "s1", modDirStr + "file14");
        Path file15_s1 = SnapshotTestHelper.getSnapshotPath(snapshotRoot, "s1", modDirStr + "file15");
        Assert.assertFalse((boolean)this.hdfs.exists(file14_s1));
        Assert.assertFalse((boolean)this.hdfs.exists(file15_s1));
        for (BlockInfo b : blocks_14) {
            Assert.assertEquals((long)-1L, (long)b.getBlockCollectionId());
        }
        INodeFile nodeFile13 = (INodeFile)this.fsdir.getINode(file13.toString());
        for (BlockInfo b : nodeFile13.getBlocks()) {
            Assert.assertEquals((long)2L, (long)b.getReplication());
        }
        TestSnapshotBlocksMap.assertBlockCollection(file13.toString(), 1, this.fsdir, this.blockmanager);
        INodeFile nodeFile12 = (INodeFile)this.fsdir.getINode(file12_s1.toString());
        for (BlockInfo b : nodeFile12.getBlocks()) {
            Assert.assertEquals((long)2L, (long)b.getReplication());
        }
    }

    @Test(timeout=300000L)
    public void testDeleteSnapshotWithDirModification() throws Exception {
        Path file = new Path(this.sub, "file");
        DFSTestUtil.createFile((FileSystem)this.hdfs, file, 1024L, (short)3, 0L);
        this.hdfs.setOwner(this.sub, "user1", "group1");
        SnapshotTestHelper.createSnapshot(this.hdfs, this.sub, "s1");
        this.checkQuotaUsageComputation(this.sub, 2L, 3072L);
        this.hdfs.setOwner(this.sub, "user2", "group2");
        this.checkQuotaUsageComputation(this.sub, 2L, 3072L);
        this.hdfs.createSnapshot(this.sub, "s2");
        this.checkQuotaUsageComputation(this.sub, 2L, 3072L);
        this.hdfs.createSnapshot(this.sub, "s3");
        this.checkQuotaUsageComputation(this.sub, 2L, 3072L);
        this.hdfs.setOwner(this.sub, "user3", "group3");
        this.checkQuotaUsageComputation(this.sub, 2L, 3072L);
        this.hdfs.deleteSnapshot(this.sub, "s3");
        this.checkQuotaUsageComputation(this.sub, 2L, 3072L);
        FileStatus statusOfS2 = this.hdfs.getFileStatus(new Path(this.sub, ".snapshot/s2"));
        Assert.assertEquals((Object)"user2", (Object)statusOfS2.getOwner());
        Assert.assertEquals((Object)"group2", (Object)statusOfS2.getGroup());
        this.hdfs.deleteSnapshot(this.sub, "s2");
        this.checkQuotaUsageComputation(this.sub, 2L, 3072L);
        FileStatus statusOfS1 = this.hdfs.getFileStatus(new Path(this.sub, ".snapshot/s1"));
        Assert.assertEquals((Object)"user1", (Object)statusOfS1.getOwner());
        Assert.assertEquals((Object)"group1", (Object)statusOfS1.getGroup());
    }

    @Test
    public void testDeleteSnapshotWithPermissionsDisabled() throws Exception {
        this.cluster.shutdown();
        Configuration newConf = new Configuration(this.conf);
        newConf.setBoolean("dfs.permissions.enabled", false);
        this.cluster = new MiniDFSCluster.Builder(newConf).numDataNodes(0).build();
        this.cluster.waitActive();
        this.hdfs = this.cluster.getFileSystem();
        final Path path = new Path("/dir");
        this.hdfs.mkdirs(path);
        this.hdfs.allowSnapshot(path);
        this.hdfs.mkdirs(new Path(path, "/test"));
        this.hdfs.createSnapshot(path, "s1");
        UserGroupInformation anotherUser = UserGroupInformation.createRemoteUser((String)"anotheruser");
        anotherUser.doAs((PrivilegedAction)new PrivilegedAction<Object>(){

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public Object run() {
                DistributedFileSystem anotherUserFS = null;
                try {
                    anotherUserFS = TestSnapshotDeletion.this.cluster.getFileSystem();
                    anotherUserFS.deleteSnapshot(path, "s1");
                }
                catch (IOException e) {
                    try {
                        Assert.fail((String)("Failed to delete snapshot : " + e.getLocalizedMessage()));
                    }
                    catch (Throwable throwable) {
                        IOUtils.closeStream(anotherUserFS);
                        throw throwable;
                    }
                    IOUtils.closeStream((Closeable)anotherUserFS);
                }
                IOUtils.closeStream((Closeable)anotherUserFS);
                return null;
            }
        });
    }

    @Test(timeout=300000L)
    public void testRenameSnapshotDiff() throws Exception {
        this.cluster.getNamesystem().getSnapshotManager().setAllowNestedSnapshots(true);
        Path subFile0 = new Path(this.sub, "file0");
        Path subsubFile0 = new Path(this.subsub, "file0");
        DFSTestUtil.createFile((FileSystem)this.hdfs, subFile0, 1024L, (short)3, 0L);
        DFSTestUtil.createFile((FileSystem)this.hdfs, subsubFile0, 1024L, (short)3, 0L);
        this.hdfs.setOwner(this.subsub, "owner", "group");
        SnapshotTestHelper.createSnapshot(this.hdfs, this.sub, "s0");
        this.checkQuotaUsageComputation(this.sub, 4L, 6144L);
        Path subFile1 = new Path(this.sub, "file1");
        Path subsubFile1 = new Path(this.subsub, "file1");
        DFSTestUtil.createFile((FileSystem)this.hdfs, subFile1, 1024L, (short)2, 0L);
        DFSTestUtil.createFile((FileSystem)this.hdfs, subsubFile1, 1024L, (short)3, 0L);
        this.checkQuotaUsageComputation(this.sub, 6L, 11264L);
        SnapshotTestHelper.createSnapshot(this.hdfs, this.sub, "s1");
        this.checkQuotaUsageComputation(this.sub, 6L, 11264L);
        SnapshotTestHelper.createSnapshot(this.hdfs, this.dir, "s2");
        this.checkQuotaUsageComputation(this.dir, 7L, 11264L);
        this.checkQuotaUsageComputation(this.sub, 6L, 11264L);
        this.hdfs.setOwner(this.subsub, "unknown", "unknown");
        this.hdfs.setReplication(subsubFile1, (short)2);
        this.checkQuotaUsageComputation(this.dir, 7L, 11264L);
        this.checkQuotaUsageComputation(this.sub, 6L, 11264L);
        this.hdfs.delete(subFile1, true);
        this.checkQuotaUsageComputation(new Path("/"), 8L, 11264L);
        this.checkQuotaUsageComputation(this.dir, 7L, 11264L);
        this.checkQuotaUsageComputation(this.sub, 6L, 11264L);
        Path subsubSnapshotCopy = SnapshotTestHelper.getSnapshotPath(this.dir, "s2", this.sub.getName() + "/" + this.subsub.getName());
        Path subsubFile1SCopy = SnapshotTestHelper.getSnapshotPath(this.dir, "s2", this.sub.getName() + "/" + this.subsub.getName() + "/" + subsubFile1.getName());
        Path subFile1SCopy = SnapshotTestHelper.getSnapshotPath(this.dir, "s2", this.sub.getName() + "/" + subFile1.getName());
        FileStatus subsubStatus = this.hdfs.getFileStatus(subsubSnapshotCopy);
        Assert.assertEquals((Object)"owner", (Object)subsubStatus.getOwner());
        Assert.assertEquals((Object)"group", (Object)subsubStatus.getGroup());
        FileStatus subsubFile1Status = this.hdfs.getFileStatus(subsubFile1SCopy);
        Assert.assertEquals((long)3L, (long)subsubFile1Status.getReplication());
        FileStatus subFile1Status = this.hdfs.getFileStatus(subFile1SCopy);
        Assert.assertEquals((long)2L, (long)subFile1Status.getReplication());
        this.hdfs.deleteSnapshot(this.dir, "s2");
        this.checkQuotaUsageComputation(new Path("/"), 8L, 11264L);
        this.checkQuotaUsageComputation(this.dir, 7L, 11264L);
        this.checkQuotaUsageComputation(this.sub, 6L, 11264L);
        try {
            this.hdfs.getFileStatus(subsubSnapshotCopy);
            Assert.fail((String)"should throw FileNotFoundException");
        }
        catch (FileNotFoundException e) {
            GenericTestUtils.assertExceptionContains((String)("File does not exist: " + subsubSnapshotCopy.toString()), (Throwable)e);
        }
        try {
            this.hdfs.getFileStatus(subsubFile1SCopy);
            Assert.fail((String)"should throw FileNotFoundException");
        }
        catch (FileNotFoundException e) {
            GenericTestUtils.assertExceptionContains((String)("File does not exist: " + subsubFile1SCopy.toString()), (Throwable)e);
        }
        try {
            this.hdfs.getFileStatus(subFile1SCopy);
            Assert.fail((String)"should throw FileNotFoundException");
        }
        catch (FileNotFoundException e) {
            GenericTestUtils.assertExceptionContains((String)("File does not exist: " + subFile1SCopy.toString()), (Throwable)e);
        }
        subsubSnapshotCopy = SnapshotTestHelper.getSnapshotPath(this.sub, "s1", this.subsub.getName());
        subsubFile1SCopy = SnapshotTestHelper.getSnapshotPath(this.sub, "s1", this.subsub.getName() + "/" + subsubFile1.getName());
        subFile1SCopy = SnapshotTestHelper.getSnapshotPath(this.sub, "s1", subFile1.getName());
        subsubStatus = this.hdfs.getFileStatus(subsubSnapshotCopy);
        Assert.assertEquals((Object)"owner", (Object)subsubStatus.getOwner());
        Assert.assertEquals((Object)"group", (Object)subsubStatus.getGroup());
        subsubFile1Status = this.hdfs.getFileStatus(subsubFile1SCopy);
        Assert.assertEquals((long)3L, (long)subsubFile1Status.getReplication());
        subFile1Status = this.hdfs.getFileStatus(subFile1SCopy);
        Assert.assertEquals((long)2L, (long)subFile1Status.getReplication());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testDeleteSnapshotCommandWithIllegalArguments() throws Exception {
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        PrintStream psOut = new PrintStream(out);
        PrintStream oldOut = System.out;
        PrintStream oldErr = System.err;
        try {
            System.setOut(psOut);
            System.setErr(psOut);
            FsShell shell = new FsShell();
            shell.setConf(this.conf);
            String[] argv1 = new String[]{"-deleteSnapshot", "/tmp"};
            int val = shell.run(argv1);
            Assert.assertTrue((val == -1 ? 1 : 0) != 0);
            Assert.assertTrue((boolean)out.toString().contains(argv1[0] + ": Incorrect number of arguments."));
            out.reset();
            String[] argv2 = new String[]{"-deleteSnapshot", "/tmp", "s1", "s2"};
            val = shell.run(argv2);
            Assert.assertTrue((val == -1 ? 1 : 0) != 0);
            Assert.assertTrue((boolean)out.toString().contains(argv2[0] + ": Incorrect number of arguments."));
            psOut.close();
            out.close();
        }
        finally {
            System.setOut(oldOut);
            System.setErr(oldErr);
        }
    }

    @Test(timeout=60000L)
    public void testHANNRestartAfterSnapshotDeletion() throws Exception {
        this.hdfs.close();
        this.cluster.shutdown();
        this.conf = new Configuration();
        this.cluster = new MiniDFSCluster.Builder(this.conf).nnTopology(MiniDFSNNTopology.simpleHATopology()).numDataNodes(1).build();
        this.cluster.transitionToActive(0);
        NameNode snn = this.cluster.getNameNode(1);
        snn.stop();
        this.hdfs = HATestUtil.configureFailoverFs(this.cluster, this.conf);
        Path dir = new Path("/dir");
        Path subDir = new Path(dir, "sub");
        this.hdfs.mkdirs(dir);
        this.hdfs.allowSnapshot(dir);
        for (int i = 0; i < 5; ++i) {
            DFSTestUtil.createFile((FileSystem)this.hdfs, new Path(subDir, "" + i), 100L, (short)1, 1024L);
        }
        this.hdfs.createSnapshot(dir, "s0");
        this.hdfs.delete(subDir, true);
        NameNode ann = this.cluster.getNameNode(0);
        ann.getRpcServer().rollEditLog();
        this.hdfs.deleteSnapshot(dir, "s0");
        Thread.sleep(2000L);
        NameNodeAdapter.abortEditLogs(ann);
        this.cluster.restartNameNode(0, false, new String[0]);
        this.cluster.transitionToActive(0);
        this.cluster.waitClusterUp();
    }

    @Test
    public void testCorrectNumberOfBlocksAfterRestart() throws IOException {
        Path foo = new Path("/foo");
        Path bar = new Path(foo, "bar");
        Path file = new Path(foo, "file");
        String snapshotName = "ss0";
        DFSTestUtil.createFile((FileSystem)this.hdfs, file, 1024L, (short)3, 0L);
        this.hdfs.mkdirs(bar);
        this.hdfs.setQuota(foo, 0x7FFFFFFFFFFFFFFEL, 0x7FFFFFFFFFFFFFFEL);
        this.hdfs.setQuota(bar, 0x7FFFFFFFFFFFFFFEL, 0x7FFFFFFFFFFFFFFEL);
        this.hdfs.allowSnapshot(foo);
        this.hdfs.createSnapshot(foo, "ss0");
        this.hdfs.setSafeMode(HdfsConstants.SafeModeAction.SAFEMODE_ENTER);
        this.hdfs.saveNamespace();
        this.hdfs.setSafeMode(HdfsConstants.SafeModeAction.SAFEMODE_LEAVE);
        this.hdfs.deleteSnapshot(foo, "ss0");
        this.hdfs.delete(bar, true);
        this.hdfs.delete(foo, true);
        long numberOfBlocks = this.cluster.getNamesystem().getBlocksTotal();
        this.cluster.restartNameNode(0);
        Assert.assertEquals((long)numberOfBlocks, (long)this.cluster.getNamesystem().getBlocksTotal());
    }

    @Test
    public void testFsImageCorruption() throws Exception {
        Path st = new Path("/st");
        Path nonst = new Path("/nonst");
        Path stY = new Path(st, "y");
        Path nonstTrash = new Path(nonst, "trash");
        this.hdfs.mkdirs(stY);
        this.hdfs.allowSnapshot(st);
        this.hdfs.createSnapshot(st, "s0");
        Path f = new Path(stY, "nn.log");
        this.hdfs.createNewFile(f);
        this.hdfs.createSnapshot(st, "s1");
        Path f2 = new Path(stY, "nn2.log");
        this.hdfs.rename(f, f2);
        this.hdfs.createSnapshot(st, "s2");
        Path trashSt = new Path(nonstTrash, "st");
        this.hdfs.mkdirs(trashSt);
        this.hdfs.rename(stY, trashSt);
        this.hdfs.delete(nonstTrash, true);
        this.hdfs.deleteSnapshot(st, "s1");
        this.hdfs.deleteSnapshot(st, "s2");
        this.hdfs.setSafeMode(HdfsConstants.SafeModeAction.SAFEMODE_ENTER);
        this.hdfs.saveNamespace();
        this.hdfs.setSafeMode(HdfsConstants.SafeModeAction.SAFEMODE_LEAVE);
        this.cluster.restartNameNodes();
    }

    @Test
    public void testRenameAndDelete() throws IOException {
        Path foo = new Path("/foo");
        Path x = new Path(foo, "x");
        Path y = new Path(foo, "y");
        Path trash = new Path("/trash");
        this.hdfs.mkdirs(x);
        this.hdfs.mkdirs(y);
        long parentId = this.fsdir.getINode4Write(y.toString()).getId();
        this.hdfs.mkdirs(trash);
        this.hdfs.allowSnapshot(foo);
        this.hdfs.createSnapshot(foo, "s0");
        Path file = new Path(x, "bar");
        DFSTestUtil.createFile((FileSystem)this.hdfs, file, 1024L, (short)1, 0L);
        long fileId = this.fsdir.getINode4Write(file.toString()).getId();
        Path newFile = new Path(y, "bar");
        this.hdfs.rename(file, newFile);
        this.hdfs.createSnapshot(foo, "s1");
        Path deletedY = new Path(trash, "y");
        this.hdfs.rename(y, deletedY);
        this.hdfs.createSnapshot(foo, "s2");
        this.hdfs.delete(deletedY, true);
        this.hdfs.deleteSnapshot(foo, "s1");
        INode p = this.fsdir.getInode(parentId);
        Assert.assertNotNull((Object)p);
        INodeDirectory pd = p.asDirectory();
        Assert.assertNotNull((Object)pd);
        Assert.assertNull((Object)pd.getChild("bar".getBytes(), 0x7FFFFFFE));
        Assert.assertNull((Object)this.fsdir.getInode(fileId));
    }
}

