/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.qjournal.server;

import java.io.ByteArrayOutputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.IOException;
import java.nio.Buffer;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import org.apache.commons.io.FileUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdfs.qjournal.QJMTestUtil;
import org.apache.hadoop.hdfs.qjournal.server.Journal;
import org.apache.hadoop.hdfs.qjournal.server.JournaledEditsCache;
import org.apache.hadoop.hdfs.server.namenode.EditLogFileOutputStream;
import org.apache.hadoop.hdfs.server.namenode.NameNodeLayoutVersion;
import org.apache.hadoop.test.GenericTestUtils;
import org.apache.hadoop.test.PathUtils;
import org.apache.hadoop.thirdparty.com.google.common.primitives.Bytes;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.slf4j.Logger;

public class TestJournaledEditsCache {
    private static final int EDITS_CAPACITY = 100;
    private static final File TEST_DIR = PathUtils.getTestDir(TestJournaledEditsCache.class, false);
    private JournaledEditsCache cache;

    @Before
    public void setup() throws Exception {
        Configuration conf = new Configuration();
        conf.setInt("dfs.journalnode.edit-cache-size.bytes", QJMTestUtil.createTxnData(1, 1).length * 100);
        this.cache = new JournaledEditsCache(conf);
        TEST_DIR.mkdirs();
    }

    @After
    public void cleanup() throws Exception {
        FileUtils.deleteQuietly((File)TEST_DIR);
    }

    @Test
    public void testCacheSingleSegment() throws Exception {
        this.storeEdits(1, 20);
        this.assertTxnCountAndContents(1, 5, 5);
        this.assertTxnCountAndContents(1, 20, 20);
        this.assertTxnCountAndContents(1, 40, 20);
        this.assertTxnCountAndContents(10, 11, 20);
        this.assertTxnCountAndContents(10, 20, 20);
    }

    @Test
    public void testCacheBelowCapacityRequestOnBoundary() throws Exception {
        this.storeEdits(1, 5);
        this.storeEdits(6, 20);
        this.storeEdits(21, 30);
        this.assertTxnCountAndContents(1, 3, 3);
        this.assertTxnCountAndContents(6, 10, 15);
        this.assertTxnCountAndContents(1, 7, 7);
        this.assertTxnCountAndContents(1, 25, 25);
        this.assertTxnCountAndContents(6, 20, 25);
        this.assertTxnCountAndContents(6, 50, 30);
        this.assertTxnCountAndContents(21, 20, 30);
    }

    @Test
    public void testCacheBelowCapacityRequestOffBoundary() throws Exception {
        this.storeEdits(1, 5);
        this.storeEdits(6, 20);
        this.storeEdits(21, 30);
        this.assertTxnCountAndContents(3, 1, 3);
        this.assertTxnCountAndContents(3, 6, 8);
        this.assertTxnCountAndContents(15, 10, 24);
        this.assertTxnCountAndContents(15, 50, 30);
        ArrayList buffers = new ArrayList();
        Assert.assertEquals((long)0L, (long)this.cache.retrieveEdits(31L, 10, buffers));
        Assert.assertTrue((boolean)buffers.isEmpty());
    }

    @Test
    public void testCacheAboveCapacity() throws Exception {
        int thirdCapacity = 33;
        this.storeEdits(1, thirdCapacity);
        this.storeEdits(thirdCapacity + 1, thirdCapacity * 2);
        this.storeEdits(thirdCapacity * 2 + 1, 100);
        this.storeEdits(101, thirdCapacity * 4);
        this.storeEdits(thirdCapacity * 4 + 1, thirdCapacity * 5);
        try {
            this.cache.retrieveEdits(1L, 10, new ArrayList());
            Assert.fail();
        }
        catch (IOException iOException) {
            // empty catch block
        }
        this.assertTxnCountAndContents(101, 100, thirdCapacity * 5);
    }

    @Test
    public void testCacheSingleAdditionAboveCapacity() throws Exception {
        GenericTestUtils.LogCapturer logs = GenericTestUtils.LogCapturer.captureLogs((Logger)Journal.LOG);
        this.storeEdits(1, 200);
        logs.stopCapturing();
        Assert.assertTrue((boolean)logs.getOutput().contains("batch of edits was too large"));
        try {
            this.cache.retrieveEdits(1L, 1, new ArrayList());
            Assert.fail();
        }
        catch (IOException iOException) {
            // empty catch block
        }
        this.storeEdits(201, 205);
        this.assertTxnCountAndContents(201, 5, 205);
    }

    @Test
    public void testCacheWithFutureLayoutVersion() throws Exception {
        byte[] firstHalf = QJMTestUtil.createGabageTxns(1L, 5);
        byte[] secondHalf = QJMTestUtil.createGabageTxns(6L, 5);
        int futureVersion = NameNodeLayoutVersion.CURRENT_LAYOUT_VERSION - 1;
        this.cache.storeEdits(Bytes.concat((byte[][])new byte[][]{firstHalf, secondHalf}), 1L, 10L, futureVersion);
        ArrayList buffers = new ArrayList();
        Assert.assertEquals((long)5L, (long)this.cache.retrieveEdits(6L, 5, buffers));
        Assert.assertArrayEquals((byte[])TestJournaledEditsCache.getHeaderForLayoutVersion(futureVersion), (byte[])((ByteBuffer)buffers.get(0)).array());
        byte[] retBytes = new byte[((ByteBuffer)buffers.get(1)).remaining()];
        System.arraycopy(((ByteBuffer)buffers.get(1)).array(), ((ByteBuffer)buffers.get(1)).position(), retBytes, 0, ((ByteBuffer)buffers.get(1)).remaining());
        Assert.assertArrayEquals((byte[])secondHalf, (byte[])retBytes);
    }

    @Test
    public void testCacheWithMultipleLayoutVersions() throws Exception {
        int oldLayout = NameNodeLayoutVersion.CURRENT_LAYOUT_VERSION + 1;
        this.cache.storeEdits(QJMTestUtil.createTxnData(1, 5), 1L, 5L, oldLayout);
        this.storeEdits(6, 10);
        try {
            this.cache.retrieveEdits(1L, 50, new ArrayList());
            Assert.fail((String)"Expected a cache miss");
        }
        catch (JournaledEditsCache.CacheMissException cacheMissException) {
            // empty catch block
        }
        this.assertTxnCountAndContents(6, 50, 10);
    }

    @Test
    public void testCacheEditsWithGaps() throws Exception {
        this.storeEdits(1, 5);
        this.storeEdits(10, 15);
        try {
            this.cache.retrieveEdits(1L, 20, new ArrayList());
            Assert.fail();
        }
        catch (JournaledEditsCache.CacheMissException cme) {
            Assert.assertEquals((long)9L, (long)cme.getCacheMissAmount());
        }
        this.assertTxnCountAndContents(10, 10, 15);
    }

    @Test(expected=JournaledEditsCache.CacheMissException.class)
    public void testReadUninitializedCache() throws Exception {
        this.cache.retrieveEdits(1L, 10, new ArrayList());
    }

    @Test(expected=JournaledEditsCache.CacheMissException.class)
    public void testCacheMalformedInput() throws Exception {
        this.storeEdits(1, 1);
        this.cache.retrieveEdits(-1L, 10, new ArrayList());
    }

    private void storeEdits(int startTxn, int endTxn) throws Exception {
        this.cache.storeEdits(QJMTestUtil.createTxnData(startTxn, endTxn - startTxn + 1), (long)startTxn, (long)endTxn, NameNodeLayoutVersion.CURRENT_LAYOUT_VERSION);
    }

    private void assertTxnCountAndContents(int startTxn, int requestedMaxTxns, int expectedEndTxn) throws Exception {
        ArrayList buffers = new ArrayList();
        int expectedTxnCount = expectedEndTxn - startTxn + 1;
        Assert.assertEquals((long)expectedTxnCount, (long)this.cache.retrieveEdits((long)startTxn, requestedMaxTxns, buffers));
        byte[] expectedBytes = Bytes.concat((byte[][])new byte[][]{TestJournaledEditsCache.getHeaderForLayoutVersion(NameNodeLayoutVersion.CURRENT_LAYOUT_VERSION), QJMTestUtil.createTxnData(startTxn, expectedTxnCount)});
        byte[] actualBytes = new byte[buffers.stream().mapToInt(Buffer::remaining).sum()];
        int pos = 0;
        for (ByteBuffer buf : buffers) {
            System.arraycopy(buf.array(), buf.position(), actualBytes, pos, buf.remaining());
            pos += buf.remaining();
        }
        Assert.assertArrayEquals((byte[])expectedBytes, (byte[])actualBytes);
    }

    private static byte[] getHeaderForLayoutVersion(int version) throws IOException {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        EditLogFileOutputStream.writeHeader((int)version, (DataOutputStream)new DataOutputStream(baos));
        return baos.toByteArray();
    }
}

