/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.io.hfile;

import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.IOException;
import java.lang.ref.WeakReference;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collections;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.PriorityQueue;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.io.HeapSize;
import org.apache.hadoop.hbase.io.encoding.DataBlockEncoding;
import org.apache.hadoop.hbase.io.hfile.BlockCache;
import org.apache.hadoop.hbase.io.hfile.BlockCacheColumnFamilySummary;
import org.apache.hadoop.hbase.io.hfile.BlockCacheKey;
import org.apache.hadoop.hbase.io.hfile.BlockType;
import org.apache.hadoop.hbase.io.hfile.CacheStats;
import org.apache.hadoop.hbase.io.hfile.Cacheable;
import org.apache.hadoop.hbase.io.hfile.CachedBlock;
import org.apache.hadoop.hbase.io.hfile.CachedBlockQueue;
import org.apache.hadoop.hbase.io.hfile.HFileBlock;
import org.apache.hadoop.hbase.io.hfile.bucket.BucketCache;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.ClassSize;
import org.apache.hadoop.hbase.util.FSUtils;
import org.apache.hadoop.hbase.util.HasThread;
import org.apache.hadoop.hbase.util.Threads;
import org.apache.hadoop.util.StringUtils;

@InterfaceAudience.Private
public class LruBlockCache
implements BlockCache,
HeapSize {
    static final Log LOG = LogFactory.getLog(LruBlockCache.class);
    static final String LRU_MIN_FACTOR_CONFIG_NAME = "hbase.lru.blockcache.min.factor";
    static final String LRU_ACCEPTABLE_FACTOR_CONFIG_NAME = "hbase.lru.blockcache.acceptable.factor";
    static final String LRU_SINGLE_PERCENTAGE_CONFIG_NAME = "hbase.lru.blockcache.single.percentage";
    static final String LRU_MULTI_PERCENTAGE_CONFIG_NAME = "hbase.lru.blockcache.multi.percentage";
    static final String LRU_MEMORY_PERCENTAGE_CONFIG_NAME = "hbase.lru.blockcache.memory.percentage";
    static final String LRU_IN_MEMORY_FORCE_MODE_CONFIG_NAME = "hbase.lru.rs.inmemoryforcemode";
    static final float DEFAULT_LOAD_FACTOR = 0.75f;
    static final int DEFAULT_CONCURRENCY_LEVEL = 16;
    static final float DEFAULT_MIN_FACTOR = 0.95f;
    static final float DEFAULT_ACCEPTABLE_FACTOR = 0.99f;
    static final float DEFAULT_SINGLE_FACTOR = 0.25f;
    static final float DEFAULT_MULTI_FACTOR = 0.5f;
    static final float DEFAULT_MEMORY_FACTOR = 0.25f;
    static final boolean DEFAULT_IN_MEMORY_FORCE_MODE = false;
    static final int statThreadPeriod = 300;
    private final ConcurrentHashMap<BlockCacheKey, CachedBlock> map;
    private final ReentrantLock evictionLock = new ReentrantLock(true);
    private volatile boolean evictionInProgress = false;
    private final EvictionThread evictionThread;
    private final ScheduledExecutorService scheduleThreadPool = Executors.newScheduledThreadPool(1, new ThreadFactoryBuilder().setNameFormat("LruStats #%d").setDaemon(true).build());
    private final AtomicLong size;
    private final AtomicLong elements;
    private final AtomicLong count;
    private final CacheStats stats;
    private long maxSize;
    private long blockSize;
    private float acceptableFactor;
    private float minFactor;
    private float singleFactor;
    private float multiFactor;
    private float memoryFactor;
    private long overhead;
    private boolean forceInMemory;
    private BucketCache victimHandler = null;
    public static final long CACHE_FIXED_OVERHEAD = ClassSize.align((int)(24 + 9 * ClassSize.REFERENCE + 20 + 1 + ClassSize.OBJECT));

    public LruBlockCache(long maxSize, long blockSize) {
        this(maxSize, blockSize, true);
    }

    public LruBlockCache(long maxSize, long blockSize, boolean evictionThread) {
        this(maxSize, blockSize, evictionThread, (int)Math.ceil(1.2 * (double)maxSize / (double)blockSize), 0.75f, 16, 0.95f, 0.99f, 0.25f, 0.5f, 0.25f, false);
    }

    public LruBlockCache(long maxSize, long blockSize, boolean evictionThread, Configuration conf) {
        this(maxSize, blockSize, evictionThread, (int)Math.ceil(1.2 * (double)maxSize / (double)blockSize), 0.75f, 16, conf.getFloat(LRU_MIN_FACTOR_CONFIG_NAME, 0.95f), conf.getFloat(LRU_ACCEPTABLE_FACTOR_CONFIG_NAME, 0.99f), conf.getFloat(LRU_SINGLE_PERCENTAGE_CONFIG_NAME, 0.25f), conf.getFloat(LRU_MULTI_PERCENTAGE_CONFIG_NAME, 0.5f), conf.getFloat(LRU_MEMORY_PERCENTAGE_CONFIG_NAME, 0.25f), conf.getBoolean(LRU_IN_MEMORY_FORCE_MODE_CONFIG_NAME, false));
    }

    public LruBlockCache(long maxSize, long blockSize, Configuration conf) {
        this(maxSize, blockSize, true, conf);
    }

    public LruBlockCache(long maxSize, long blockSize, boolean evictionThread, int mapInitialSize, float mapLoadFactor, int mapConcurrencyLevel, float minFactor, float acceptableFactor, float singleFactor, float multiFactor, float memoryFactor, boolean forceInMemory) {
        if (singleFactor + multiFactor + memoryFactor != 1.0f || singleFactor < 0.0f || multiFactor < 0.0f || memoryFactor < 0.0f) {
            throw new IllegalArgumentException("Single, multi, and memory factors  should be non-negative and total 1.0");
        }
        if (minFactor >= acceptableFactor) {
            throw new IllegalArgumentException("minFactor must be smaller than acceptableFactor");
        }
        if (minFactor >= 1.0f || acceptableFactor >= 1.0f) {
            throw new IllegalArgumentException("all factors must be < 1");
        }
        this.maxSize = maxSize;
        this.blockSize = blockSize;
        this.forceInMemory = forceInMemory;
        this.map = new ConcurrentHashMap(mapInitialSize, mapLoadFactor, mapConcurrencyLevel);
        this.minFactor = minFactor;
        this.acceptableFactor = acceptableFactor;
        this.singleFactor = singleFactor;
        this.multiFactor = multiFactor;
        this.memoryFactor = memoryFactor;
        this.stats = new CacheStats();
        this.count = new AtomicLong(0L);
        this.elements = new AtomicLong(0L);
        this.overhead = LruBlockCache.calculateOverhead(maxSize, blockSize, mapConcurrencyLevel);
        this.size = new AtomicLong(this.overhead);
        if (evictionThread) {
            this.evictionThread = new EvictionThread(this);
            this.evictionThread.start();
        } else {
            this.evictionThread = null;
        }
        this.scheduleThreadPool.scheduleAtFixedRate(new StatisticsThread(this), 300L, 300L, TimeUnit.SECONDS);
    }

    public void setMaxSize(long maxSize) {
        this.maxSize = maxSize;
        if (this.size.get() > this.acceptableSize() && !this.evictionInProgress) {
            this.runEviction();
        }
    }

    @Override
    public void cacheBlock(BlockCacheKey cacheKey, Cacheable buf, boolean inMemory) {
        CachedBlock cb = this.map.get(cacheKey);
        if (cb != null) {
            if (this.compare(buf, cb.getBuffer()) != 0) {
                throw new RuntimeException("Cached block contents differ, which should not have happened.cacheKey:" + cacheKey);
            }
            String msg = "Cached an already cached block: " + cacheKey + " cb:" + cb.getCacheKey();
            msg = msg + ". This is harmless and can happen in rare cases (see HBASE-8547)";
            LOG.warn((Object)msg);
            return;
        }
        cb = new CachedBlock(cacheKey, buf, this.count.incrementAndGet(), inMemory);
        long newSize = this.updateSizeMetrics(cb, false);
        this.map.put(cacheKey, cb);
        this.elements.incrementAndGet();
        if (newSize > this.acceptableSize() && !this.evictionInProgress) {
            this.runEviction();
        }
    }

    private int compare(Cacheable left, Cacheable right) {
        ByteBuffer l = ByteBuffer.allocate(left.getSerializedLength());
        left.serialize(l);
        ByteBuffer r = ByteBuffer.allocate(right.getSerializedLength());
        right.serialize(r);
        return Bytes.compareTo((byte[])l.array(), (int)l.arrayOffset(), (int)l.limit(), (byte[])r.array(), (int)r.arrayOffset(), (int)r.limit());
    }

    @Override
    public void cacheBlock(BlockCacheKey cacheKey, Cacheable buf) {
        this.cacheBlock(cacheKey, buf, false);
    }

    protected long updateSizeMetrics(CachedBlock cb, boolean evict) {
        long heapsize = cb.heapSize();
        if (evict) {
            heapsize *= -1L;
        }
        return this.size.addAndGet(heapsize);
    }

    @Override
    public Cacheable getBlock(BlockCacheKey cacheKey, boolean caching, boolean repeat) {
        CachedBlock cb = this.map.get(cacheKey);
        if (cb == null) {
            if (!repeat) {
                this.stats.miss(caching);
            }
            if (this.victimHandler != null) {
                return this.victimHandler.getBlock(cacheKey, caching, repeat);
            }
            return null;
        }
        this.stats.hit(caching);
        cb.access(this.count.incrementAndGet());
        return cb.getBuffer();
    }

    public boolean containsBlock(BlockCacheKey cacheKey) {
        return this.map.containsKey(cacheKey);
    }

    @Override
    public boolean evictBlock(BlockCacheKey cacheKey) {
        CachedBlock cb = this.map.get(cacheKey);
        if (cb == null) {
            return false;
        }
        this.evictBlock(cb, false);
        return true;
    }

    @Override
    public int evictBlocksByHfileName(String hfileName) {
        int numEvicted = 0;
        for (BlockCacheKey key : this.map.keySet()) {
            if (!key.getHfileName().equals(hfileName) || !this.evictBlock(key)) continue;
            ++numEvicted;
        }
        if (this.victimHandler != null) {
            numEvicted += this.victimHandler.evictBlocksByHfileName(hfileName);
        }
        return numEvicted;
    }

    protected long evictBlock(CachedBlock block, boolean evictedByEvictionProcess) {
        this.map.remove(block.getCacheKey());
        this.updateSizeMetrics(block, true);
        this.elements.decrementAndGet();
        this.stats.evicted();
        if (evictedByEvictionProcess && this.victimHandler != null) {
            boolean wait = this.getCurrentSize() < this.acceptableSize();
            boolean inMemory = block.getPriority() == CachedBlock.BlockPriority.MEMORY;
            this.victimHandler.cacheBlockWithWait(block.getCacheKey(), block.getBuffer(), inMemory, wait);
        }
        return block.heapSize();
    }

    private void runEviction() {
        if (this.evictionThread == null) {
            this.evict();
        } else {
            this.evictionThread.evict();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void evict() {
        if (!this.evictionLock.tryLock()) {
            return;
        }
        try {
            long bytesFreed;
            BlockBucket bucketMemory;
            BlockBucket bucketMulti;
            BlockBucket bucketSingle;
            block19: {
                BlockBucket bucket;
                long bytesToFree;
                block17: {
                    long bytesRemain;
                    long s;
                    block21: {
                        long m;
                        block20: {
                            block18: {
                                this.evictionInProgress = true;
                                long currentSize = this.size.get();
                                bytesToFree = currentSize - this.minSize();
                                if (LOG.isTraceEnabled()) {
                                    LOG.trace((Object)("Block cache LRU eviction started; Attempting to free " + StringUtils.byteDesc((long)bytesToFree) + " of total=" + StringUtils.byteDesc((long)currentSize)));
                                }
                                if (bytesToFree <= 0L) {
                                    return;
                                }
                                bucketSingle = new BlockBucket(bytesToFree, this.blockSize, this.singleSize());
                                bucketMulti = new BlockBucket(bytesToFree, this.blockSize, this.multiSize());
                                bucketMemory = new BlockBucket(bytesToFree, this.blockSize, this.memorySize());
                                for (CachedBlock cachedBlock : this.map.values()) {
                                    switch (cachedBlock.getPriority()) {
                                        case SINGLE: {
                                            bucketSingle.add(cachedBlock);
                                            break;
                                        }
                                        case MULTI: {
                                            bucketMulti.add(cachedBlock);
                                            break;
                                        }
                                        case MEMORY: {
                                            bucketMemory.add(cachedBlock);
                                        }
                                    }
                                }
                                bytesFreed = 0L;
                                if (!this.forceInMemory && !(this.memoryFactor > 0.999f)) break block17;
                                s = bucketSingle.totalSize();
                                if (bytesToFree <= s + (m = bucketMulti.totalSize())) break block18;
                                bytesFreed = bucketSingle.free(s);
                                bytesFreed += bucketMulti.free(m);
                                bytesFreed += bucketMemory.free(bytesToFree - bytesFreed);
                                break block19;
                            }
                            bytesRemain = s + m - bytesToFree;
                            if (3L * s > bytesRemain) break block20;
                            bytesFreed = bucketMulti.free(bytesToFree);
                            break block19;
                        }
                        if (3L * m > 2L * bytesRemain) break block21;
                        bytesFreed = bucketSingle.free(bytesToFree);
                        break block19;
                    }
                    bytesFreed = bucketSingle.free(s - bytesRemain / 3L);
                    if (bytesFreed >= bytesToFree) break block19;
                    bytesFreed += bucketMulti.free(bytesToFree - bytesFreed);
                    break block19;
                }
                PriorityQueue<BlockBucket> bucketQueue = new PriorityQueue<BlockBucket>(3);
                bucketQueue.add(bucketSingle);
                bucketQueue.add(bucketMulti);
                bucketQueue.add(bucketMemory);
                int remainingBuckets = 3;
                while ((bucket = (BlockBucket)bucketQueue.poll()) != null) {
                    long overflow = bucket.overflow();
                    if (overflow > 0L) {
                        long bucketBytesToFree = Math.min(overflow, (bytesToFree - bytesFreed) / (long)remainingBuckets);
                        bytesFreed += bucket.free(bucketBytesToFree);
                    }
                    --remainingBuckets;
                }
            }
            if (LOG.isTraceEnabled()) {
                long single = bucketSingle.totalSize();
                long multi = bucketMulti.totalSize();
                long memory = bucketMemory.totalSize();
                LOG.trace((Object)("Block cache LRU eviction completed; freed=" + StringUtils.byteDesc((long)bytesFreed) + ", " + "total=" + StringUtils.byteDesc((long)this.size.get()) + ", " + "single=" + StringUtils.byteDesc((long)single) + ", " + "multi=" + StringUtils.byteDesc((long)multi) + ", " + "memory=" + StringUtils.byteDesc((long)memory)));
            }
        }
        finally {
            this.stats.evict();
            this.evictionInProgress = false;
            this.evictionLock.unlock();
        }
    }

    public long getMaxSize() {
        return this.maxSize;
    }

    @Override
    public long getCurrentSize() {
        return this.size.get();
    }

    @Override
    public long getFreeSize() {
        return this.getMaxSize() - this.getCurrentSize();
    }

    @Override
    public long size() {
        return this.elements.get();
    }

    @Override
    public long getBlockCount() {
        return this.elements.get();
    }

    public long getEvictionCount() {
        return this.stats.getEvictionCount();
    }

    @Override
    public long getEvictedCount() {
        return this.stats.getEvictedCount();
    }

    EvictionThread getEvictionThread() {
        return this.evictionThread;
    }

    public void logStats() {
        if (!LOG.isDebugEnabled()) {
            return;
        }
        long totalSize = this.heapSize();
        long freeSize = this.maxSize - totalSize;
        LOG.debug((Object)("Total=" + StringUtils.byteDesc((long)totalSize) + ", " + "free=" + StringUtils.byteDesc((long)freeSize) + ", " + "max=" + StringUtils.byteDesc((long)this.maxSize) + ", " + "blocks=" + this.size() + ", " + "accesses=" + this.stats.getRequestCount() + ", " + "hits=" + this.stats.getHitCount() + ", " + "hitRatio=" + (this.stats.getHitCount() == 0L ? "0" : StringUtils.formatPercent((double)this.stats.getHitRatio(), (int)2) + ", ") + ", " + "cachingAccesses=" + this.stats.getRequestCachingCount() + ", " + "cachingHits=" + this.stats.getHitCachingCount() + ", " + "cachingHitsRatio=" + (this.stats.getHitCachingCount() == 0L ? "0," : StringUtils.formatPercent((double)this.stats.getHitCachingRatio(), (int)2) + ", ") + "evictions=" + this.stats.getEvictionCount() + ", " + "evicted=" + this.stats.getEvictedCount() + ", " + "evictedPerRun=" + this.stats.evictedPerEviction()));
    }

    @Override
    public CacheStats getStats() {
        return this.stats;
    }

    public long heapSize() {
        return this.getCurrentSize();
    }

    public static long calculateOverhead(long maxSize, long blockSize, int concurrency) {
        return CACHE_FIXED_OVERHEAD + (long)ClassSize.CONCURRENT_HASHMAP + (long)Math.ceil((double)maxSize * 1.2 / (double)blockSize) * (long)ClassSize.CONCURRENT_HASHMAP_ENTRY + (long)concurrency * (long)ClassSize.CONCURRENT_HASHMAP_SEGMENT;
    }

    @Override
    public List<BlockCacheColumnFamilySummary> getBlockCacheColumnFamilySummaries(Configuration conf) throws IOException {
        Map<String, Path> sfMap = FSUtils.getTableStoreFilePathMap(FileSystem.get((Configuration)conf), FSUtils.getRootDir(conf));
        HashMap<BlockCacheColumnFamilySummary, BlockCacheColumnFamilySummary> bcs = new HashMap<BlockCacheColumnFamilySummary, BlockCacheColumnFamilySummary>();
        for (CachedBlock cb : this.map.values()) {
            String sf = cb.getCacheKey().getHfileName();
            Path path = sfMap.get(sf);
            if (path == null) continue;
            BlockCacheColumnFamilySummary lookup = BlockCacheColumnFamilySummary.createFromStoreFilePath(path);
            BlockCacheColumnFamilySummary bcse = (BlockCacheColumnFamilySummary)bcs.get(lookup);
            if (bcse == null) {
                bcse = BlockCacheColumnFamilySummary.create(lookup);
                bcs.put(lookup, bcse);
            }
            bcse.incrementBlocks();
            bcse.incrementHeapSize(cb.heapSize());
        }
        ArrayList<BlockCacheColumnFamilySummary> list = new ArrayList<BlockCacheColumnFamilySummary>(bcs.values());
        Collections.sort(list);
        return list;
    }

    private long acceptableSize() {
        return (long)Math.floor((float)this.maxSize * this.acceptableFactor);
    }

    private long minSize() {
        return (long)Math.floor((float)this.maxSize * this.minFactor);
    }

    private long singleSize() {
        return (long)Math.floor((float)this.maxSize * this.singleFactor * this.minFactor);
    }

    private long multiSize() {
        return (long)Math.floor((float)this.maxSize * this.multiFactor * this.minFactor);
    }

    private long memorySize() {
        return (long)Math.floor((float)this.maxSize * this.memoryFactor * this.minFactor);
    }

    @Override
    public void shutdown() {
        if (this.victimHandler != null) {
            this.victimHandler.shutdown();
        }
        this.scheduleThreadPool.shutdown();
        for (int i = 0; i < 10; ++i) {
            if (this.scheduleThreadPool.isShutdown()) continue;
            Threads.sleep((long)10L);
        }
        if (!this.scheduleThreadPool.isShutdown()) {
            List<Runnable> runnables = this.scheduleThreadPool.shutdownNow();
            LOG.debug((Object)("Still running " + runnables));
        }
        this.evictionThread.shutdown();
    }

    public void clearCache() {
        this.map.clear();
    }

    SortedSet<String> getCachedFileNamesForTest() {
        TreeSet<String> fileNames = new TreeSet<String>();
        for (BlockCacheKey cacheKey : this.map.keySet()) {
            fileNames.add(cacheKey.getHfileName());
        }
        return fileNames;
    }

    Map<BlockType, Integer> getBlockTypeCountsForTest() {
        EnumMap<BlockType, Integer> counts = new EnumMap<BlockType, Integer>(BlockType.class);
        for (CachedBlock cb : this.map.values()) {
            BlockType blockType;
            Integer count = (Integer)counts.get(blockType = ((HFileBlock)cb.getBuffer()).getBlockType());
            counts.put(blockType, (count == null ? 0 : count) + 1);
        }
        return counts;
    }

    public Map<DataBlockEncoding, Integer> getEncodingCountsForTest() {
        EnumMap<DataBlockEncoding, Integer> counts = new EnumMap<DataBlockEncoding, Integer>(DataBlockEncoding.class);
        for (BlockCacheKey cacheKey : this.map.keySet()) {
            DataBlockEncoding encoding;
            Integer count = (Integer)counts.get(encoding = cacheKey.getDataBlockEncoding());
            counts.put(encoding, (count == null ? 0 : count) + 1);
        }
        return counts;
    }

    public void setVictimCache(BucketCache handler) {
        assert (this.victimHandler == null);
        this.victimHandler = handler;
    }

    static class StatisticsThread
    extends Thread {
        LruBlockCache lru;

        public StatisticsThread(LruBlockCache lru) {
            super("LruBlockCache.StatisticsThread");
            this.setDaemon(true);
            this.lru = lru;
        }

        @Override
        public void run() {
            this.lru.logStats();
        }
    }

    static class EvictionThread
    extends HasThread {
        private WeakReference<LruBlockCache> cache;
        private boolean go = true;
        private boolean enteringRun = false;

        public EvictionThread(LruBlockCache cache) {
            super(Thread.currentThread().getName() + ".LruBlockCache.EvictionThread");
            this.setDaemon(true);
            this.cache = new WeakReference<LruBlockCache>(cache);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void run() {
            this.enteringRun = true;
            while (this.go) {
                EvictionThread evictionThread = this;
                synchronized (evictionThread) {
                    try {
                        ((Object)((Object)this)).wait();
                    }
                    catch (InterruptedException interruptedException) {
                        // empty catch block
                    }
                }
                LruBlockCache cache = (LruBlockCache)this.cache.get();
                if (cache == null) break;
                cache.evict();
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void evict() {
            EvictionThread evictionThread = this;
            synchronized (evictionThread) {
                ((Object)((Object)this)).notifyAll();
            }
        }

        synchronized void shutdown() {
            this.go = false;
            ((Object)((Object)this)).notifyAll();
        }

        boolean isEnteringRun() {
            return this.enteringRun;
        }
    }

    private class BlockBucket
    implements Comparable<BlockBucket> {
        private CachedBlockQueue queue;
        private long totalSize = 0L;
        private long bucketSize;

        public BlockBucket(long bytesToFree, long blockSize, long bucketSize) {
            this.bucketSize = bucketSize;
            this.queue = new CachedBlockQueue(bytesToFree, blockSize);
            this.totalSize = 0L;
        }

        public void add(CachedBlock block) {
            this.totalSize += block.heapSize();
            this.queue.add(block);
        }

        public long free(long toFree) {
            CachedBlock cb;
            long freedBytes = 0L;
            while ((cb = this.queue.pollLast()) != null) {
                if ((freedBytes += LruBlockCache.this.evictBlock(cb, true)) < toFree) continue;
                return freedBytes;
            }
            return freedBytes;
        }

        public long overflow() {
            return this.totalSize - this.bucketSize;
        }

        public long totalSize() {
            return this.totalSize;
        }

        @Override
        public int compareTo(BlockBucket that) {
            if (this.overflow() == that.overflow()) {
                return 0;
            }
            return this.overflow() > that.overflow() ? 1 : -1;
        }

        public boolean equals(Object that) {
            if (that == null || !(that instanceof BlockBucket)) {
                return false;
            }
            return this.compareTo((BlockBucket)that) == 0;
        }
    }
}

