/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.db.writelog.node;

import java.io.File;
import java.io.IOException;
import java.nio.BufferOverflowException;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Comparator;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import org.apache.commons.io.FileUtils;
import org.apache.iotdb.db.conf.IoTDBConfig;
import org.apache.iotdb.db.conf.IoTDBDescriptor;
import org.apache.iotdb.db.conf.directories.DirectoryManager;
import org.apache.iotdb.db.engine.fileSystem.SystemFileFactory;
import org.apache.iotdb.db.qp.physical.PhysicalPlan;
import org.apache.iotdb.db.writelog.io.ILogReader;
import org.apache.iotdb.db.writelog.io.ILogWriter;
import org.apache.iotdb.db.writelog.io.LogWriter;
import org.apache.iotdb.db.writelog.io.MultiFileLogReader;
import org.apache.iotdb.db.writelog.node.WriteLogNode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ExclusiveWriteLogNode
implements WriteLogNode,
Comparable<ExclusiveWriteLogNode> {
    public static final String WAL_FILE_NAME = "wal";
    private static final Logger logger = LoggerFactory.getLogger(ExclusiveWriteLogNode.class);
    private String identifier;
    private String logDirectory;
    private ILogWriter currentFileWriter;
    private IoTDBConfig config = IoTDBDescriptor.getInstance().getConfig();
    private ByteBuffer logBuffer = ByteBuffer.allocate(IoTDBDescriptor.getInstance().getConfig().getWalBufferSize());
    private ReadWriteLock lock = new ReentrantReadWriteLock();
    private long fileId = 0L;
    private long lastFlushedId = 0L;
    private int bufferedLogNum = 0;

    public ExclusiveWriteLogNode(String identifier) {
        this.identifier = identifier;
        this.logDirectory = DirectoryManager.getInstance().getWALFolder() + File.separator + this.identifier;
        if (SystemFileFactory.INSTANCE.getFile(this.logDirectory).mkdirs()) {
            logger.info("create the WAL folder {}." + this.logDirectory);
        }
    }

    @Override
    public void write(PhysicalPlan plan) throws IOException {
        this.lock.writeLock().lock();
        try {
            this.putLog(plan);
            if (this.bufferedLogNum >= this.config.getFlushWalThreshold()) {
                this.sync();
            }
        }
        catch (BufferOverflowException e) {
            throw new IOException("Log cannot fit into the buffer, please increase wal_buffer_size", e);
        }
        finally {
            this.lock.writeLock().unlock();
        }
    }

    private void putLog(PhysicalPlan plan) {
        this.logBuffer.mark();
        try {
            plan.serialize(this.logBuffer);
        }
        catch (BufferOverflowException e) {
            logger.info("WAL BufferOverflow !");
            this.logBuffer.reset();
            this.sync();
            plan.serialize(this.logBuffer);
        }
        ++this.bufferedLogNum;
    }

    @Override
    public void close() {
        this.sync();
        this.forceWal();
        this.lock.writeLock().lock();
        try {
            if (this.currentFileWriter != null) {
                this.currentFileWriter.close();
                this.currentFileWriter = null;
            }
            logger.debug("Log node {} closed successfully", (Object)this.identifier);
        }
        catch (IOException e) {
            logger.error("Cannot close log node {} because:", (Object)this.identifier, (Object)e);
        }
        finally {
            this.lock.writeLock().unlock();
        }
    }

    @Override
    public void forceSync() {
        this.sync();
        this.forceWal();
    }

    @Override
    public void notifyStartFlush() {
        this.lock.writeLock().lock();
        try {
            this.close();
            this.nextFileWriter();
        }
        finally {
            this.lock.writeLock().unlock();
        }
    }

    @Override
    public void notifyEndFlush() {
        this.lock.writeLock().lock();
        try {
            File logFile = SystemFileFactory.INSTANCE.getFile(this.logDirectory, WAL_FILE_NAME + ++this.lastFlushedId);
            this.discard(logFile);
        }
        finally {
            this.lock.writeLock().unlock();
        }
    }

    @Override
    public String getIdentifier() {
        return this.identifier;
    }

    @Override
    public String getLogDirectory() {
        return this.logDirectory;
    }

    @Override
    public void delete() throws IOException {
        this.lock.writeLock().lock();
        try {
            this.logBuffer.clear();
            this.close();
            FileUtils.deleteDirectory((File)SystemFileFactory.INSTANCE.getFile(this.logDirectory));
        }
        finally {
            this.lock.writeLock().unlock();
        }
    }

    @Override
    public ILogReader getLogReader() {
        File[] logFiles = SystemFileFactory.INSTANCE.getFile(this.logDirectory).listFiles();
        Arrays.sort(logFiles, Comparator.comparingInt(f -> Integer.parseInt(f.getName().replace(WAL_FILE_NAME, ""))));
        return new MultiFileLogReader(logFiles);
    }

    private void discard(File logFile) {
        if (!logFile.exists()) {
            logger.info("Log file does not exist");
        } else {
            try {
                FileUtils.forceDelete((File)logFile);
                logger.info("Log node {} cleaned old file", (Object)this.identifier);
            }
            catch (IOException e) {
                logger.error("Old log file {} of {} cannot be deleted", new Object[]{logFile.getName(), this.identifier, e});
            }
        }
    }

    private void forceWal() {
        this.lock.writeLock().lock();
        try {
            try {
                if (this.currentFileWriter != null) {
                    this.currentFileWriter.force();
                }
            }
            catch (IOException e) {
                logger.error("Log node {} force failed.", (Object)this.identifier, (Object)e);
            }
        }
        finally {
            this.lock.writeLock().unlock();
        }
    }

    private void sync() {
        this.lock.writeLock().lock();
        try {
            if (this.bufferedLogNum == 0) {
                return;
            }
            try {
                this.getCurrentFileWriter().write(this.logBuffer);
            }
            catch (IOException e) {
                logger.error("Log node {} sync failed, change system mode to read-only", (Object)this.identifier, (Object)e);
                IoTDBDescriptor.getInstance().getConfig().setReadOnly(true);
                this.lock.writeLock().unlock();
                return;
            }
            this.logBuffer.clear();
            this.bufferedLogNum = 0;
            logger.debug("Log node {} ends sync.", (Object)this.identifier);
        }
        finally {
            this.lock.writeLock().unlock();
        }
    }

    private ILogWriter getCurrentFileWriter() {
        if (this.currentFileWriter == null) {
            this.nextFileWriter();
        }
        return this.currentFileWriter;
    }

    private void nextFileWriter() {
        ++this.fileId;
        File newFile = SystemFileFactory.INSTANCE.getFile(this.logDirectory, WAL_FILE_NAME + this.fileId);
        if (newFile.getParentFile().mkdirs()) {
            logger.info("create WAL parent folder {}.", (Object)newFile.getParent());
        }
        this.currentFileWriter = new LogWriter(newFile);
    }

    public int hashCode() {
        return this.identifier.hashCode();
    }

    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (this.getClass() != obj.getClass()) {
            return false;
        }
        return this.compareTo((ExclusiveWriteLogNode)obj) == 0;
    }

    public String toString() {
        return "Log node " + this.identifier;
    }

    @Override
    public int compareTo(ExclusiveWriteLogNode o) {
        return this.identifier.compareTo(o.identifier);
    }
}

