/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.db.query.control;

import java.time.ZoneId;
import java.util.Map;
import java.util.Set;
import java.util.TimeZone;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Consumer;
import org.apache.iotdb.common.rpc.thrift.TSStatus;
import org.apache.iotdb.commons.auth.AuthException;
import org.apache.iotdb.commons.conf.IoTDBConstant;
import org.apache.iotdb.db.auth.AuthorityChecker;
import org.apache.iotdb.db.auth.AuthorizerManager;
import org.apache.iotdb.db.conf.OperationType;
import org.apache.iotdb.db.exception.StorageEngineException;
import org.apache.iotdb.db.mpp.common.SessionInfo;
import org.apache.iotdb.db.qp.physical.PhysicalPlan;
import org.apache.iotdb.db.qp.physical.sys.AuthorPlan;
import org.apache.iotdb.db.query.control.QueryResourceManager;
import org.apache.iotdb.db.query.control.SessionTimeoutManager;
import org.apache.iotdb.db.query.dataset.UDTFDataSet;
import org.apache.iotdb.db.service.basic.BasicOpenSessionResp;
import org.apache.iotdb.db.utils.ErrorHandlingUtils;
import org.apache.iotdb.rpc.RpcUtils;
import org.apache.iotdb.rpc.TSStatusCode;
import org.apache.iotdb.service.rpc.thrift.TSProtocolVersion;
import org.apache.iotdb.tsfile.read.query.dataset.QueryDataSet;
import org.apache.thrift.TException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SessionManager {
    private static final Logger LOGGER = LoggerFactory.getLogger(SessionManager.class);
    public static final Logger AUDIT_LOGGER = LoggerFactory.getLogger((String)"IoTDB_AUDIT_LOGGER");
    private final ThreadLocal<Long> currSessionId = new ThreadLocal();
    private final Map<Long, String> sessionIdToUsername = new ConcurrentHashMap<Long, String>();
    private final Map<Long, ZoneId> sessionIdToZoneId = new ConcurrentHashMap<Long, ZoneId>();
    private final AtomicLong sessionIdGenerator = new AtomicLong();
    private final AtomicLong statementIdGenerator = new AtomicLong();
    private final Map<Long, Set<Long>> sessionIdToStatementId = new ConcurrentHashMap<Long, Set<Long>>();
    private final Map<Long, Set<Long>> statementIdToQueryId = new ConcurrentHashMap<Long, Set<Long>>();
    private final Map<Long, QueryDataSet> queryIdToDataSet = new ConcurrentHashMap<Long, QueryDataSet>();
    private final Map<Long, IoTDBConstant.ClientVersion> sessionIdToClientVersion = new ConcurrentHashMap<Long, IoTDBConstant.ClientVersion>();
    private final Map<Long, SessionInfo> sessionIdToSessionInfo = new ConcurrentHashMap<Long, SessionInfo>();
    public static final TSProtocolVersion CURRENT_RPC_VERSION = TSProtocolVersion.IOTDB_SERVICE_PROTOCOL_V3;

    protected SessionManager() {
    }

    public BasicOpenSessionResp openSession(String username, String password, String zoneId, TSProtocolVersion tsProtocolVersion, IoTDBConstant.ClientVersion clientVersion) throws TException {
        boolean loginStatus = false;
        String loginMessage = null;
        try {
            loginStatus = AuthorizerManager.getInstance().login(username, password);
        }
        catch (AuthException e) {
            loginMessage = e.getMessage();
            LOGGER.info("meet error while logging in.", (Throwable)e);
        }
        BasicOpenSessionResp openSessionResp = new BasicOpenSessionResp();
        if (loginStatus) {
            if (!tsProtocolVersion.equals((Object)CURRENT_RPC_VERSION)) {
                openSessionResp.sessionId(-1L).setCode(TSStatusCode.INCOMPATIBLE_VERSION.getStatusCode()).setMessage("The version is incompatible, please upgrade to " + IoTDBConstant.VERSION);
            } else {
                long sessionId = this.requestSessionId(username, zoneId, clientVersion);
                SessionTimeoutManager.getInstance().register(sessionId);
                openSessionResp.sessionId(sessionId).setCode(TSStatusCode.SUCCESS_STATUS.getStatusCode()).setMessage("Login successfully");
                LOGGER.info("{}: Login status: {}. User : {}, opens Session-{}", new Object[]{"IoTDB", openSessionResp.getMessage(), username, sessionId});
            }
        } else {
            AUDIT_LOGGER.info("User {} opens Session failed with an incorrect password", (Object)username);
            openSessionResp.sessionId(-1L).setMessage(loginMessage != null ? loginMessage : "Authentication failed.").setCode(TSStatusCode.WRONG_LOGIN_PASSWORD_ERROR.getStatusCode());
        }
        return openSessionResp;
    }

    public BasicOpenSessionResp openSession(String username, String password, String zoneId, TSProtocolVersion tsProtocolVersion) throws TException {
        return this.openSession(username, password, zoneId, tsProtocolVersion, IoTDBConstant.ClientVersion.V_0_12);
    }

    public boolean closeSession(long sessionId) {
        AUDIT_LOGGER.info("Session-{} is closing", (Object)sessionId);
        this.currSessionId.remove();
        return SessionTimeoutManager.getInstance().unregister(sessionId);
    }

    public TSStatus closeOperation(long sessionId, long queryId, long statementId, boolean haveStatementId, boolean haveSetQueryId, Consumer<Long> releaseByQueryId) {
        if (!this.checkLogin(sessionId)) {
            return RpcUtils.getStatus((TSStatusCode)TSStatusCode.NOT_LOGIN_ERROR, (String)"Log in failed. Either you are not authorized or the session has timed out.");
        }
        if (AUDIT_LOGGER.isDebugEnabled()) {
            AUDIT_LOGGER.debug("{}: receive close operation from Session {}", (Object)"IoTDB", this.currSessionId);
        }
        try {
            if (haveStatementId) {
                if (haveSetQueryId) {
                    this.closeDataset(statementId, queryId, releaseByQueryId);
                } else {
                    this.closeStatement(sessionId, statementId, releaseByQueryId);
                }
                return RpcUtils.getStatus((TSStatusCode)TSStatusCode.SUCCESS_STATUS);
            }
            return RpcUtils.getStatus((TSStatusCode)TSStatusCode.CLOSE_OPERATION_ERROR, (String)"statement id not set by client.");
        }
        catch (Exception e) {
            return ErrorHandlingUtils.onNPEOrUnexpectedException(e, OperationType.CLOSE_OPERATION, TSStatusCode.CLOSE_OPERATION_ERROR);
        }
    }

    public TSStatus closeOperation(long sessionId, long queryId, long statementId, boolean haveStatementId, boolean haveSetQueryId) {
        return this.closeOperation(sessionId, queryId, statementId, haveStatementId, haveSetQueryId, this::releaseQueryResourceNoExceptions);
    }

    public boolean checkLogin(long sessionId) {
        boolean isLoggedIn;
        boolean bl = isLoggedIn = this.sessionIdToUsername.get(sessionId) != null;
        if (!isLoggedIn) {
            LOGGER.info("{}: Not login. ", (Object)"IoTDB");
        } else {
            SessionTimeoutManager.getInstance().refresh(sessionId);
        }
        return isLoggedIn;
    }

    public long requestSessionId(String username, String zoneId, IoTDBConstant.ClientVersion clientVersion) {
        long sessionId = this.sessionIdGenerator.incrementAndGet();
        this.currSessionId.set(sessionId);
        this.sessionIdToUsername.put(sessionId, username);
        this.sessionIdToZoneId.put(sessionId, ZoneId.of(zoneId));
        this.sessionIdToClientVersion.put(sessionId, clientVersion);
        return sessionId;
    }

    public boolean releaseSessionResource(long sessionId) {
        return this.releaseSessionResource(sessionId, this::releaseQueryResourceNoExceptions);
    }

    public boolean releaseSessionResource(long sessionId, Consumer<Long> releaseQueryResource) {
        this.sessionIdToZoneId.remove(sessionId);
        this.sessionIdToClientVersion.remove(sessionId);
        Set<Long> statementIdSet = this.sessionIdToStatementId.remove(sessionId);
        if (statementIdSet != null) {
            for (Long statementId : statementIdSet) {
                Set<Long> queryIdSet = this.statementIdToQueryId.remove(statementId);
                if (queryIdSet == null) continue;
                for (Long queryId : queryIdSet) {
                    releaseQueryResource.accept(queryId);
                }
            }
        }
        return this.sessionIdToUsername.remove(sessionId) != null;
    }

    public long getSessionIdByQueryId(long queryId) {
        for (Map.Entry<Long, Set<Long>> statementToQueries : this.statementIdToQueryId.entrySet()) {
            if (!statementToQueries.getValue().contains(queryId)) continue;
            for (Map.Entry<Long, Set<Long>> sessionToStatements : this.sessionIdToStatementId.entrySet()) {
                if (!sessionToStatements.getValue().contains(statementToQueries.getKey())) continue;
                return sessionToStatements.getKey();
            }
        }
        return -1L;
    }

    public long requestStatementId(long sessionId) {
        long statementId = this.statementIdGenerator.incrementAndGet();
        this.sessionIdToStatementId.computeIfAbsent(sessionId, s -> new CopyOnWriteArraySet()).add(statementId);
        return statementId;
    }

    public void closeStatement(long sessionId, long statementId, Consumer<Long> releaseByQueryId) {
        Set<Long> queryIdSet = this.statementIdToQueryId.remove(statementId);
        if (queryIdSet != null) {
            for (Long queryId : queryIdSet) {
                releaseByQueryId.accept(queryId);
            }
        }
        if (this.sessionIdToStatementId.containsKey(sessionId)) {
            this.sessionIdToStatementId.get(sessionId).remove(statementId);
        }
    }

    public long requestQueryId(Long statementId, boolean isDataQuery) {
        long queryId = this.requestQueryId(isDataQuery);
        this.statementIdToQueryId.computeIfAbsent(statementId, k -> new CopyOnWriteArraySet()).add(queryId);
        return queryId;
    }

    public long requestQueryId(boolean isDataQuery) {
        return QueryResourceManager.getInstance().assignQueryId(isDataQuery);
    }

    public void releaseQueryResource(long queryId) throws StorageEngineException {
        QueryDataSet dataSet = this.queryIdToDataSet.remove(queryId);
        if (dataSet instanceof UDTFDataSet) {
            ((UDTFDataSet)dataSet).finalizeUDFs(queryId);
        }
        QueryResourceManager.getInstance().endQuery(queryId);
    }

    public void releaseQueryResourceNoExceptions(long queryId) {
        if (queryId != -1L) {
            try {
                this.releaseQueryResource(queryId);
            }
            catch (Exception e) {
                LOGGER.warn("Error occurred while releasing query resource: ", (Throwable)e);
            }
        }
    }

    public boolean checkAuthorization(PhysicalPlan plan, String username) throws AuthException {
        if (!plan.isAuthenticationRequired()) {
            return true;
        }
        String targetUser = null;
        if (plan instanceof AuthorPlan) {
            targetUser = ((AuthorPlan)plan).getUserName();
        }
        return AuthorityChecker.check(username, plan.getAuthPaths(), plan.getOperatorType(), targetUser);
    }

    public TSStatus checkAuthority(PhysicalPlan plan, long sessionId) {
        try {
            if (!this.checkAuthorization(plan, this.sessionIdToUsername.get(sessionId))) {
                return RpcUtils.getStatus((TSStatusCode)TSStatusCode.NO_PERMISSION_ERROR, (String)("No permissions for this operation " + (Object)((Object)plan.getOperatorType())));
            }
        }
        catch (AuthException e) {
            LOGGER.warn("meet error while checking authorization.", (Throwable)e);
            return RpcUtils.getStatus((TSStatusCode)TSStatusCode.UNINITIALIZED_AUTH_ERROR, (String)e.getMessage());
        }
        catch (Exception e) {
            return ErrorHandlingUtils.onNPEOrUnexpectedException(e, OperationType.CHECK_AUTHORITY, TSStatusCode.EXECUTE_STATEMENT_ERROR);
        }
        return null;
    }

    public Long getCurrSessionId() {
        return this.currSessionId.get();
    }

    public TimeZone getCurrSessionTimeZone() {
        if (this.getCurrSessionId() != null) {
            return TimeZone.getTimeZone(SessionManager.getInstance().getZoneId(this.getCurrSessionId()));
        }
        return TimeZone.getTimeZone("+08:00");
    }

    public String getUsername(Long sessionId) {
        return this.sessionIdToUsername.get(sessionId);
    }

    public ZoneId getZoneId(Long sessionId) {
        return this.sessionIdToZoneId.get(sessionId);
    }

    public void setTimezone(Long sessionId, String zone) {
        this.sessionIdToZoneId.put(sessionId, ZoneId.of(zone));
    }

    public boolean hasDataset(Long queryId) {
        return this.queryIdToDataSet.containsKey(queryId);
    }

    public QueryDataSet getDataset(Long queryId) {
        return this.queryIdToDataSet.get(queryId);
    }

    public void setDataset(Long queryId, QueryDataSet dataSet) {
        this.queryIdToDataSet.put(queryId, dataSet);
    }

    public void removeDataset(Long queryId) {
        this.queryIdToDataSet.remove(queryId);
    }

    public void closeDataset(Long statementId, Long queryId, Consumer<Long> releaseByQueryId) {
        releaseByQueryId.accept(queryId);
        if (this.statementIdToQueryId.containsKey(statementId)) {
            this.statementIdToQueryId.get(statementId).remove(queryId);
        }
    }

    public IoTDBConstant.ClientVersion getClientVersion(Long sessionId) {
        return this.sessionIdToClientVersion.get(sessionId);
    }

    public static SessionManager getInstance() {
        return SessionManagerHelper.INSTANCE;
    }

    public SessionInfo getSessionInfo(long sessionId) {
        return this.sessionIdToSessionInfo.get(sessionId);
    }

    private static class SessionManagerHelper {
        private static final SessionManager INSTANCE = new SessionManager();

        private SessionManagerHelper() {
        }
    }
}

