/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.tsfile.write;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import org.apache.iotdb.tsfile.common.conf.TSFileConfig;
import org.apache.iotdb.tsfile.common.conf.TSFileDescriptor;
import org.apache.iotdb.tsfile.exception.write.NoMeasurementException;
import org.apache.iotdb.tsfile.exception.write.WriteProcessException;
import org.apache.iotdb.tsfile.write.chunk.ChunkGroupWriterImpl;
import org.apache.iotdb.tsfile.write.chunk.IChunkGroupWriter;
import org.apache.iotdb.tsfile.write.record.RowBatch;
import org.apache.iotdb.tsfile.write.record.TSRecord;
import org.apache.iotdb.tsfile.write.record.datapoint.DataPoint;
import org.apache.iotdb.tsfile.write.schema.MeasurementSchema;
import org.apache.iotdb.tsfile.write.schema.Schema;
import org.apache.iotdb.tsfile.write.writer.TsFileIOWriter;
import org.apache.iotdb.tsfile.write.writer.TsFileOutput;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TsFileWriter
implements AutoCloseable {
    private static final Logger LOG = LoggerFactory.getLogger(TsFileWriter.class);
    protected static final TSFileConfig config = TSFileDescriptor.getInstance().getConfig();
    protected final Schema schema;
    private final TsFileIOWriter fileWriter;
    private final int pageSize;
    private long recordCount = 0L;
    private Map<String, IChunkGroupWriter> groupWriters = new HashMap<String, IChunkGroupWriter>();
    private long recordCountForNextMemCheck = 100L;
    private long chunkGroupSizeThreshold;

    public TsFileWriter(File file) throws IOException {
        this(new TsFileIOWriter(file), new Schema(), TSFileDescriptor.getInstance().getConfig());
    }

    public TsFileWriter(TsFileIOWriter fileWriter) throws IOException {
        this(fileWriter, new Schema(), TSFileDescriptor.getInstance().getConfig());
    }

    public TsFileWriter(File file, Schema schema) throws IOException {
        this(new TsFileIOWriter(file), schema, TSFileDescriptor.getInstance().getConfig());
    }

    public TsFileWriter(TsFileOutput output, Schema schema) throws IOException {
        this(new TsFileIOWriter(output), schema, TSFileDescriptor.getInstance().getConfig());
    }

    public TsFileWriter(File file, Schema schema, TSFileConfig conf) throws IOException {
        this(new TsFileIOWriter(file), schema, conf);
    }

    protected TsFileWriter(TsFileIOWriter fileWriter, Schema schema, TSFileConfig conf) throws IOException {
        if (!fileWriter.canWrite()) {
            throw new IOException("the given file Writer does not support writing any more. Maybe it is an complete TsFile");
        }
        this.fileWriter = fileWriter;
        this.schema = schema;
        this.schema.registerMeasurements(fileWriter.getKnownSchema());
        this.pageSize = conf.getPageSizeInByte();
        this.chunkGroupSizeThreshold = conf.getGroupSizeInByte();
        config.setTSFileStorageFs(conf.getTSFileStorageFs().name());
        if ((long)this.pageSize >= this.chunkGroupSizeThreshold) {
            LOG.warn("TsFile's page size {} is greater than chunk group size {}, please enlarge the chunk group size or decrease page size. ", (Object)this.pageSize, (Object)this.chunkGroupSizeThreshold);
        }
    }

    public void addMeasurement(MeasurementSchema measurementSchema) throws WriteProcessException {
        if (this.schema.hasMeasurement(measurementSchema.getMeasurementId())) {
            throw new WriteProcessException("given measurement has exists! " + measurementSchema.getMeasurementId());
        }
        this.schema.registerMeasurement(measurementSchema);
    }

    private boolean checkIsTimeSeriesExist(TSRecord record) throws WriteProcessException {
        IChunkGroupWriter groupWriter;
        if (!this.groupWriters.containsKey(record.deviceId)) {
            groupWriter = new ChunkGroupWriterImpl(record.deviceId);
            this.groupWriters.put(record.deviceId, groupWriter);
        } else {
            groupWriter = this.groupWriters.get(record.deviceId);
        }
        Map<String, MeasurementSchema> schemaDescriptorMap = this.schema.getMeasurementSchemaMap();
        for (DataPoint dp : record.dataPointList) {
            String measurementId = dp.getMeasurementId();
            if (schemaDescriptorMap.containsKey(measurementId)) {
                groupWriter.tryToAddSeriesWriter(schemaDescriptorMap.get(measurementId), this.pageSize);
                continue;
            }
            throw new NoMeasurementException("input measurement is invalid: " + measurementId);
        }
        return true;
    }

    private void checkIsTimeSeriesExist(RowBatch rowBatch) throws WriteProcessException {
        IChunkGroupWriter groupWriter;
        if (!this.groupWriters.containsKey(rowBatch.deviceId)) {
            groupWriter = new ChunkGroupWriterImpl(rowBatch.deviceId);
            this.groupWriters.put(rowBatch.deviceId, groupWriter);
        } else {
            groupWriter = this.groupWriters.get(rowBatch.deviceId);
        }
        Map<String, MeasurementSchema> schemaDescriptorMap = this.schema.getMeasurementSchemaMap();
        for (MeasurementSchema measurement : rowBatch.measurements) {
            String measurementId = measurement.getMeasurementId();
            if (schemaDescriptorMap.containsKey(measurementId)) {
                groupWriter.tryToAddSeriesWriter(schemaDescriptorMap.get(measurementId), this.pageSize);
                continue;
            }
            throw new NoMeasurementException("input measurement is invalid: " + measurementId);
        }
    }

    public boolean write(TSRecord record) throws IOException, WriteProcessException {
        this.checkIsTimeSeriesExist(record);
        this.groupWriters.get(record.deviceId).write(record.time, record.dataPointList);
        ++this.recordCount;
        return this.checkMemorySizeAndMayFlushGroup();
    }

    public boolean write(RowBatch rowBatch) throws IOException, WriteProcessException {
        this.checkIsTimeSeriesExist(rowBatch);
        this.groupWriters.get(rowBatch.deviceId).write(rowBatch);
        this.recordCount += (long)rowBatch.batchSize;
        return this.checkMemorySizeAndMayFlushGroup();
    }

    private long calculateMemSizeForAllGroup() {
        int memTotalSize = 0;
        for (IChunkGroupWriter group : this.groupWriters.values()) {
            memTotalSize = (int)((long)memTotalSize + group.updateMaxGroupMemSize());
        }
        return memTotalSize;
    }

    private boolean checkMemorySizeAndMayFlushGroup() throws IOException {
        if (this.recordCount >= this.recordCountForNextMemCheck) {
            long memSize = this.calculateMemSizeForAllGroup();
            assert (memSize > 0L);
            if (memSize > this.chunkGroupSizeThreshold) {
                LOG.debug("start to flush chunk groups, memory space occupy:{}", (Object)memSize);
                this.recordCountForNextMemCheck = this.recordCount * this.chunkGroupSizeThreshold / memSize;
                return this.flushAllChunkGroups();
            }
            this.recordCountForNextMemCheck = this.recordCount * this.chunkGroupSizeThreshold / memSize;
            return false;
        }
        return false;
    }

    private boolean flushAllChunkGroups() throws IOException {
        if (this.recordCount > 0L) {
            for (Map.Entry<String, IChunkGroupWriter> entry : this.groupWriters.entrySet()) {
                long pos = this.fileWriter.getPos();
                String deviceId = entry.getKey();
                IChunkGroupWriter groupWriter = entry.getValue();
                this.fileWriter.startChunkGroup(deviceId);
                long dataSize = groupWriter.flushToFileWriter(this.fileWriter);
                if (this.fileWriter.getPos() - pos != dataSize) {
                    throw new IOException(String.format("Flushed data size is inconsistent with computation! Estimated: %d, Actual: %d", dataSize, this.fileWriter.getPos() - pos));
                }
                this.fileWriter.endChunkGroup(0L);
            }
            this.reset();
        }
        return false;
    }

    private void reset() {
        this.groupWriters.clear();
        this.recordCount = 0L;
    }

    @Override
    public void close() throws IOException {
        LOG.info("start close file");
        this.flushAllChunkGroups();
        this.fileWriter.endFile(this.schema);
    }

    public TsFileIOWriter getIOWriter() {
        return this.fileWriter;
    }

    public void flushForTest() throws IOException {
        this.flushAllChunkGroups();
    }
}

