/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pinot.controller.validation;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import java.io.File;
import org.apache.commons.io.FileUtils;
import org.apache.pinot.common.config.QuotaConfig;
import org.apache.pinot.common.config.TableConfig;
import org.apache.pinot.common.exception.InvalidConfigException;
import org.apache.pinot.common.metrics.AbstractMetrics;
import org.apache.pinot.common.metrics.ControllerGauge;
import org.apache.pinot.common.metrics.ControllerMetrics;
import org.apache.pinot.controller.util.TableSizeReader;
import org.apache.pinot.spi.utils.DataSize;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class StorageQuotaChecker {
    private static final Logger LOGGER = LoggerFactory.getLogger(StorageQuotaChecker.class);
    private final TableSizeReader _tableSizeReader;
    private final TableConfig _tableConfig;
    private final ControllerMetrics _controllerMetrics;
    private final boolean _isLeaderForTable;

    public StorageQuotaChecker(TableConfig tableConfig, TableSizeReader tableSizeReader, ControllerMetrics controllerMetrics, boolean isLeaderForTable) {
        this._tableConfig = tableConfig;
        this._tableSizeReader = tableSizeReader;
        this._controllerMetrics = controllerMetrics;
        this._isLeaderForTable = isLeaderForTable;
    }

    public static QuotaCheckerResponse success(String msg) {
        return new QuotaCheckerResponse(true, msg);
    }

    public static QuotaCheckerResponse failure(String msg) {
        return new QuotaCheckerResponse(false, msg);
    }

    public QuotaCheckerResponse isSegmentStorageWithinQuota(File segmentFile, String segmentName, int timeoutMs) throws InvalidConfigException {
        TableSizeReader.TableSubTypeSizeDetails tableSubtypeSize;
        Preconditions.checkArgument((timeoutMs > 0 ? 1 : 0) != 0, (String)"Timeout value must be > 0, input: %s", (int)timeoutMs);
        Preconditions.checkArgument((boolean)segmentFile.exists(), (String)"Segment file: %s does not exist", (Object)segmentFile);
        Preconditions.checkArgument((boolean)segmentFile.isDirectory(), (String)"Segment file: %s is not a directory", (Object)segmentFile);
        QuotaConfig quotaConfig = this._tableConfig.getQuotaConfig();
        int numReplicas = this._tableConfig.getValidationConfig().getReplicationNumber();
        String tableNameWithType = this._tableConfig.getTableName();
        if (quotaConfig == null || Strings.isNullOrEmpty((String)quotaConfig.getStorage())) {
            String message = String.format("Storage quota is not configured for table: %s, skipping the check", tableNameWithType);
            LOGGER.info(message);
            return StorageQuotaChecker.success(message);
        }
        long allowedStorageBytes = (long)numReplicas * quotaConfig.storageSizeBytes();
        if (allowedStorageBytes <= 0L) {
            String message = String.format("Invalid storage quota: %s for table: %s, skipping the check", quotaConfig.getStorage(), tableNameWithType);
            LOGGER.warn(message);
            return StorageQuotaChecker.success(message);
        }
        this._controllerMetrics.setValueOfTableGauge(tableNameWithType, (AbstractMetrics.Gauge)ControllerGauge.TABLE_QUOTA, allowedStorageBytes);
        long incomingSegmentSizeBytes = FileUtils.sizeOfDirectory((File)segmentFile);
        try {
            tableSubtypeSize = this._tableSizeReader.getTableSubtypeSize(tableNameWithType, timeoutMs);
        }
        catch (InvalidConfigException e) {
            LOGGER.error("Failed to get table size for table {}", (Object)tableNameWithType, (Object)e);
            throw e;
        }
        if (tableSubtypeSize.estimatedSizeInBytes == -1L) {
            return StorageQuotaChecker.success("Missing size reports from all servers. Bypassing storage quota check for " + tableNameWithType);
        }
        if (tableSubtypeSize.missingSegments > 0) {
            if (tableSubtypeSize.estimatedSizeInBytes > allowedStorageBytes) {
                return StorageQuotaChecker.failure("Table " + tableNameWithType + " already over quota. Estimated size for all replicas is " + DataSize.fromBytes((long)tableSubtypeSize.estimatedSizeInBytes) + ". Configured size for " + numReplicas + " is " + DataSize.fromBytes((long)allowedStorageBytes));
            }
            return StorageQuotaChecker.success("Missing size report for " + tableSubtypeSize.missingSegments + " segments. Bypassing storage quota check for " + tableNameWithType);
        }
        TableSizeReader.SegmentSizeDetails sizeDetails = tableSubtypeSize.segments.get(segmentName);
        long existingSegmentSizeBytes = sizeDetails != null ? sizeDetails.estimatedSizeInBytes : 0L;
        this._controllerMetrics.setValueOfTableGauge(tableNameWithType, (AbstractMetrics.Gauge)ControllerGauge.OFFLINE_TABLE_ESTIMATED_SIZE, tableSubtypeSize.estimatedSizeInBytes);
        LOGGER.info("Table {}'s estimatedSizeInBytes is {}. ReportedSizeInBytes (actual reports from servers) is {}", new Object[]{tableNameWithType, tableSubtypeSize.estimatedSizeInBytes, tableSubtypeSize.reportedSizeInBytes});
        if (this._isLeaderForTable) {
            long existingStorageQuotaUtilization = tableSubtypeSize.estimatedSizeInBytes * 100L / allowedStorageBytes;
            this._controllerMetrics.setValueOfTableGauge(tableNameWithType, (AbstractMetrics.Gauge)ControllerGauge.TABLE_STORAGE_QUOTA_UTILIZATION, existingStorageQuotaUtilization);
        } else {
            this._controllerMetrics.setValueOfTableGauge(tableNameWithType, (AbstractMetrics.Gauge)ControllerGauge.TABLE_STORAGE_QUOTA_UTILIZATION, 0L);
        }
        long totalIncomingSegmentSizeBytes = incomingSegmentSizeBytes * (long)numReplicas;
        long estimatedFinalSizeBytes = tableSubtypeSize.estimatedSizeInBytes - existingSegmentSizeBytes + totalIncomingSegmentSizeBytes;
        if (estimatedFinalSizeBytes <= allowedStorageBytes) {
            String message = sizeDetails == null ? String.format("Appending Segment %s of Table %s is within quota. Total allowed storage size: %s ( = configured quota: %s * number replicas: %d). New estimated table size of all replicas: %s. Current table size of all replicas: %s. Incoming uncompressed segment size of all replicas: %s ( = single incoming uncompressed segment size: %s * number replicas: %d). Formula: New estimated size = current table size + incoming segment size", segmentName, tableNameWithType, DataSize.fromBytes((long)allowedStorageBytes), DataSize.fromBytes((long)quotaConfig.storageSizeBytes()), numReplicas, DataSize.fromBytes((long)estimatedFinalSizeBytes), DataSize.fromBytes((long)tableSubtypeSize.estimatedSizeInBytes), DataSize.fromBytes((long)totalIncomingSegmentSizeBytes), DataSize.fromBytes((long)incomingSegmentSizeBytes), numReplicas) : String.format("Refreshing Segment %s of Table %s is within quota. Total allowed storage size: %s ( = configured quota: %s * number replicas: %d). New estimated table size of all replicas: %s. Current table size of all replicas: %s. Incoming uncompressed segment size of all replicas: %s ( = single incoming uncompressed segment size: %s * number replicas: %d). Existing same segment size of all replicas: %s. Formula: New estimated size = current table size - existing same segment size + incoming segment size", segmentName, tableNameWithType, DataSize.fromBytes((long)allowedStorageBytes), DataSize.fromBytes((long)quotaConfig.storageSizeBytes()), numReplicas, DataSize.fromBytes((long)estimatedFinalSizeBytes), DataSize.fromBytes((long)tableSubtypeSize.estimatedSizeInBytes), DataSize.fromBytes((long)totalIncomingSegmentSizeBytes), DataSize.fromBytes((long)incomingSegmentSizeBytes), numReplicas, DataSize.fromBytes((long)existingSegmentSizeBytes));
            LOGGER.info(message);
            return StorageQuotaChecker.success(message);
        }
        String message = tableSubtypeSize.estimatedSizeInBytes > allowedStorageBytes ? String.format("Table %s already over quota. Existing estimated uncompressed table size of all replicas: %s > total allowed storage size: %s ( = configured quota: %s * num replicas: %d). Check if indexes were enabled recently and adjust table quota accordingly.", tableNameWithType, DataSize.fromBytes((long)tableSubtypeSize.estimatedSizeInBytes), DataSize.fromBytes((long)allowedStorageBytes), DataSize.fromBytes((long)quotaConfig.storageSizeBytes()), numReplicas) : String.format("Storage quota exceeded for Table %s. New estimated size: %s > total allowed storage size: %s, where new estimated size = existing estimated uncompressed size of all replicas: %s - existing segment sizes of all replicas: %s + (incoming uncompressed segment size: %s * number replicas: %d), total allowed storage size = configured quota: %s * number replicas: %d", tableNameWithType, DataSize.fromBytes((long)estimatedFinalSizeBytes), DataSize.fromBytes((long)allowedStorageBytes), DataSize.fromBytes((long)tableSubtypeSize.estimatedSizeInBytes), DataSize.fromBytes((long)existingSegmentSizeBytes), DataSize.fromBytes((long)incomingSegmentSizeBytes), numReplicas, DataSize.fromBytes((long)quotaConfig.storageSizeBytes()), numReplicas);
        LOGGER.warn(message);
        return StorageQuotaChecker.failure(message);
    }

    public static class QuotaCheckerResponse {
        public boolean isSegmentWithinQuota;
        public String reason;

        QuotaCheckerResponse(boolean isSegmentWithinQuota, String reason) {
            this.isSegmentWithinQuota = isSegmentWithinQuota;
            this.reason = reason;
        }
    }
}

