/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pinot.controller.helix.core.rebalance;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import javax.annotation.Nullable;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.commons.lang3.tuple.Triple;
import org.apache.helix.AccessOption;
import org.apache.helix.HelixDataAccessor;
import org.apache.helix.HelixManager;
import org.apache.helix.PropertyKey;
import org.apache.helix.model.ExternalView;
import org.apache.helix.model.IdealState;
import org.apache.helix.model.InstanceConfig;
import org.apache.helix.store.HelixPropertyStore;
import org.apache.helix.zookeeper.datamodel.ZNRecord;
import org.apache.helix.zookeeper.zkclient.exception.ZkBadVersionException;
import org.apache.pinot.common.assignment.InstanceAssignmentConfigUtils;
import org.apache.pinot.common.assignment.InstancePartitions;
import org.apache.pinot.common.assignment.InstancePartitionsUtils;
import org.apache.pinot.common.metrics.AbstractMetrics;
import org.apache.pinot.common.metrics.ControllerMetrics;
import org.apache.pinot.common.metrics.ControllerTimer;
import org.apache.pinot.common.tier.PinotServerTierStorage;
import org.apache.pinot.common.tier.Tier;
import org.apache.pinot.common.utils.config.TableConfigUtils;
import org.apache.pinot.common.utils.config.TierConfigUtils;
import org.apache.pinot.controller.helix.core.assignment.instance.InstanceAssignmentDriver;
import org.apache.pinot.controller.helix.core.assignment.segment.SegmentAssignment;
import org.apache.pinot.controller.helix.core.assignment.segment.SegmentAssignmentFactory;
import org.apache.pinot.controller.helix.core.assignment.segment.SegmentAssignmentUtils;
import org.apache.pinot.controller.helix.core.assignment.segment.StrictRealtimeSegmentAssignment;
import org.apache.pinot.controller.helix.core.rebalance.NoOpTableRebalanceObserver;
import org.apache.pinot.controller.helix.core.rebalance.RebalanceConfig;
import org.apache.pinot.controller.helix.core.rebalance.RebalanceResult;
import org.apache.pinot.controller.helix.core.rebalance.TableRebalanceObserver;
import org.apache.pinot.spi.config.table.TableConfig;
import org.apache.pinot.spi.config.table.TableType;
import org.apache.pinot.spi.config.table.assignment.InstanceAssignmentConfig;
import org.apache.pinot.spi.config.table.assignment.InstancePartitionsType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TableRebalancer {
    private static final Logger LOGGER = LoggerFactory.getLogger(TableRebalancer.class);
    private final HelixManager _helixManager;
    private final HelixDataAccessor _helixDataAccessor;
    private final TableRebalanceObserver _tableRebalanceObserver;
    private final ControllerMetrics _controllerMetrics;

    public TableRebalancer(HelixManager helixManager, @Nullable TableRebalanceObserver tableRebalanceObserver, @Nullable ControllerMetrics controllerMetrics) {
        this._helixManager = helixManager;
        this._tableRebalanceObserver = tableRebalanceObserver != null ? tableRebalanceObserver : new NoOpTableRebalanceObserver();
        this._helixDataAccessor = helixManager.getHelixDataAccessor();
        this._controllerMetrics = controllerMetrics;
    }

    public TableRebalancer(HelixManager helixManager) {
        this(helixManager, null, null);
    }

    public static String createUniqueRebalanceJobIdentifier() {
        return UUID.randomUUID().toString();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public RebalanceResult rebalance(TableConfig tableConfig, RebalanceConfig rebalanceConfig, @Nullable String rebalanceJobId) {
        RebalanceResult rebalanceResult;
        block3: {
            long startTime = System.currentTimeMillis();
            String tableNameWithType = tableConfig.getTableName();
            RebalanceResult.Status status = RebalanceResult.Status.UNKNOWN_ERROR;
            try {
                RebalanceResult result = this.doRebalance(tableConfig, rebalanceConfig, rebalanceJobId);
                status = result.getStatus();
                rebalanceResult = result;
                if (this._controllerMetrics == null) break block3;
            }
            catch (Throwable throwable) {
                if (this._controllerMetrics != null) {
                    this._controllerMetrics.addTimedTableValue(String.format("%s.%s", tableNameWithType, status.toString()), (AbstractMetrics.Timer)ControllerTimer.TABLE_REBALANCE_EXECUTION_TIME_MS, System.currentTimeMillis() - startTime, TimeUnit.MILLISECONDS);
                }
                throw throwable;
            }
            this._controllerMetrics.addTimedTableValue(String.format("%s.%s", tableNameWithType, status.toString()), (AbstractMetrics.Timer)ControllerTimer.TABLE_REBALANCE_EXECUTION_TIME_MS, System.currentTimeMillis() - startTime, TimeUnit.MILLISECONDS);
        }
        return rebalanceResult;
    }

    private RebalanceResult doRebalance(TableConfig tableConfig, RebalanceConfig rebalanceConfig, @Nullable String rebalanceJobId) {
        int minAvailableReplicas;
        Map<String, Map<String, String>> targetAssignment;
        boolean tierInstancePartitionsUnchanged;
        Map tierToInstancePartitionsMap;
        List<Tier> sortedTiers;
        boolean instancePartitionsUnchanged;
        Map instancePartitionsMap;
        IdealState currentIdealState;
        long startTimeMs = System.currentTimeMillis();
        String tableNameWithType = tableConfig.getTableName();
        if (rebalanceJobId == null) {
            rebalanceJobId = TableRebalancer.createUniqueRebalanceJobIdentifier();
        }
        boolean dryRun = rebalanceConfig.isDryRun();
        boolean reassignInstances = rebalanceConfig.isReassignInstances();
        boolean includeConsuming = rebalanceConfig.isIncludeConsuming();
        boolean bootstrap = rebalanceConfig.isBootstrap();
        boolean downtime = rebalanceConfig.isDowntime();
        int minReplicasToKeepUpForNoDowntime = rebalanceConfig.getMinAvailableReplicas();
        boolean lowDiskMode = rebalanceConfig.isLowDiskMode();
        boolean bestEfforts = rebalanceConfig.isBestEfforts();
        long externalViewCheckIntervalInMs = rebalanceConfig.getExternalViewCheckIntervalInMs();
        long externalViewStabilizationTimeoutInMs = rebalanceConfig.getExternalViewStabilizationTimeoutInMs();
        boolean enableStrictReplicaGroup = tableConfig.getRoutingConfig() != null && "strictReplicaGroup".equalsIgnoreCase(tableConfig.getRoutingConfig().getInstanceSelectorType());
        LOGGER.info("Start rebalancing table: {} with dryRun: {}, reassignInstances: {}, includeConsuming: {}, bootstrap: {}, downtime: {}, minReplicasToKeepUpForNoDowntime: {}, enableStrictReplicaGroup: {}, lowDiskMode: {}, bestEfforts: {}, externalViewCheckIntervalInMs: {}, externalViewStabilizationTimeoutInMs: {}", new Object[]{tableNameWithType, dryRun, reassignInstances, includeConsuming, bootstrap, downtime, minReplicasToKeepUpForNoDowntime, enableStrictReplicaGroup, lowDiskMode, bestEfforts, externalViewCheckIntervalInMs, externalViewStabilizationTimeoutInMs});
        PropertyKey idealStatePropertyKey = this._helixDataAccessor.keyBuilder().idealStates(tableNameWithType);
        try {
            currentIdealState = (IdealState)this._helixDataAccessor.getProperty(idealStatePropertyKey);
        }
        catch (Exception e) {
            LOGGER.warn("For rebalanceId: {}, caught exception while fetching IdealState for table: {}, aborting the rebalance", new Object[]{rebalanceJobId, tableNameWithType, e});
            return new RebalanceResult(rebalanceJobId, RebalanceResult.Status.FAILED, "Caught exception while fetching IdealState: " + e, null, null, null);
        }
        if (currentIdealState == null) {
            LOGGER.warn("For rebalanceId: {}, cannot find the IdealState for table: {}, aborting the rebalance", (Object)rebalanceJobId, (Object)tableNameWithType);
            return new RebalanceResult(rebalanceJobId, RebalanceResult.Status.FAILED, "Cannot find the IdealState for table", null, null, null);
        }
        if (!currentIdealState.isEnabled() && !downtime) {
            LOGGER.warn("For rebalanceId: {}, cannot rebalance disabled table: {} without downtime, aborting the rebalance", (Object)rebalanceJobId, (Object)tableNameWithType);
            return new RebalanceResult(rebalanceJobId, RebalanceResult.Status.FAILED, "Cannot rebalance disabled table without downtime", null, null, null);
        }
        LOGGER.info("For rebalanceId: {}, processing instance partitions for table: {}", (Object)rebalanceJobId, (Object)tableNameWithType);
        try {
            Pair<Map<InstancePartitionsType, InstancePartitions>, Boolean> instancePartitionsMapAndUnchanged = this.getInstancePartitionsMap(tableConfig, reassignInstances, bootstrap, dryRun);
            instancePartitionsMap = (Map)instancePartitionsMapAndUnchanged.getLeft();
            instancePartitionsUnchanged = (Boolean)instancePartitionsMapAndUnchanged.getRight();
        }
        catch (Exception e) {
            LOGGER.warn("For rebalanceId: {}, caught exception while fetching/calculating instance partitions for table: {}, aborting the rebalance", new Object[]{rebalanceJobId, tableNameWithType, e});
            return new RebalanceResult(rebalanceJobId, RebalanceResult.Status.FAILED, "Caught exception while fetching/calculating instance partitions: " + e, null, null, null);
        }
        try {
            sortedTiers = this.getSortedTiers(tableConfig);
            Pair<Map<String, InstancePartitions>, Boolean> tierToInstancePartitionsMapAndUnchanged = this.getTierToInstancePartitionsMap(tableConfig, sortedTiers, reassignInstances, bootstrap, dryRun);
            tierToInstancePartitionsMap = (Map)tierToInstancePartitionsMapAndUnchanged.getLeft();
            tierInstancePartitionsUnchanged = (Boolean)tierToInstancePartitionsMapAndUnchanged.getRight();
        }
        catch (Exception e) {
            LOGGER.warn("For rebalanceId: {}, caught exception while fetching/calculating tier instance partitions for table: {}, aborting the rebalance", new Object[]{rebalanceJobId, tableNameWithType, e});
            return new RebalanceResult(rebalanceJobId, RebalanceResult.Status.FAILED, "Caught exception while fetching/calculating tier instance partitions: " + e, null, null, null);
        }
        LOGGER.info("For rebalanceId: {}, calculating the target assignment for table: {}", (Object)rebalanceJobId, (Object)tableNameWithType);
        SegmentAssignment segmentAssignment = SegmentAssignmentFactory.getSegmentAssignment(this._helixManager, tableConfig, this._controllerMetrics);
        Map<String, Map<String, String>> currentAssignment = currentIdealState.getRecord().getMapFields();
        try {
            targetAssignment = segmentAssignment.rebalanceTable(currentAssignment, instancePartitionsMap, sortedTiers, tierToInstancePartitionsMap, rebalanceConfig);
        }
        catch (Exception e) {
            LOGGER.warn("For rebalanceId: {}, caught exception while calculating target assignment for table: {}, aborting the rebalance", new Object[]{rebalanceJobId, tableNameWithType, e});
            return new RebalanceResult(rebalanceJobId, RebalanceResult.Status.FAILED, "Caught exception while calculating target assignment: " + e, instancePartitionsMap, tierToInstancePartitionsMap, null);
        }
        boolean segmentAssignmentUnchanged = currentAssignment.equals(targetAssignment);
        LOGGER.info("For rebalanceId: {}, instancePartitionsUnchanged: {}, tierInstancePartitionsUnchanged: {}, segmentAssignmentUnchanged: {} for table: {}", new Object[]{rebalanceJobId, instancePartitionsUnchanged, tierInstancePartitionsUnchanged, segmentAssignmentUnchanged, tableNameWithType});
        if (segmentAssignmentUnchanged) {
            LOGGER.info("Table: {} is already balanced", (Object)tableNameWithType);
            if (instancePartitionsUnchanged && tierInstancePartitionsUnchanged) {
                return new RebalanceResult(rebalanceJobId, RebalanceResult.Status.NO_OP, "Table is already balanced", instancePartitionsMap, tierToInstancePartitionsMap, targetAssignment);
            }
            if (dryRun) {
                return new RebalanceResult(rebalanceJobId, RebalanceResult.Status.DONE, "Instance reassigned in dry-run mode, table is already balanced", instancePartitionsMap, tierToInstancePartitionsMap, targetAssignment);
            }
            return new RebalanceResult(rebalanceJobId, RebalanceResult.Status.DONE, "Instance reassigned, table is already balanced", instancePartitionsMap, tierToInstancePartitionsMap, targetAssignment);
        }
        if (dryRun) {
            LOGGER.info("For rebalanceId: {}, rebalancing table: {} in dry-run mode, returning the target assignment", (Object)rebalanceJobId, (Object)tableNameWithType);
            return new RebalanceResult(rebalanceJobId, RebalanceResult.Status.DONE, "Dry-run mode", instancePartitionsMap, tierToInstancePartitionsMap, targetAssignment);
        }
        if (downtime) {
            LOGGER.info("For rebalanceId: {}, rebalancing table: {} with downtime", (Object)rebalanceJobId, (Object)tableNameWithType);
            ZNRecord idealStateRecord = currentIdealState.getRecord();
            idealStateRecord.setMapFields(targetAssignment);
            currentIdealState.setNumPartitions(targetAssignment.size());
            currentIdealState.setReplicas(Integer.toString(targetAssignment.values().iterator().next().size()));
            try {
                Preconditions.checkState((boolean)this._helixDataAccessor.getBaseDataAccessor().set(idealStatePropertyKey.getPath(), (Object)idealStateRecord, idealStateRecord.getVersion(), AccessOption.PERSISTENT), (Object)"Failed to update IdealState");
                LOGGER.info("For rebalanceId: {}, finished rebalancing table: {} with downtime in {}ms.", new Object[]{rebalanceJobId, tableNameWithType, System.currentTimeMillis() - startTimeMs});
                return new RebalanceResult(rebalanceJobId, RebalanceResult.Status.DONE, "Success with downtime (replaced IdealState with the target segment assignment, ExternalView might not reach the target segment assignment yet)", instancePartitionsMap, tierToInstancePartitionsMap, targetAssignment);
            }
            catch (Exception e) {
                LOGGER.warn("For rebalanceId: {}, caught exception while updating IdealState for table: {}, aborting the rebalance", new Object[]{rebalanceJobId, tableNameWithType, e});
                return new RebalanceResult(rebalanceJobId, RebalanceResult.Status.FAILED, "Caught exception while updating IdealState: " + e, instancePartitionsMap, tierToInstancePartitionsMap, targetAssignment);
            }
        }
        this._tableRebalanceObserver.onTrigger(TableRebalanceObserver.Trigger.START_TRIGGER, currentAssignment, targetAssignment);
        List<String> segmentsToMove = SegmentAssignmentUtils.getSegmentsToMove(currentAssignment, targetAssignment);
        int numReplicas = Integer.MAX_VALUE;
        for (String segment : segmentsToMove) {
            numReplicas = Math.min(targetAssignment.get(segment).size(), numReplicas);
        }
        if (minReplicasToKeepUpForNoDowntime >= 0) {
            if (minReplicasToKeepUpForNoDowntime >= numReplicas) {
                String errorMsg = String.format("For rebalanceId: %s, Illegal config for minReplicasToKeepUpForNoDowntime: %d for table: %s, must be less than number of replicas: %d, aborting the rebalance", rebalanceJobId, minReplicasToKeepUpForNoDowntime, tableNameWithType, numReplicas);
                LOGGER.warn(errorMsg);
                this._tableRebalanceObserver.onError(errorMsg);
                return new RebalanceResult(rebalanceJobId, RebalanceResult.Status.FAILED, "Illegal min available replicas config", instancePartitionsMap, tierToInstancePartitionsMap, targetAssignment);
            }
            minAvailableReplicas = minReplicasToKeepUpForNoDowntime;
        } else {
            minAvailableReplicas = Math.max(numReplicas + minReplicasToKeepUpForNoDowntime, 0);
        }
        LOGGER.info("For rebalanceId: {}, rebalancing table: {} with minAvailableReplicas: {}, enableStrictReplicaGroup: {}, bestEfforts: {}, externalViewCheckIntervalInMs: {}, externalViewStabilizationTimeoutInMs: {}", new Object[]{rebalanceJobId, tableNameWithType, minAvailableReplicas, enableStrictReplicaGroup, bestEfforts, externalViewCheckIntervalInMs, externalViewStabilizationTimeoutInMs});
        int expectedVersion = currentIdealState.getRecord().getVersion();
        while (true) {
            IdealState idealState;
            HashSet<String> segmentsToMonitor = new HashSet<String>(segmentsToMove);
            segmentsToMove = SegmentAssignmentUtils.getSegmentsToMove(currentAssignment, targetAssignment);
            segmentsToMonitor.addAll(segmentsToMove);
            try {
                idealState = this.waitForExternalViewToConverge(tableNameWithType, bestEfforts, segmentsToMonitor, externalViewCheckIntervalInMs, externalViewStabilizationTimeoutInMs);
            }
            catch (Exception e) {
                String errorMsg = String.format("For rebalanceId: %s, caught exception while waiting for ExternalView to converge for table: %s, aborting the rebalance", rebalanceJobId, tableNameWithType);
                LOGGER.warn(errorMsg, (Throwable)e);
                if (this._tableRebalanceObserver.isStopped()) {
                    return new RebalanceResult(rebalanceJobId, this._tableRebalanceObserver.getStopStatus(), "Caught exception while waiting for ExternalView to converge: " + e, instancePartitionsMap, tierToInstancePartitionsMap, targetAssignment);
                }
                this._tableRebalanceObserver.onError(errorMsg);
                return new RebalanceResult(rebalanceJobId, RebalanceResult.Status.FAILED, "Caught exception while waiting for ExternalView to converge: " + e, instancePartitionsMap, tierToInstancePartitionsMap, targetAssignment);
            }
            ZNRecord idealStateRecord = idealState.getRecord();
            if (idealStateRecord.getVersion() != expectedVersion) {
                LOGGER.info("For rebalanceId: {}, idealState version changed while waiting for ExternalView to converge for table: {}, re-calculating the target assignment", (Object)rebalanceJobId, (Object)tableNameWithType);
                Map<String, Map<String, String>> oldAssignment = currentAssignment;
                currentAssignment = idealStateRecord.getMapFields();
                expectedVersion = idealStateRecord.getVersion();
                boolean segmentsToMoveChanged = false;
                if (segmentAssignment instanceof StrictRealtimeSegmentAssignment) {
                    segmentsToMoveChanged = true;
                } else {
                    for (String segment : segmentsToMove) {
                        Map currentInstanceStateMap;
                        Map oldInstanceStateMap = (Map)oldAssignment.get(segment);
                        if (oldInstanceStateMap.equals(currentInstanceStateMap = (Map)currentAssignment.get(segment))) continue;
                        LOGGER.info("For rebalanceId: {}, segment state changed in IdealState from: {} to: {} for table: {}, segment: {}, re-calculating the target assignment based on the new IdealState", new Object[]{rebalanceJobId, oldInstanceStateMap, currentInstanceStateMap, tableNameWithType, segment});
                        segmentsToMoveChanged = true;
                        break;
                    }
                }
                if (segmentsToMoveChanged) {
                    try {
                        instancePartitionsMap = (Map)this.getInstancePartitionsMap(tableConfig, reassignInstances, bootstrap, false).getLeft();
                        tierToInstancePartitionsMap = (Map)this.getTierToInstancePartitionsMap(tableConfig, sortedTiers, reassignInstances, bootstrap, false).getLeft();
                        targetAssignment = segmentAssignment.rebalanceTable(currentAssignment, instancePartitionsMap, sortedTiers, tierToInstancePartitionsMap, rebalanceConfig);
                    }
                    catch (Exception e) {
                        String errorMsg = String.format("For rebalanceId: %s, caught exception while re-calculating the target assignment for table: %s, aborting the rebalance", rebalanceJobId, tableNameWithType);
                        LOGGER.warn(errorMsg, (Throwable)e);
                        this._tableRebalanceObserver.onError(errorMsg);
                        return new RebalanceResult(rebalanceJobId, RebalanceResult.Status.FAILED, "Caught exception while re-calculating the target assignment: " + e, instancePartitionsMap, tierToInstancePartitionsMap, targetAssignment);
                    }
                } else {
                    LOGGER.info("For rebalanceId:{}, no state change found for segments to be moved, re-calculating the target assignment based on the previous target assignment for table: {}", (Object)rebalanceJobId, (Object)tableNameWithType);
                    Map<String, Map<String, String>> oldTargetAssignment = targetAssignment;
                    targetAssignment = new HashMap<String, Map<String, String>>(currentAssignment);
                    for (String segment : segmentsToMove) {
                        targetAssignment.put(segment, oldTargetAssignment.get(segment));
                    }
                }
            }
            if (currentAssignment.equals(targetAssignment)) {
                String msg = String.format("For rebalanceId: %s, finished rebalancing table: %s with minAvailableReplicas: %d, enableStrictReplicaGroup: %b, bestEfforts: %b in %d ms.", rebalanceJobId, tableNameWithType, minAvailableReplicas, enableStrictReplicaGroup, bestEfforts, System.currentTimeMillis() - startTimeMs);
                LOGGER.info(msg);
                this._tableRebalanceObserver.onSuccess(msg);
                return new RebalanceResult(rebalanceJobId, RebalanceResult.Status.DONE, "Success with minAvailableReplicas: " + minAvailableReplicas + " (both IdealState and ExternalView should reach the target segment assignment)", instancePartitionsMap, tierToInstancePartitionsMap, targetAssignment);
            }
            this._tableRebalanceObserver.onTrigger(TableRebalanceObserver.Trigger.IDEAL_STATE_CHANGE_TRIGGER, currentAssignment, targetAssignment);
            if (this._tableRebalanceObserver.isStopped()) {
                return new RebalanceResult(rebalanceJobId, this._tableRebalanceObserver.getStopStatus(), "Rebalance has stopped already before updating the IdealState", instancePartitionsMap, tierToInstancePartitionsMap, targetAssignment);
            }
            Map<String, Map<String, String>> nextAssignment = TableRebalancer.getNextAssignment(currentAssignment, targetAssignment, minAvailableReplicas, enableStrictReplicaGroup, lowDiskMode);
            LOGGER.info("For rebalanceId: {}, got the next assignment for table: {} with number of segments to be moved to each instance: {}", new Object[]{rebalanceJobId, tableNameWithType, SegmentAssignmentUtils.getNumSegmentsToBeMovedPerInstance(currentAssignment, nextAssignment)});
            idealStateRecord.setMapFields(nextAssignment);
            idealState.setNumPartitions(nextAssignment.size());
            idealState.setReplicas(Integer.toString(nextAssignment.values().iterator().next().size()));
            try {
                Preconditions.checkState((boolean)this._helixDataAccessor.getBaseDataAccessor().set(idealStatePropertyKey.getPath(), (Object)idealStateRecord, expectedVersion, AccessOption.PERSISTENT), (Object)"Failed to update IdealState");
                currentAssignment = nextAssignment;
                ++expectedVersion;
                LOGGER.info("For rebalanceId: {}, successfully updated the IdealState for table: {}", (Object)rebalanceJobId, (Object)tableNameWithType);
                continue;
            }
            catch (ZkBadVersionException e) {
                LOGGER.info("For rebalanceId: {}, version changed while updating IdealState for table: {}", (Object)rebalanceJobId, (Object)tableNameWithType);
                continue;
            }
            catch (Exception e) {
                String errorMsg = String.format("For rebalanceId: %s, caught exception while updating IdealState for table: %s, aborting the rebalance", rebalanceJobId, tableNameWithType);
                LOGGER.warn(errorMsg, (Throwable)e);
                this._tableRebalanceObserver.onError(errorMsg);
                return new RebalanceResult(rebalanceJobId, RebalanceResult.Status.FAILED, "Caught exception while updating IdealState: " + e, instancePartitionsMap, tierToInstancePartitionsMap, targetAssignment);
            }
            break;
        }
    }

    private Pair<Map<InstancePartitionsType, InstancePartitions>, Boolean> getInstancePartitionsMap(TableConfig tableConfig, boolean reassignInstances, boolean bootstrap, boolean dryRun) {
        boolean instancePartitionsUnchanged;
        TreeMap<InstancePartitionsType, InstancePartitions> instancePartitionsMap = new TreeMap<InstancePartitionsType, InstancePartitions>();
        if (tableConfig.getTableType() == TableType.OFFLINE) {
            Pair<InstancePartitions, Boolean> partitionAndUnchangedForOffline = this.getInstancePartitions(tableConfig, InstancePartitionsType.OFFLINE, reassignInstances, bootstrap, dryRun);
            instancePartitionsMap.put(InstancePartitionsType.OFFLINE, (InstancePartitions)partitionAndUnchangedForOffline.getLeft());
            instancePartitionsUnchanged = (Boolean)partitionAndUnchangedForOffline.getRight();
        } else {
            Pair<InstancePartitions, Boolean> partitionAndUnchangedForConsuming = this.getInstancePartitions(tableConfig, InstancePartitionsType.CONSUMING, reassignInstances, bootstrap, dryRun);
            instancePartitionsMap.put(InstancePartitionsType.CONSUMING, (InstancePartitions)partitionAndUnchangedForConsuming.getLeft());
            instancePartitionsUnchanged = (Boolean)partitionAndUnchangedForConsuming.getRight();
            String tableNameWithType = tableConfig.getTableName();
            if (InstanceAssignmentConfigUtils.shouldRelocateCompletedSegments((TableConfig)tableConfig)) {
                Pair<InstancePartitions, Boolean> partitionAndUnchangedForCompleted = this.getInstancePartitions(tableConfig, InstancePartitionsType.COMPLETED, reassignInstances, bootstrap, dryRun);
                LOGGER.info("COMPLETED segments should be relocated, fetching/computing COMPLETED instance partitions for table: {}", (Object)tableNameWithType);
                instancePartitionsMap.put(InstancePartitionsType.COMPLETED, (InstancePartitions)partitionAndUnchangedForCompleted.getLeft());
                instancePartitionsUnchanged &= ((Boolean)partitionAndUnchangedForCompleted.getRight()).booleanValue();
            } else {
                LOGGER.info("COMPLETED segments should not be relocated, skipping fetching/computing COMPLETED instance partitions for table: {}", (Object)tableNameWithType);
                if (!dryRun) {
                    String instancePartitionsName = InstancePartitionsUtils.getInstancePartitionsName((String)tableNameWithType, (String)InstancePartitionsType.COMPLETED.toString());
                    LOGGER.info("Removing instance partitions: {} from ZK if it exists", (Object)instancePartitionsName);
                    InstancePartitionsUtils.removeInstancePartitions((HelixPropertyStore)this._helixManager.getHelixPropertyStore(), (String)instancePartitionsName);
                }
            }
        }
        return Pair.of(instancePartitionsMap, (Object)instancePartitionsUnchanged);
    }

    private Pair<InstancePartitions, Boolean> getInstancePartitions(TableConfig tableConfig, InstancePartitionsType instancePartitionsType, boolean reassignInstances, boolean bootstrap, boolean dryRun) {
        String tableNameWithType = tableConfig.getTableName();
        String instancePartitionsName = InstancePartitionsUtils.getInstancePartitionsName((String)tableNameWithType, (String)instancePartitionsType.toString());
        InstancePartitions existingInstancePartitions = InstancePartitionsUtils.fetchInstancePartitions((HelixPropertyStore)this._helixManager.getHelixPropertyStore(), (String)instancePartitionsName);
        if (reassignInstances) {
            boolean noExistingInstancePartitions;
            if (InstanceAssignmentConfigUtils.allowInstanceAssignment((TableConfig)tableConfig, (InstancePartitionsType)instancePartitionsType)) {
                boolean instancePartitionsUnchanged;
                InstancePartitions instancePartitions;
                boolean hasPreConfiguredInstancePartitions = TableConfigUtils.hasPreConfiguredInstancePartitions((TableConfig)tableConfig, (InstancePartitionsType)instancePartitionsType);
                boolean isPreConfigurationBasedAssignment = InstanceAssignmentConfigUtils.isMirrorServerSetAssignment((TableConfig)tableConfig, (InstancePartitionsType)instancePartitionsType);
                InstanceAssignmentDriver instanceAssignmentDriver = new InstanceAssignmentDriver(tableConfig);
                if (!hasPreConfiguredInstancePartitions) {
                    LOGGER.info("Reassigning {} instances for table: {}", (Object)instancePartitionsType, (Object)tableNameWithType);
                    instancePartitions = instanceAssignmentDriver.assignInstances(instancePartitionsType, this._helixDataAccessor.getChildValues(this._helixDataAccessor.keyBuilder().instanceConfigs(), true), bootstrap ? null : existingInstancePartitions);
                    instancePartitionsUnchanged = instancePartitions.equals((Object)existingInstancePartitions);
                    if (!dryRun && !instancePartitionsUnchanged) {
                        LOGGER.info("Persisting instance partitions: {} to ZK", (Object)instancePartitions);
                        InstancePartitionsUtils.persistInstancePartitions((HelixPropertyStore)this._helixManager.getHelixPropertyStore(), (InstancePartitions)instancePartitions);
                    }
                } else {
                    String referenceInstancePartitionsName = (String)tableConfig.getInstancePartitionsMap().get(instancePartitionsType);
                    if (isPreConfigurationBasedAssignment) {
                        InstancePartitions preConfiguredInstancePartitions = InstancePartitionsUtils.fetchInstancePartitionsWithRename((HelixPropertyStore)this._helixManager.getHelixPropertyStore(), (String)referenceInstancePartitionsName, (String)instancePartitionsName);
                        instancePartitions = instanceAssignmentDriver.assignInstances(instancePartitionsType, (List<InstanceConfig>)this._helixDataAccessor.getChildValues(this._helixDataAccessor.keyBuilder().instanceConfigs(), true), bootstrap ? null : existingInstancePartitions, preConfiguredInstancePartitions);
                        instancePartitionsUnchanged = instancePartitions.equals((Object)existingInstancePartitions);
                        if (!dryRun && !instancePartitionsUnchanged) {
                            LOGGER.info("Persisting instance partitions: {} (based on {})", (Object)instancePartitions, (Object)preConfiguredInstancePartitions);
                            InstancePartitionsUtils.persistInstancePartitions((HelixPropertyStore)this._helixManager.getHelixPropertyStore(), (InstancePartitions)instancePartitions);
                        }
                    } else {
                        instancePartitions = InstancePartitionsUtils.fetchInstancePartitionsWithRename((HelixPropertyStore)this._helixManager.getHelixPropertyStore(), (String)referenceInstancePartitionsName, (String)instancePartitionsName);
                        instancePartitionsUnchanged = instancePartitions.equals((Object)existingInstancePartitions);
                        if (!dryRun && !instancePartitionsUnchanged) {
                            LOGGER.info("Persisting instance partitions: {} (referencing {})", (Object)instancePartitions, (Object)referenceInstancePartitionsName);
                            InstancePartitionsUtils.persistInstancePartitions((HelixPropertyStore)this._helixManager.getHelixPropertyStore(), (InstancePartitions)instancePartitions);
                        }
                    }
                }
                return Pair.of((Object)instancePartitions, (Object)instancePartitionsUnchanged);
            }
            LOGGER.info("{} instance assignment is not allowed, using default instance partitions for table: {}", (Object)instancePartitionsType, (Object)tableNameWithType);
            InstancePartitions instancePartitions = InstancePartitionsUtils.computeDefaultInstancePartitions((HelixManager)this._helixManager, (TableConfig)tableConfig, (InstancePartitionsType)instancePartitionsType);
            boolean bl = noExistingInstancePartitions = existingInstancePartitions == null;
            if (!dryRun && !noExistingInstancePartitions) {
                LOGGER.info("Removing instance partitions: {} from ZK", (Object)instancePartitionsName);
                InstancePartitionsUtils.removeInstancePartitions((HelixPropertyStore)this._helixManager.getHelixPropertyStore(), (String)instancePartitionsName);
            }
            return Pair.of((Object)instancePartitions, (Object)noExistingInstancePartitions);
        }
        LOGGER.info("Fetching/computing {} instance partitions for table: {}", (Object)instancePartitionsType, (Object)tableNameWithType);
        return Pair.of((Object)InstancePartitionsUtils.fetchOrComputeInstancePartitions((HelixManager)this._helixManager, (TableConfig)tableConfig, (InstancePartitionsType)instancePartitionsType), (Object)true);
    }

    @Nullable
    private List<Tier> getSortedTiers(TableConfig tableConfig) {
        List tierConfigs = tableConfig.getTierConfigsList();
        if (CollectionUtils.isNotEmpty((Collection)tierConfigs)) {
            return TierConfigUtils.getSortedTiersForStorageType((List)tierConfigs, (String)"pinot_server", (HelixManager)this._helixManager);
        }
        return null;
    }

    private Pair<Map<String, InstancePartitions>, Boolean> getTierToInstancePartitionsMap(TableConfig tableConfig, @Nullable List<Tier> sortedTiers, boolean reassignInstances, boolean bootstrap, boolean dryRun) {
        if (sortedTiers == null) {
            return Pair.of(null, (Object)true);
        }
        boolean instancePartitionsUnchanged = true;
        HashMap<String, InstancePartitions> tierToInstancePartitionsMap = new HashMap<String, InstancePartitions>();
        for (Tier tier : sortedTiers) {
            LOGGER.info("Fetching/computing instance partitions for tier: {} of table: {}", (Object)tier.getName(), (Object)tableConfig.getTableName());
            Pair<InstancePartitions, Boolean> partitionsAndUnchanged = this.getInstancePartitionsForTier(tableConfig, tier, reassignInstances, bootstrap, dryRun);
            tierToInstancePartitionsMap.put(tier.getName(), (InstancePartitions)partitionsAndUnchanged.getLeft());
            instancePartitionsUnchanged = instancePartitionsUnchanged && (Boolean)partitionsAndUnchanged.getRight() != false;
        }
        return Pair.of(tierToInstancePartitionsMap, (Object)instancePartitionsUnchanged);
    }

    private Pair<InstancePartitions, Boolean> getInstancePartitionsForTier(TableConfig tableConfig, Tier tier, boolean reassignInstances, boolean bootstrap, boolean dryRun) {
        String tableNameWithType = tableConfig.getTableName();
        String tierName = tier.getName();
        String instancePartitionsName = InstancePartitionsUtils.getInstancePartitionsNameForTier((String)tableNameWithType, (String)tierName);
        InstancePartitions existingInstancePartitions = InstancePartitionsUtils.fetchInstancePartitions((HelixPropertyStore)this._helixManager.getHelixPropertyStore(), (String)instancePartitionsName);
        if (reassignInstances) {
            InstanceAssignmentConfig instanceAssignmentConfig;
            Map instanceAssignmentConfigMap = tableConfig.getInstanceAssignmentConfigMap();
            InstanceAssignmentConfig instanceAssignmentConfig2 = instanceAssignmentConfig = instanceAssignmentConfigMap != null ? (InstanceAssignmentConfig)instanceAssignmentConfigMap.get(tierName) : null;
            if (instanceAssignmentConfig == null) {
                boolean noExistingInstancePartitions;
                LOGGER.info("Instance assignment config for tier: {} does not exist for table: {}, using default instance partitions", (Object)tierName, (Object)tableNameWithType);
                PinotServerTierStorage storage = (PinotServerTierStorage)tier.getStorage();
                InstancePartitions instancePartitions = InstancePartitionsUtils.computeDefaultInstancePartitionsForTag((HelixManager)this._helixManager, (String)tableNameWithType, (String)tierName, (String)storage.getServerTag());
                boolean bl = noExistingInstancePartitions = existingInstancePartitions == null;
                if (!dryRun && !noExistingInstancePartitions) {
                    LOGGER.info("Removing instance partitions: {} from ZK", (Object)instancePartitionsName);
                    InstancePartitionsUtils.removeInstancePartitions((HelixPropertyStore)this._helixManager.getHelixPropertyStore(), (String)instancePartitionsName);
                }
                return Pair.of((Object)instancePartitions, (Object)noExistingInstancePartitions);
            }
            InstanceAssignmentDriver instanceAssignmentDriver = new InstanceAssignmentDriver(tableConfig);
            InstancePartitions instancePartitions = instanceAssignmentDriver.assignInstances(tierName, (List<InstanceConfig>)this._helixDataAccessor.getChildValues(this._helixDataAccessor.keyBuilder().instanceConfigs(), true), bootstrap ? null : existingInstancePartitions, instanceAssignmentConfig);
            boolean instancePartitionsUnchanged = instancePartitions.equals((Object)existingInstancePartitions);
            if (!dryRun && !instancePartitionsUnchanged) {
                LOGGER.info("Persisting instance partitions: {} to ZK", (Object)instancePartitions);
                InstancePartitionsUtils.persistInstancePartitions((HelixPropertyStore)this._helixManager.getHelixPropertyStore(), (InstancePartitions)instancePartitions);
            }
            return Pair.of((Object)instancePartitions, (Object)instancePartitionsUnchanged);
        }
        if (existingInstancePartitions != null) {
            return Pair.of((Object)existingInstancePartitions, (Object)true);
        }
        PinotServerTierStorage storage = (PinotServerTierStorage)tier.getStorage();
        InstancePartitions instancePartitions = InstancePartitionsUtils.computeDefaultInstancePartitionsForTag((HelixManager)this._helixManager, (String)tableNameWithType, (String)tierName, (String)storage.getServerTag());
        return Pair.of((Object)instancePartitions, (Object)true);
    }

    private IdealState waitForExternalViewToConverge(String tableNameWithType, boolean bestEfforts, Set<String> segmentsToMonitor, long externalViewCheckIntervalInMs, long externalViewStabilizationTimeoutInMs) throws InterruptedException, TimeoutException {
        IdealState idealState;
        long endTimeMs = System.currentTimeMillis() + externalViewStabilizationTimeoutInMs;
        do {
            LOGGER.debug("Start to check if ExternalView converges to IdealStates");
            idealState = (IdealState)this._helixDataAccessor.getProperty(this._helixDataAccessor.keyBuilder().idealStates(tableNameWithType));
            Preconditions.checkState((idealState != null ? 1 : 0) != 0, (Object)"Failed to find the IdealState");
            ExternalView externalView = (ExternalView)this._helixDataAccessor.getProperty(this._helixDataAccessor.keyBuilder().externalView(tableNameWithType));
            if (externalView != null) {
                this._tableRebalanceObserver.onTrigger(TableRebalanceObserver.Trigger.EXTERNAL_VIEW_TO_IDEAL_STATE_CONVERGENCE_TRIGGER, externalView.getRecord().getMapFields(), idealState.getRecord().getMapFields());
                if (this._tableRebalanceObserver.isStopped()) {
                    throw new RuntimeException(String.format("Rebalance for table: %s has already stopped with status: %s", new Object[]{tableNameWithType, this._tableRebalanceObserver.getStopStatus()}));
                }
                if (TableRebalancer.isExternalViewConverged(tableNameWithType, externalView.getRecord().getMapFields(), idealState.getRecord().getMapFields(), bestEfforts, segmentsToMonitor)) {
                    LOGGER.info("ExternalView converged for table: {}", (Object)tableNameWithType);
                    return idealState;
                }
            }
            LOGGER.debug("ExternalView has not converged to IdealStates. Retry after: {}ms", (Object)externalViewCheckIntervalInMs);
            Thread.sleep(externalViewCheckIntervalInMs);
        } while (System.currentTimeMillis() < endTimeMs);
        if (bestEfforts) {
            LOGGER.warn("ExternalView has not converged within: {}ms for table: {}, continuing the rebalance (best-efforts)", (Object)externalViewStabilizationTimeoutInMs, (Object)tableNameWithType);
            return idealState;
        }
        throw new TimeoutException(String.format("ExternalView has not converged within: %d ms for table: %s", externalViewStabilizationTimeoutInMs, tableNameWithType));
    }

    @VisibleForTesting
    static boolean isExternalViewConverged(String tableNameWithType, Map<String, Map<String, String>> externalViewSegmentStates, Map<String, Map<String, String>> idealStateSegmentStates, boolean bestEfforts, @Nullable Set<String> segmentsToMonitor) {
        for (Map.Entry<String, Map<String, String>> entry : idealStateSegmentStates.entrySet()) {
            String segmentName = entry.getKey();
            if (segmentsToMonitor != null && !segmentsToMonitor.contains(segmentName)) continue;
            Map<String, String> externalViewInstanceStateMap = externalViewSegmentStates.get(segmentName);
            Map<String, String> idealStateInstanceStateMap = entry.getValue();
            for (Map.Entry<String, String> instanceStateEntry : idealStateInstanceStateMap.entrySet()) {
                String idealStateInstanceState = instanceStateEntry.getValue();
                if (idealStateInstanceState.equals("OFFLINE")) continue;
                if (externalViewInstanceStateMap == null) {
                    return false;
                }
                String instanceName = instanceStateEntry.getKey();
                String externalViewInstanceState = externalViewInstanceStateMap.get(instanceName);
                if (idealStateInstanceState.equals(externalViewInstanceState)) continue;
                if ("ERROR".equals(externalViewInstanceState)) {
                    if (bestEfforts) {
                        LOGGER.warn("Found ERROR instance: {} for segment: {}, table: {}, counting it as good state (best-efforts)", new Object[]{instanceName, segmentName, tableNameWithType});
                        continue;
                    }
                    LOGGER.warn("Found ERROR instance: {} for segment: {}, table: {}", new Object[]{instanceName, segmentName, tableNameWithType});
                    throw new IllegalStateException("Found segments in ERROR state");
                }
                return false;
            }
        }
        return true;
    }

    @VisibleForTesting
    static Map<String, Map<String, String>> getNextAssignment(Map<String, Map<String, String>> currentAssignment, Map<String, Map<String, String>> targetAssignment, int minAvailableReplicas, boolean enableStrictReplicaGroup, boolean lowDiskMode) {
        return enableStrictReplicaGroup ? TableRebalancer.getNextStrictReplicaGroupAssignment(currentAssignment, targetAssignment, minAvailableReplicas, lowDiskMode) : TableRebalancer.getNextNonStrictReplicaGroupAssignment(currentAssignment, targetAssignment, minAvailableReplicas, lowDiskMode);
    }

    private static Map<String, Map<String, String>> getNextStrictReplicaGroupAssignment(Map<String, Map<String, String>> currentAssignment, Map<String, Map<String, String>> targetAssignment, int minAvailableReplicas, boolean lowDiskMode) {
        TreeMap<String, Map<String, String>> nextAssignment = new TreeMap<String, Map<String, String>>();
        Map<String, Integer> numSegmentsToOffloadMap = TableRebalancer.getNumSegmentsToOffloadMap(currentAssignment, targetAssignment);
        HashMap<Pair<Set<String>, Set<String>>, Set<String>> assignmentMap = new HashMap<Pair<Set<String>, Set<String>>, Set<String>>();
        HashMap<Set, Set> availableInstancesMap = new HashMap<Set, Set>();
        for (Map.Entry<String, Map<String, String>> entry : currentAssignment.entrySet()) {
            String segmentName = entry.getKey();
            Map<String, String> currentInstanceStateMap = entry.getValue();
            Map<String, String> targetInstanceStateMap = targetAssignment.get(segmentName);
            SingleSegmentAssignment assignment = TableRebalancer.getNextSingleSegmentAssignment(currentInstanceStateMap, targetInstanceStateMap, minAvailableReplicas, lowDiskMode, numSegmentsToOffloadMap, assignmentMap);
            Set<String> assignedInstances = assignment._instanceStateMap.keySet();
            Set<String> availableInstances = assignment._availableInstances;
            availableInstancesMap.compute(assignedInstances, (k, currentAvailableInstances) -> {
                if (currentAvailableInstances == null) {
                    nextAssignment.put(segmentName, assignment._instanceStateMap);
                    TableRebalancer.updateNumSegmentsToOffloadMap(numSegmentsToOffloadMap, currentInstanceStateMap.keySet(), k);
                    return availableInstances;
                }
                availableInstances.retainAll((Collection<?>)currentAvailableInstances);
                if (availableInstances.size() >= minAvailableReplicas) {
                    nextAssignment.put(segmentName, assignment._instanceStateMap);
                    TableRebalancer.updateNumSegmentsToOffloadMap(numSegmentsToOffloadMap, currentInstanceStateMap.keySet(), k);
                    return availableInstances;
                }
                nextAssignment.put(segmentName, currentInstanceStateMap);
                return currentAvailableInstances;
            });
        }
        return nextAssignment;
    }

    private static Map<String, Map<String, String>> getNextNonStrictReplicaGroupAssignment(Map<String, Map<String, String>> currentAssignment, Map<String, Map<String, String>> targetAssignment, int minAvailableReplicas, boolean lowDiskMode) {
        TreeMap<String, Map<String, String>> nextAssignment = new TreeMap<String, Map<String, String>>();
        Map<String, Integer> numSegmentsToOffloadMap = TableRebalancer.getNumSegmentsToOffloadMap(currentAssignment, targetAssignment);
        HashMap<Pair<Set<String>, Set<String>>, Set<String>> assignmentMap = new HashMap<Pair<Set<String>, Set<String>>, Set<String>>();
        for (Map.Entry<String, Map<String, String>> entry : currentAssignment.entrySet()) {
            String segmentName = entry.getKey();
            Map<String, String> currentInstanceStateMap = entry.getValue();
            Map<String, String> targetInstanceStateMap = targetAssignment.get(segmentName);
            Map<String, String> nextInstanceStateMap = TableRebalancer.getNextSingleSegmentAssignment(currentInstanceStateMap, targetInstanceStateMap, (int)minAvailableReplicas, (boolean)lowDiskMode, numSegmentsToOffloadMap, assignmentMap)._instanceStateMap;
            nextAssignment.put(segmentName, nextInstanceStateMap);
            TableRebalancer.updateNumSegmentsToOffloadMap(numSegmentsToOffloadMap, currentInstanceStateMap.keySet(), nextInstanceStateMap.keySet());
        }
        return nextAssignment;
    }

    @VisibleForTesting
    static Map<String, Integer> getNumSegmentsToOffloadMap(Map<String, Map<String, String>> currentAssignment, Map<String, Map<String, String>> targetAssignment) {
        HashMap<String, Integer> numSegmentsToOffloadMap = new HashMap<String, Integer>();
        for (Map<String, String> currentInstanceStateMap : currentAssignment.values()) {
            for (String currentInstance : currentInstanceStateMap.keySet()) {
                numSegmentsToOffloadMap.merge(currentInstance, 1, Integer::sum);
            }
        }
        for (Map<String, String> targetInstanceStateMap : targetAssignment.values()) {
            for (String targetInstance : targetInstanceStateMap.keySet()) {
                numSegmentsToOffloadMap.merge(targetInstance, -1, Integer::sum);
            }
        }
        return numSegmentsToOffloadMap;
    }

    private static void updateNumSegmentsToOffloadMap(Map<String, Integer> numSegmentsToOffloadMap, Set<String> currentInstances, Set<String> newInstances) {
        for (String currentInstance : currentInstances) {
            numSegmentsToOffloadMap.merge(currentInstance, -1, Integer::sum);
        }
        for (String newInstance : newInstances) {
            numSegmentsToOffloadMap.merge(newInstance, 1, Integer::sum);
        }
    }

    @VisibleForTesting
    static SingleSegmentAssignment getNextSingleSegmentAssignment(Map<String, String> currentInstanceStateMap, Map<String, String> targetInstanceStateMap, int minAvailableReplicas, boolean lowDiskMode, Map<String, Integer> numSegmentsToOffloadMap, Map<Pair<Set<String>, Set<String>>, Set<String>> assignmentMap) {
        int numInstancesToAdd;
        Set<String> targetInstances;
        TreeMap<String, String> nextInstanceStateMap = new TreeMap<String, String>();
        Set<String> currentInstances = currentInstanceStateMap.keySet();
        Pair assignmentKey = Pair.of(currentInstances, targetInstances = targetInstanceStateMap.keySet());
        Set<String> instancesToAssign = assignmentMap.get(assignmentKey);
        if (instancesToAssign != null) {
            TreeSet<String> availableInstances = new TreeSet<String>();
            for (String instanceName : instancesToAssign) {
                String currentInstanceState = currentInstanceStateMap.get(instanceName);
                String targetInstanceState = targetInstanceStateMap.get(instanceName);
                if (currentInstanceState != null) {
                    availableInstances.add(instanceName);
                    nextInstanceStateMap.put(instanceName, targetInstanceState != null ? targetInstanceState : currentInstanceState);
                    continue;
                }
                nextInstanceStateMap.put(instanceName, targetInstanceState);
            }
            return new SingleSegmentAssignment(nextInstanceStateMap, availableInstances);
        }
        for (Map.Entry<String, String> entry : targetInstanceStateMap.entrySet()) {
            String instanceName = entry.getKey();
            if (!currentInstanceStateMap.containsKey(instanceName)) continue;
            nextInstanceStateMap.put(instanceName, entry.getValue());
        }
        int numInstancesToKeep = minAvailableReplicas - nextInstanceStateMap.size();
        if (numInstancesToKeep > 0) {
            List<Triple<String, String, Integer>> instancesInfo = TableRebalancer.getSortedInstancesOnNumSegmentsToOffload(currentInstanceStateMap, nextInstanceStateMap, numSegmentsToOffloadMap);
            numInstancesToKeep = Integer.min(numInstancesToKeep, instancesInfo.size());
            for (int i = 0; i < numInstancesToKeep; ++i) {
                Triple<String, String, Integer> instanceInfo = instancesInfo.get(i);
                nextInstanceStateMap.put((String)instanceInfo.getLeft(), (String)instanceInfo.getMiddle());
            }
        }
        TreeSet<String> availableInstances = new TreeSet<String>(nextInstanceStateMap.keySet());
        if (!(lowDiskMode && currentInstanceStateMap.size() != nextInstanceStateMap.size() || (numInstancesToAdd = targetInstanceStateMap.size() - nextInstanceStateMap.size()) <= 0)) {
            List<Triple<String, String, Integer>> instancesInfo = TableRebalancer.getSortedInstancesOnNumSegmentsToOffload(targetInstanceStateMap, nextInstanceStateMap, numSegmentsToOffloadMap);
            for (int i = 0; i < numInstancesToAdd; ++i) {
                Triple<String, String, Integer> instanceInfo = instancesInfo.get(i);
                nextInstanceStateMap.put((String)instanceInfo.getLeft(), (String)instanceInfo.getMiddle());
            }
        }
        assignmentMap.put((Pair<Set<String>, Set<String>>)assignmentKey, nextInstanceStateMap.keySet());
        return new SingleSegmentAssignment(nextInstanceStateMap, availableInstances);
    }

    private static List<Triple<String, String, Integer>> getSortedInstancesOnNumSegmentsToOffload(Map<String, String> instanceStateMap, Map<String, String> nextInstanceStateMap, Map<String, Integer> numSegmentsToOffloadMap) {
        ArrayList<Triple<String, String, Integer>> instancesInfo = new ArrayList<Triple<String, String, Integer>>(instanceStateMap.size());
        for (Map.Entry<String, String> entry : instanceStateMap.entrySet()) {
            String instanceName = entry.getKey();
            if (nextInstanceStateMap.containsKey(instanceName)) continue;
            instancesInfo.add((Triple<String, String, Integer>)Triple.of((Object)instanceName, (Object)entry.getValue(), (Object)numSegmentsToOffloadMap.get(instanceName)));
        }
        instancesInfo.sort(Comparator.comparingInt(Triple::getRight).thenComparing(Triple::getLeft));
        return instancesInfo;
    }

    @VisibleForTesting
    static class SingleSegmentAssignment {
        final Map<String, String> _instanceStateMap;
        final Set<String> _availableInstances;

        SingleSegmentAssignment(Map<String, String> instanceStateMap, Set<String> availableInstances) {
            this._instanceStateMap = instanceStateMap;
            this._availableInstances = availableInstances;
        }
    }
}

