/*
 * Decompiled with CFR 0.152.
 */
package org.apache.shiro.samples.aspectj.bank;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.apache.shiro.SecurityUtils;
import org.apache.shiro.authz.annotation.RequiresPermissions;
import org.apache.shiro.samples.aspectj.bank.Account;
import org.apache.shiro.samples.aspectj.bank.AccountNotFoundException;
import org.apache.shiro.samples.aspectj.bank.AccountTransaction;
import org.apache.shiro.samples.aspectj.bank.BankService;
import org.apache.shiro.samples.aspectj.bank.InactiveAccountException;
import org.apache.shiro.samples.aspectj.bank.NotEnoughFundsException;
import org.apache.shiro.subject.Subject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SecureBankService
implements BankService {
    private static final Logger log = LoggerFactory.getLogger(SecureBankService.class);
    private volatile boolean _isRunning;
    private final List<Account> _accounts = new ArrayList<Account>();
    private Map<Long, Account> _accountsById = new HashMap<Long, Account>();

    public void start() throws Exception {
        this._isRunning = true;
        log.info("Bank service started");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void dispose() {
        log.info("Stopping bank service...");
        this._isRunning = false;
        List<Account> list = this._accounts;
        synchronized (list) {
            this._accountsById.clear();
            this._accounts.clear();
        }
        log.info("Bank service stopped");
    }

    protected void assertServiceState() {
        if (!this._isRunning) {
            throw new IllegalStateException("This bank service is not running");
        }
    }

    public int getAccountCount() {
        return this._accounts.size();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @RequiresPermissions(value={"bankAccount:create"})
    public long createNewAccount(String anOwnerName) {
        this.assertServiceState();
        log.info("Creating new account for " + anOwnerName);
        List<Account> list = this._accounts;
        synchronized (list) {
            Account account = new Account(anOwnerName);
            account.setCreatedBy(this.getCurrentUsername());
            this._accounts.add(account);
            this._accountsById.put(account.getId(), account);
            log.debug("Created new account: " + account);
            return account.getId();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public long[] searchAccountIdsByOwner(String anOwnerName) {
        this.assertServiceState();
        log.info("Searching existing accounts for " + anOwnerName);
        ArrayList<Account> matchAccounts = new ArrayList<Account>();
        List<Account> list = this._accounts;
        synchronized (list) {
            for (Account a : this._accounts) {
                if (!a.getOwnerName().toLowerCase().contains(anOwnerName.toLowerCase())) continue;
                matchAccounts.add(a);
            }
        }
        long[] accountIds = new long[matchAccounts.size()];
        int index = 0;
        for (Account a : matchAccounts) {
            accountIds[index++] = a.getId();
        }
        log.debug("Found " + accountIds.length + " account(s) matching the name " + anOwnerName);
        return accountIds;
    }

    @RequiresPermissions(value={"bankAccount:read"})
    public String getOwnerOf(long anAccountId) throws AccountNotFoundException {
        this.assertServiceState();
        log.info("Getting owner of account " + anAccountId);
        Account a = this.safellyRetrieveAccountForId(anAccountId);
        return a.getOwnerName();
    }

    @RequiresPermissions(value={"bankAccount:read"})
    public double getBalanceOf(long anAccountId) throws AccountNotFoundException {
        this.assertServiceState();
        log.info("Getting balance of account " + anAccountId);
        Account a = this.safellyRetrieveAccountForId(anAccountId);
        return a.getBalance();
    }

    @RequiresPermissions(value={"bankAccount:operate"})
    public double depositInto(long anAccountId, double anAmount) throws AccountNotFoundException, InactiveAccountException {
        this.assertServiceState();
        log.info("Making deposit of " + anAmount + " into account " + anAccountId);
        try {
            Account a = this.safellyRetrieveAccountForId(anAccountId);
            AccountTransaction tx = AccountTransaction.createDepositTx(anAccountId, anAmount);
            tx.setCreatedBy(this.getCurrentUsername());
            log.debug("Created a new transaction " + tx);
            a.applyTransaction(tx);
            log.debug("New balance of account " + a.getId() + " after deposit is " + a.getBalance());
            return a.getBalance();
        }
        catch (NotEnoughFundsException nefe) {
            throw new IllegalStateException("Should never happen", nefe);
        }
    }

    @RequiresPermissions(value={"bankAccount:operate"})
    public double withdrawFrom(long anAccountId, double anAmount) throws AccountNotFoundException, NotEnoughFundsException, InactiveAccountException {
        this.assertServiceState();
        log.info("Making withdrawal of " + anAmount + " from account " + anAccountId);
        Account a = this.safellyRetrieveAccountForId(anAccountId);
        AccountTransaction tx = AccountTransaction.createWithdrawalTx(anAccountId, anAmount);
        tx.setCreatedBy(this.getCurrentUsername());
        log.debug("Created a new transaction " + tx);
        a.applyTransaction(tx);
        log.debug("New balance of account " + a.getId() + " after withdrawal is " + a.getBalance());
        return a.getBalance();
    }

    @RequiresPermissions(value={"bankAccount:read"})
    public BankService.TxLog[] getTxHistoryFor(long anAccountId) throws AccountNotFoundException {
        this.assertServiceState();
        log.info("Getting transactions of account " + anAccountId);
        Account a = this.safellyRetrieveAccountForId(anAccountId);
        BankService.TxLog[] txs = new BankService.TxLog[a.getTransactions().size()];
        int index = 0;
        for (AccountTransaction tx : a.getTransactions()) {
            log.debug("Retrieved transaction " + tx);
            if (AccountTransaction.TransactionType.DEPOSIT == tx.getType()) {
                txs[index++] = new BankService.TxLog(tx.getCreationDate(), tx.getAmount(), tx.getCreatedBy());
                continue;
            }
            txs[index++] = new BankService.TxLog(tx.getCreationDate(), -1.0 * tx.getAmount(), tx.getCreatedBy());
        }
        return txs;
    }

    @RequiresPermissions(value={"bankAccount:close"})
    public double closeAccount(long anAccountId) throws AccountNotFoundException, InactiveAccountException {
        this.assertServiceState();
        log.info("Closing account " + anAccountId);
        Account a = this.safellyRetrieveAccountForId(anAccountId);
        if (!a.isActive()) {
            throw new InactiveAccountException("The account " + anAccountId + " is already closed");
        }
        try {
            AccountTransaction tx = AccountTransaction.createWithdrawalTx(a.getId(), a.getBalance());
            tx.setCreatedBy(this.getCurrentUsername());
            log.debug("Created a new transaction " + tx);
            a.applyTransaction(tx);
            a.setActive(false);
            log.debug("Account " + a.getId() + " is now closed and an amount of " + tx.getAmount() + " is given to the owner");
            return tx.getAmount();
        }
        catch (NotEnoughFundsException nefe) {
            throw new IllegalStateException("Should never happen", nefe);
        }
    }

    @RequiresPermissions(value={"bankAccount:read"})
    public boolean isAccountActive(long anAccountId) throws AccountNotFoundException {
        this.assertServiceState();
        log.info("Getting active status of account " + anAccountId);
        Account a = this.safellyRetrieveAccountForId(anAccountId);
        return a.isActive();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected Account safellyRetrieveAccountForId(long anAccountId) throws AccountNotFoundException {
        Account account = null;
        List<Account> list = this._accounts;
        synchronized (list) {
            account = this._accountsById.get(anAccountId);
        }
        if (account == null) {
            throw new AccountNotFoundException("No account found for the id " + anAccountId);
        }
        log.info("Retrieved account " + account);
        return account;
    }

    protected String getCurrentUsername() {
        Subject subject = SecurityUtils.getSubject();
        if (subject == null || subject.getPrincipal() == null || !subject.isAuthenticated()) {
            throw new IllegalStateException("Unable to retrieve the current authenticated subject");
        }
        return SecurityUtils.getSubject().getPrincipal().toString();
    }
}

