/*
 * Decompiled with CFR 0.152.
 */
package org.graylog2.indexer;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Lists;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import javax.inject.Inject;
import org.elasticsearch.action.admin.indices.stats.IndexStats;
import org.elasticsearch.indices.InvalidAliasNameException;
import org.graylog2.configuration.ElasticsearchConfiguration;
import org.graylog2.indexer.NoTargetIndexException;
import org.graylog2.indexer.SetIndexReadOnlyJob;
import org.graylog2.indexer.indices.Indices;
import org.graylog2.indexer.ranges.CreateNewSingleIndexRangeJob;
import org.graylog2.indexer.ranges.RebuildIndexRangesJob;
import org.graylog2.shared.system.activities.Activity;
import org.graylog2.shared.system.activities.ActivityWriter;
import org.graylog2.system.jobs.SystemJobConcurrencyException;
import org.graylog2.system.jobs.SystemJobManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Deflector {
    private static final Logger LOG = LoggerFactory.getLogger(Deflector.class);
    public static final String DEFLECTOR_SUFFIX = "deflector";
    public static final String SEPARATOR = "_";
    private final SystemJobManager systemJobManager;
    private final ActivityWriter activityWriter;
    private final RebuildIndexRangesJob.Factory rebuildIndexRangesJobFactory;
    private final CreateNewSingleIndexRangeJob.Factory createNewSingleIndexRangeJobFactory;
    private final String indexPrefix;
    private final String deflectorName;
    private final Indices indices;
    private final ElasticsearchConfiguration configuration;
    private final SetIndexReadOnlyJob.Factory indexReadOnlyJobFactory;

    @Inject
    public Deflector(SystemJobManager systemJobManager, ElasticsearchConfiguration configuration, ActivityWriter activityWriter, RebuildIndexRangesJob.Factory rebuildIndexRangesJobFactory, SetIndexReadOnlyJob.Factory indexReadOnlyJobFactory, CreateNewSingleIndexRangeJob.Factory createNewSingleIndexRangeJobFactory, Indices indices) {
        this.configuration = configuration;
        this.indexPrefix = configuration.getIndexPrefix();
        this.systemJobManager = systemJobManager;
        this.activityWriter = activityWriter;
        this.rebuildIndexRangesJobFactory = rebuildIndexRangesJobFactory;
        this.indexReadOnlyJobFactory = indexReadOnlyJobFactory;
        this.createNewSingleIndexRangeJobFactory = createNewSingleIndexRangeJobFactory;
        this.deflectorName = Deflector.buildName(configuration.getIndexPrefix());
        this.indices = indices;
    }

    public boolean isUp() {
        return this.indices.aliasExists(this.getName());
    }

    public void setUp() {
        if (this.isUp()) {
            LOG.info("Found deflector alias <{}>. Using it.", (Object)this.getName());
        } else {
            LOG.info("Did not find an deflector alias. Setting one up now.");
            try {
                try {
                    String currentTarget = this.getNewestTargetName();
                    LOG.info("Pointing to already existing index target <{}>", (Object)currentTarget);
                    this.pointTo(currentTarget);
                }
                catch (NoTargetIndexException ex) {
                    String msg = "There is no index target to point to. Creating one now.";
                    LOG.info("There is no index target to point to. Creating one now.");
                    this.activityWriter.write(new Activity("There is no index target to point to. Creating one now.", Deflector.class));
                    this.cycle();
                }
            }
            catch (InvalidAliasNameException e) {
                LOG.error("Seems like there already is an index called [{}]", (Object)this.getName());
            }
        }
    }

    public void cycle() {
        int oldTargetNumber;
        LOG.info("Cycling deflector to next index now.");
        try {
            oldTargetNumber = this.getNewestTargetNumber();
        }
        catch (NoTargetIndexException ex) {
            oldTargetNumber = -1;
        }
        int newTargetNumber = oldTargetNumber + 1;
        String newTarget = Deflector.buildIndexName(this.indexPrefix, newTargetNumber);
        String oldTarget = Deflector.buildIndexName(this.indexPrefix, oldTargetNumber);
        if (oldTargetNumber == -1) {
            LOG.info("Cycling from <none> to <{}>", (Object)newTarget);
        } else {
            LOG.info("Cycling from <{}> to <{}>", (Object)oldTarget, (Object)newTarget);
        }
        LOG.info("Creating index target <{}>...", (Object)newTarget);
        if (!this.indices.create(newTarget)) {
            LOG.error("Could not properly create new target <{}>", (Object)newTarget);
        }
        LOG.info("Done!");
        LOG.info("Pointing deflector to new target index....");
        Activity activity = new Activity(Deflector.class);
        if (oldTargetNumber == -1) {
            this.pointTo(newTarget);
            activity.setMessage("Cycled deflector from <none> to <" + newTarget + ">");
        } else {
            this.pointTo(newTarget, oldTarget);
            SetIndexReadOnlyJob makeReadOnlyJob = this.indexReadOnlyJobFactory.create(oldTarget);
            try {
                this.systemJobManager.submitWithDelay(makeReadOnlyJob, 30L, TimeUnit.SECONDS);
            }
            catch (SystemJobConcurrencyException e) {
                LOG.error("Cannot set index <" + oldTarget + "> to read only. It won't be optimized.", (Throwable)e);
            }
            activity.setMessage("Cycled deflector from <" + oldTarget + "> to <" + newTarget + ">");
        }
        if (this.configuration.isDisableIndexRangeCalculation() && oldTargetNumber != -1) {
            this.addSingleIndexRanges(oldTarget);
            this.addSingleIndexRanges(newTarget);
        } else {
            this.updateIndexRanges();
        }
        LOG.info("Done!");
        this.activityWriter.write(activity);
    }

    public int getNewestTargetNumber() throws NoTargetIndexException {
        Map<String, IndexStats> indices = this.indices.getAll();
        if (indices.isEmpty()) {
            throw new NoTargetIndexException();
        }
        ArrayList indexNumbers = Lists.newArrayListWithExpectedSize((int)indices.size());
        for (String indexName : indices.keySet()) {
            if (!this.isGraylog2Index(indexName)) continue;
            try {
                indexNumbers.add(Deflector.extractIndexNumber(indexName));
            }
            catch (NumberFormatException ex) {}
        }
        if (indexNumbers.isEmpty()) {
            throw new NoTargetIndexException();
        }
        return (Integer)Collections.max(indexNumbers);
    }

    public String[] getAllDeflectorIndexNames() {
        ArrayList result = Lists.newArrayListWithExpectedSize((int)this.indices.getAll().size());
        for (String indexName : this.indices.getAll().keySet()) {
            if (!this.isGraylog2Index(indexName)) continue;
            result.add(indexName);
        }
        return result.toArray(new String[result.size()]);
    }

    public Map<String, IndexStats> getAllDeflectorIndices() {
        ImmutableMap.Builder result = ImmutableMap.builder();
        if (this.indices != null) {
            for (Map.Entry<String, IndexStats> e : this.indices.getAll().entrySet()) {
                String name = e.getKey();
                if (!this.isGraylog2Index(name)) continue;
                result.put((Object)name, (Object)e.getValue());
            }
        }
        return result.build();
    }

    public String getNewestTargetName() throws NoTargetIndexException {
        return Deflector.buildIndexName(this.indexPrefix, this.getNewestTargetNumber());
    }

    public static String buildIndexName(String prefix, int number) {
        return prefix + SEPARATOR + number;
    }

    public static String buildName(String prefix) {
        return prefix + SEPARATOR + DEFLECTOR_SUFFIX;
    }

    public static int extractIndexNumber(String indexName) throws NumberFormatException {
        String[] parts = indexName.split(SEPARATOR);
        try {
            return Integer.parseInt(parts[parts.length - 1]);
        }
        catch (Exception e) {
            LOG.debug("Could not extract index number from index <" + indexName + ">.", (Throwable)e);
            throw new NumberFormatException();
        }
    }

    public void pointTo(String newIndex, String oldIndex) {
        this.indices.cycleAlias(this.getName(), newIndex, oldIndex);
    }

    public void pointTo(String newIndex) {
        this.indices.cycleAlias(this.getName(), newIndex);
    }

    private void updateIndexRanges() {
        try {
            this.systemJobManager.submit(this.rebuildIndexRangesJobFactory.create(this));
        }
        catch (SystemJobConcurrencyException e) {
            String msg = "Could not re-calculate index ranges after cycling deflector: Maximum concurrency of job is reached.";
            this.activityWriter.write(new Activity("Could not re-calculate index ranges after cycling deflector: Maximum concurrency of job is reached.", Deflector.class));
            LOG.error("Could not re-calculate index ranges after cycling deflector: Maximum concurrency of job is reached.", (Throwable)e);
        }
    }

    private void addSingleIndexRanges(String indexName) {
        try {
            this.systemJobManager.submit(this.createNewSingleIndexRangeJobFactory.create(this, indexName));
        }
        catch (SystemJobConcurrencyException e) {
            String msg = "Could not calculate index ranges for index " + indexName + " after cycling deflector: Maximum concurrency of job is reached.";
            this.activityWriter.write(new Activity(msg, Deflector.class));
            LOG.error(msg, (Throwable)e);
        }
    }

    public String getCurrentActualTargetIndex() {
        return this.indices.aliasTarget(this.getName());
    }

    public String getName() {
        return this.deflectorName;
    }

    public String getDeflectorWildcard() {
        return this.indexPrefix + SEPARATOR + "*";
    }

    public boolean isDeflectorAlias(String indexName) {
        return this.getName().equals(indexName);
    }

    public boolean isGraylog2Index(String indexName) {
        return !this.isDeflectorAlias(indexName) && indexName.startsWith(this.indexPrefix + SEPARATOR);
    }
}

