/*
 * Decompiled with CFR 0.152.
 */
package org.graylog.security.authservice.ldap;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.primitives.Ints;
import com.unboundid.ldap.sdk.Attribute;
import com.unboundid.ldap.sdk.BindRequest;
import com.unboundid.ldap.sdk.BindResult;
import com.unboundid.ldap.sdk.Entry;
import com.unboundid.ldap.sdk.ExtendedResult;
import com.unboundid.ldap.sdk.FailoverServerSet;
import com.unboundid.ldap.sdk.Filter;
import com.unboundid.ldap.sdk.LDAPBindException;
import com.unboundid.ldap.sdk.LDAPConnection;
import com.unboundid.ldap.sdk.LDAPConnectionOptions;
import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.ldap.sdk.LDAPResult;
import com.unboundid.ldap.sdk.ResultCode;
import com.unboundid.ldap.sdk.SearchRequest;
import com.unboundid.ldap.sdk.SearchResult;
import com.unboundid.ldap.sdk.SearchScope;
import com.unboundid.ldap.sdk.SimpleBindRequest;
import com.unboundid.ldap.sdk.extensions.StartTLSExtendedRequest;
import com.unboundid.util.Base64;
import com.unboundid.util.LDAPTestUtils;
import com.unboundid.util.StaticUtils;
import com.unboundid.util.ssl.SSLUtil;
import java.security.GeneralSecurityException;
import java.text.MessageFormat;
import java.util.Arrays;
import java.util.Locale;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import javax.inject.Inject;
import javax.inject.Named;
import javax.inject.Singleton;
import javax.net.ssl.SSLSocketFactory;
import javax.net.ssl.TrustManager;
import org.apache.commons.lang3.StringUtils;
import org.graylog.security.authservice.ldap.ADUserAccountControl;
import org.graylog.security.authservice.ldap.LDAPConnectorConfig;
import org.graylog.security.authservice.ldap.LDAPEntry;
import org.graylog.security.authservice.ldap.LDAPTransportSecurity;
import org.graylog.security.authservice.ldap.LDAPUser;
import org.graylog.security.authservice.ldap.UnboundLDAPConfig;
import org.graylog2.configuration.TLSProtocolsConfiguration;
import org.graylog2.security.TrustAllX509TrustManager;
import org.graylog2.security.TrustManagerProvider;
import org.graylog2.security.encryption.EncryptedValue;
import org.graylog2.security.encryption.EncryptedValueService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class UnboundLDAPConnector {
    private static final Logger LOG = LoggerFactory.getLogger(UnboundLDAPConnector.class);
    private static final String OBJECT_CLASS_ATTRIBUTE = "objectClass";
    private final int connectionTimeout;
    private final TLSProtocolsConfiguration tlsConfiguration;
    private final TrustManagerProvider trustManagerProvider;
    private final EncryptedValueService encryptedValueService;
    private final int requestTimeoutSeconds;

    @Inject
    public UnboundLDAPConnector(@Named(value="ldap_connection_timeout") int connectionTimeout, TLSProtocolsConfiguration tlsConfiguration, TrustManagerProvider trustManagerProvider, EncryptedValueService encryptedValueService) {
        this.connectionTimeout = connectionTimeout;
        this.tlsConfiguration = tlsConfiguration;
        this.trustManagerProvider = trustManagerProvider;
        this.encryptedValueService = encryptedValueService;
        this.requestTimeoutSeconds = 60;
    }

    public LDAPConnection connect(LDAPConnectorConfig ldapConfig) throws GeneralSecurityException, LDAPException {
        if (ldapConfig.serverList().isEmpty()) {
            LOG.warn("Cannot connect with empty server list");
            return null;
        }
        String[] addresses = (String[])ldapConfig.serverList().stream().map(LDAPConnectorConfig.LDAPServer::hostname).toArray(String[]::new);
        int[] ports = ldapConfig.serverList().stream().mapToInt(LDAPConnectorConfig.LDAPServer::port).toArray();
        LDAPConnectionOptions connectionOptions = new LDAPConnectionOptions();
        connectionOptions.setUseReuseAddress(true);
        connectionOptions.setConnectTimeoutMillis(this.connectionTimeout);
        StartTLSExtendedRequest startTLSRequest = null;
        SSLSocketFactory socketFactory = null;
        if (ldapConfig.transportSecurity() != LDAPTransportSecurity.NONE) {
            SSLUtil.setEnabledSSLProtocols(this.tlsConfiguration.getEnabledTlsProtocols());
            SSLUtil sslUtil = ldapConfig.verifyCertificates() ? new SSLUtil(this.trustManagerProvider.create(Arrays.asList(addresses))) : new SSLUtil((TrustManager)new TrustAllX509TrustManager());
            if (ldapConfig.transportSecurity() == LDAPTransportSecurity.START_TLS) {
                startTLSRequest = new StartTLSExtendedRequest(sslUtil.createSSLContext());
            } else if (ldapConfig.transportSecurity() == LDAPTransportSecurity.TLS) {
                socketFactory = sslUtil.createSSLSocketFactory();
            }
        }
        FailoverServerSet serverSet = new FailoverServerSet(addresses, ports, socketFactory, connectionOptions, null, null);
        LDAPConnection connection = serverSet.getConnection();
        if (startTLSRequest != null) {
            ExtendedResult startTLSResult = connection.processExtendedOperation(startTLSRequest);
            LDAPTestUtils.assertResultCodeEquals((LDAPResult)startTLSResult, (ResultCode[])new ResultCode[]{ResultCode.SUCCESS});
        }
        if (ldapConfig.systemUsername().isPresent()) {
            if (ldapConfig.systemPassword().isSet()) {
                String systemPassword = this.encryptedValueService.decrypt(ldapConfig.systemPassword());
                SimpleBindRequest bindRequest = new SimpleBindRequest(ldapConfig.systemUsername().get(), systemPassword);
                connection.bind((BindRequest)bindRequest);
            } else {
                LOG.warn("System username has been set to <{}> but no system password has been set. Skipping bind request.", (Object)ldapConfig.systemUsername().get());
            }
        }
        return connection;
    }

    public ImmutableList<LDAPEntry> search(LDAPConnection connection, String searchBase, Filter filter, String uniqueIdAttribute, Set<String> attributes) throws LDAPException {
        SearchResult searchResult;
        ImmutableSet allAttributes = ImmutableSet.builder().add((Object)OBJECT_CLASS_ATTRIBUTE).addAll(attributes).build();
        SearchRequest searchRequest = new SearchRequest(searchBase, SearchScope.SUB, filter, (String[])allAttributes.toArray((Object[])new String[0]));
        searchRequest.setTimeLimitSeconds(this.requestTimeoutSeconds);
        if (LOG.isTraceEnabled()) {
            LOG.trace("Search LDAP for <{}> using search base <{}>", (Object)filter.toNormalizedString(), (Object)searchBase);
        }
        if ((searchResult = connection.search(searchRequest)).getSearchEntries().isEmpty()) {
            LOG.trace("No LDAP entry found for filter <{}>", (Object)filter.toNormalizedString());
            return ImmutableList.of();
        }
        return (ImmutableList)searchResult.getSearchEntries().stream().map(entry -> this.createLDAPEntry((Entry)entry, uniqueIdAttribute)).collect(ImmutableList.toImmutableList());
    }

    public Optional<LDAPUser> searchUserByPrincipal(LDAPConnection connection, UnboundLDAPConfig config, String principal) throws LDAPException {
        String filterString = new MessageFormat(config.userSearchPattern(), Locale.ENGLISH).format(new Object[]{Filter.encodeValue((String)principal)});
        return this.searchUser(connection, config, Filter.create((String)filterString));
    }

    public Optional<LDAPUser> searchUserByUniqueId(LDAPConnection connection, UnboundLDAPConfig config, byte[] uniqueId) throws LDAPException {
        return this.searchUser(connection, config, Filter.createEqualityFilter((String)config.userUniqueIdAttribute(), (byte[])uniqueId));
    }

    private Optional<LDAPUser> searchUser(LDAPConnection connection, UnboundLDAPConfig config, Filter filter) throws LDAPException {
        ImmutableSet allAttributes = ImmutableSet.builder().add((Object)"userPrincipalName").add((Object)"userAccountControl").add((Object)"mail").add((Object)"rfc822Mailbox").add((Object)config.userUniqueIdAttribute()).add((Object)config.userNameAttribute()).add((Object)config.userFullNameAttribute()).build();
        ImmutableList<LDAPEntry> result = this.search(connection, config.userSearchBase(), filter, config.userUniqueIdAttribute(), (Set<String>)allAttributes);
        if (result.size() > 1) {
            LOG.warn("Found more than one user for <{}> in search base <{}> - Using the first one", (Object)filter.toString(), (Object)config.userSearchBase());
        }
        return result.stream().findFirst().map(entry -> this.createLDAPUser(config, (LDAPEntry)entry));
    }

    public boolean authenticate(LDAPConnection connection, String bindDn, EncryptedValue password) throws LDAPException {
        Preconditions.checkArgument((!Strings.isNullOrEmpty((String)bindDn) ? 1 : 0) != 0, (Object)"Binding with empty principal is forbidden.");
        Preconditions.checkArgument((password != null ? 1 : 0) != 0, (Object)"Binding with null credentials is forbidden.");
        Preconditions.checkArgument((boolean)password.isSet(), (Object)"Binding with empty credentials is forbidden.");
        SimpleBindRequest bindRequest = new SimpleBindRequest(bindDn, this.encryptedValueService.decrypt(password));
        LOG.trace("Re-binding with DN <{}> using password", (Object)bindDn);
        try {
            BindResult bind = connection.bind((BindRequest)bindRequest);
            if (!bind.getResultCode().equals((Object)ResultCode.SUCCESS)) {
                LOG.trace("Re-binding DN <{}> failed", (Object)bindDn);
                throw new RuntimeException(bind.toString());
            }
            boolean authenticated = connection.getLastBindRequest().equals(bindRequest);
            LOG.trace("Binding DN <{}> did not throw, connection authenticated: {}", (Object)bindDn, (Object)authenticated);
            return authenticated;
        }
        catch (LDAPBindException e) {
            LOG.trace("Re-binding DN <{}> failed", (Object)bindDn);
            return false;
        }
    }

    public LDAPEntry createLDAPEntry(Entry entry, String uniqueIdAttribute) {
        Objects.requireNonNull(entry, "entry cannot be null");
        Preconditions.checkArgument((!StringUtils.isBlank((CharSequence)uniqueIdAttribute) ? 1 : 0) != 0, (Object)"uniqueIdAttribute cannot be blank");
        LDAPEntry.Builder ldapEntryBuilder = LDAPEntry.builder();
        ldapEntryBuilder.dn(entry.getDN());
        byte[] uniqueId = Objects.requireNonNull(entry.getAttributeValueBytes(uniqueIdAttribute), uniqueIdAttribute + " attribute cannot be null");
        ldapEntryBuilder.base64UniqueId(Base64.encode((byte[])uniqueId));
        if (entry.getObjectClassValues() != null) {
            ldapEntryBuilder.objectClasses(Arrays.asList(entry.getObjectClassValues()));
        }
        for (Attribute attribute : entry.getAttributes()) {
            if (OBJECT_CLASS_ATTRIBUTE.equalsIgnoreCase(attribute.getBaseName())) continue;
            if (attribute.needsBase64Encoding()) {
                for (Object value : attribute.getValueByteArrays()) {
                    if (StaticUtils.isValidUTF8((byte[])value)) {
                        ldapEntryBuilder.addAttribute(attribute.getBaseName(), StaticUtils.toUTF8String((byte[])value));
                        continue;
                    }
                    ldapEntryBuilder.addAttribute(attribute.getBaseName(), Base64.encode((byte[])value));
                }
                continue;
            }
            String[] stringArray = attribute.getValues();
            int n = stringArray.length;
            for (int i = 0; i < n; ++i) {
                Object value;
                value = stringArray[i];
                ldapEntryBuilder.addAttribute(attribute.getBaseName(), (String)value);
            }
        }
        return ldapEntryBuilder.build();
    }

    public LDAPUser createLDAPUser(UnboundLDAPConfig config, Entry entry) {
        return this.createLDAPUser(config, this.createLDAPEntry(entry, config.userUniqueIdAttribute()));
    }

    public LDAPUser createLDAPUser(UnboundLDAPConfig config, LDAPEntry ldapEntry) {
        String username = ldapEntry.nonBlankAttribute(config.userNameAttribute());
        return LDAPUser.builder().base64UniqueId(ldapEntry.base64UniqueId()).accountIsEnabled(this.findAccountIsEnabled(ldapEntry)).username(username).fullName(ldapEntry.firstAttributeValue(config.userFullNameAttribute()).orElse(username)).email(ldapEntry.firstAttributeValue("mail").orElse(ldapEntry.firstAttributeValue("rfc822Mailbox").orElse("unknown@unknown.invalid"))).entry(ldapEntry).build();
    }

    private boolean findAccountIsEnabled(LDAPEntry ldapEntry) {
        Optional<String> control = ldapEntry.firstAttributeValue("userAccountControl");
        if (!control.isPresent()) {
            return true;
        }
        Integer userAccountControl = Ints.tryParse((String)control.get());
        if (userAccountControl == null) {
            LOG.warn("Ignoring non-parseable userAccountControl value");
            return true;
        }
        return !ADUserAccountControl.create(userAccountControl).accountIsDisabled();
    }
}

