/*
 * Decompiled with CFR 0.152.
 */
package org.keycloak.services.managers;

import java.security.Key;
import java.security.MessageDigest;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import java.util.function.Supplier;
import javax.crypto.SecretKey;
import org.jboss.logging.Logger;
import org.keycloak.common.util.Base64Url;
import org.keycloak.common.util.Time;
import org.keycloak.events.EventBuilder;
import org.keycloak.jose.jwe.JWEException;
import org.keycloak.models.AuthenticatedClientSessionModel;
import org.keycloak.models.CodeToTokenStoreProvider;
import org.keycloak.models.KeycloakSession;
import org.keycloak.models.RealmModel;
import org.keycloak.models.UserSessionModel;
import org.keycloak.models.utils.KeycloakModelUtils;
import org.keycloak.representations.CodeJWT;
import org.keycloak.representations.JsonWebToken;
import org.keycloak.services.managers.AuthenticationSessionManager;
import org.keycloak.services.managers.UserSessionCrossDCManager;
import org.keycloak.sessions.AuthenticationSessionModel;
import org.keycloak.sessions.CommonClientSessionModel;
import org.keycloak.util.TokenUtil;

class CodeGenerateUtil {
    private static final Logger logger = Logger.getLogger(CodeGenerateUtil.class);
    private static final String ACTIVE_CODE = "active_code";
    private static final Map<Class<? extends CommonClientSessionModel>, Supplier<ClientSessionParser>> PARSERS = new HashMap<Class<? extends CommonClientSessionModel>, Supplier<ClientSessionParser>>();

    CodeGenerateUtil() {
    }

    static <CS extends CommonClientSessionModel> ClientSessionParser<CS> getParser(Class<CS> clientSessionClass) {
        for (Class<? extends CommonClientSessionModel> c : PARSERS.keySet()) {
            if (!c.isAssignableFrom(clientSessionClass)) continue;
            return PARSERS.get(c).get();
        }
        return null;
    }

    static {
        PARSERS.put(AuthenticationSessionModel.class, () -> new AuthenticationSessionModelParser());
        PARSERS.put(AuthenticatedClientSessionModel.class, () -> new AuthenticatedClientSessionModelParser());
    }

    private static class AuthenticatedClientSessionModelParser
    implements ClientSessionParser<AuthenticatedClientSessionModel> {
        private CodeJWT codeJWT;

        private AuthenticatedClientSessionModelParser() {
        }

        @Override
        public AuthenticatedClientSessionModel parseSession(String code, KeycloakSession session, RealmModel realm, EventBuilder event) {
            SecretKey aesKey = session.keys().getActiveAesKey(realm).getSecretKey();
            SecretKey hmacKey = session.keys().getActiveHmacKey(realm).getSecretKey();
            try {
                this.codeJWT = (CodeJWT)TokenUtil.jweDirectVerifyAndDecode((Key)aesKey, (Key)hmacKey, (String)code, CodeJWT.class);
            }
            catch (JWEException jweException) {
                logger.error((Object)"Exception during JWE Verification or decode", (Throwable)jweException);
                return null;
            }
            event.detail("code_id", this.codeJWT.getUserSessionId());
            event.session(this.codeJWT.getUserSessionId());
            UserSessionModel userSession = new UserSessionCrossDCManager(session).getUserSessionWithClient(realm, this.codeJWT.getUserSessionId(), this.codeJWT.getIssuedFor());
            if (userSession == null && (userSession = session.sessions().getUserSession(realm, this.codeJWT.getUserSessionId())) == null) {
                return null;
            }
            return (AuthenticatedClientSessionModel)userSession.getAuthenticatedClientSessions().get(this.codeJWT.getIssuedFor());
        }

        @Override
        public String retrieveCode(KeycloakSession session, AuthenticatedClientSessionModel clientSession) {
            String actionId = KeycloakModelUtils.generateId();
            CodeJWT codeJWT = new CodeJWT();
            codeJWT.id(actionId);
            codeJWT.issuedFor(clientSession.getClient().getId());
            codeJWT.userSessionId(clientSession.getUserSession().getId());
            RealmModel realm = clientSession.getRealm();
            int issuedAt = Time.currentTime();
            codeJWT.issuedAt(issuedAt);
            codeJWT.expiration(issuedAt + realm.getAccessCodeLifespan());
            SecretKey aesKey = session.keys().getActiveAesKey(realm).getSecretKey();
            SecretKey hmacKey = session.keys().getActiveHmacKey(realm).getSecretKey();
            if (logger.isTraceEnabled()) {
                logger.tracef("Using AES key of length '%d' bytes and HMAC key of length '%d' bytes . Client: '%s', User Session: '%s'", new Object[]{aesKey.getEncoded().length, hmacKey.getEncoded().length, clientSession.getClient().getClientId(), clientSession.getUserSession().getId()});
            }
            try {
                return TokenUtil.jweDirectEncode((Key)aesKey, (Key)hmacKey, (JsonWebToken)codeJWT);
            }
            catch (JWEException jweEx) {
                throw new RuntimeException(jweEx);
            }
        }

        @Override
        public boolean verifyCode(KeycloakSession session, String code, AuthenticatedClientSessionModel clientSession) {
            if (this.codeJWT == null) {
                throw new IllegalStateException("Illegal use. codeJWT not yet set");
            }
            UUID codeId = UUID.fromString(this.codeJWT.getId());
            CodeToTokenStoreProvider singleUseCache = (CodeToTokenStoreProvider)session.getProvider(CodeToTokenStoreProvider.class);
            if (singleUseCache.putIfAbsent(codeId)) {
                if (logger.isTraceEnabled()) {
                    logger.tracef("Added code '%s' to single-use cache. User session: %s, client: %s", (Object)this.codeJWT.getId(), (Object)this.codeJWT.getUserSessionId(), (Object)this.codeJWT.getIssuedFor());
                }
                return true;
            }
            logger.warnf("Code '%s' already used for userSession '%s' and client '%s'.", (Object)this.codeJWT.getId(), (Object)this.codeJWT.getUserSessionId(), (Object)this.codeJWT.getIssuedFor());
            return false;
        }

        @Override
        public void removeExpiredSession(KeycloakSession session, AuthenticatedClientSessionModel clientSession) {
            throw new IllegalStateException("Not yet implemented");
        }

        @Override
        public boolean isExpired(KeycloakSession session, String code, AuthenticatedClientSessionModel clientSession) {
            return !this.codeJWT.isActive();
        }
    }

    private static class AuthenticationSessionModelParser
    implements ClientSessionParser<AuthenticationSessionModel> {
        private AuthenticationSessionModelParser() {
        }

        @Override
        public AuthenticationSessionModel parseSession(String code, KeycloakSession session, RealmModel realm, EventBuilder event) {
            return new AuthenticationSessionManager(session).getCurrentAuthenticationSession(realm);
        }

        @Override
        public String retrieveCode(KeycloakSession session, AuthenticationSessionModel authSession) {
            String nextCode = authSession.getAuthNote(CodeGenerateUtil.ACTIVE_CODE);
            if (nextCode == null) {
                String actionId = Base64Url.encode((byte[])KeycloakModelUtils.generateSecret());
                authSession.setAuthNote(CodeGenerateUtil.ACTIVE_CODE, actionId);
                nextCode = actionId;
            } else {
                logger.debug((Object)"Code already generated for authentication session, using same code");
            }
            return nextCode;
        }

        @Override
        public void removeExpiredSession(KeycloakSession session, AuthenticationSessionModel clientSession) {
            new AuthenticationSessionManager(session).removeAuthenticationSession(clientSession.getRealm(), clientSession, true);
        }

        @Override
        public boolean verifyCode(KeycloakSession session, String code, AuthenticationSessionModel authSession) {
            String activeCode = authSession.getAuthNote(CodeGenerateUtil.ACTIVE_CODE);
            if (activeCode == null) {
                logger.debug((Object)"Active code not found in authentication session");
                return false;
            }
            authSession.removeAuthNote(CodeGenerateUtil.ACTIVE_CODE);
            return MessageDigest.isEqual(code.getBytes(), activeCode.getBytes());
        }

        @Override
        public boolean isExpired(KeycloakSession session, String code, AuthenticationSessionModel clientSession) {
            return false;
        }
    }

    static interface ClientSessionParser<CS extends CommonClientSessionModel> {
        public CS parseSession(String var1, KeycloakSession var2, RealmModel var3, EventBuilder var4);

        public String retrieveCode(KeycloakSession var1, CS var2);

        public void removeExpiredSession(KeycloakSession var1, CS var2);

        public boolean verifyCode(KeycloakSession var1, String var2, CS var3);

        public boolean isExpired(KeycloakSession var1, String var2, CS var3);
    }
}

