// Metawidget
//
// This file is dual licensed under both the LGPL
// (http://www.gnu.org/licenses/lgpl-2.1.html) and the EPL
// (http://www.eclipse.org/org/documents/epl-v10.php). As a
// recipient of Metawidget, you may choose to receive it under either
// the LGPL or the EPL.
//
// Commercial licenses are also available. See http://metawidget.org
// for details.

package org.metawidget.layout.iface;

import java.util.Map;

import org.metawidget.iface.Immutable;

/**
 * Interface for Layouts.
 * <p>
 * Layouts must be immutable (or, at least, appear that way to clients. They can have caches or
 * configuration settings internally, as long as they are threadsafe). If they need to store state,
 * they should use the Metawidget passed to each method.
 * <p>
 * Don't be put off by having so many parameterized types! Metawidget needs them so that it can
 * present a consistent API across many architectures. However this complication doesn't impact your
 * own custom plugins because you're able to substitute concrete, platform-specific values for each
 * parameter (i.e. <tt>Control</tt>, <tt>Composite</tt>, <tt>SwtMetawidget</tt> ).
 * 
 * @param <W>
 *            base class of widgets that this Layout lays out
 * @param <C>
 *            base class of container widgets. Many UI frameworks make a
 *            distinction between 'container widgets' (ie. Panels) and widgets
 *            that cannot contain child controls (ie. TextBox). For frameworks
 *            that don't make such a distinction, W and C can be the same
 * @param <M>
 *            Metawidget that supports this Layout
 * @author <a href="http://kennardconsulting.com">Richard Kennard</a>
 */

public interface Layout<W, C extends W, M extends C>
	extends Immutable {

	//
	// Methods
	//

	/**
	 * Layout the given widget within the given container, using the given
	 * elementName and attributes as a guide and the given Metawidget to access
	 * additional services if needed (such as state saving)
	 * <p>
	 * <tt>layoutWidget</tt> is called immediately after <tt>WidgetBuilder.buildWidget</tt> and
	 * <tt>WidgetProcessor.processWidget</tt>, and <em>before</em> the next widget is generated. An
	 * alternate design would be to 'collect' all widgets generated by <tt>buildWidget</tt> and
	 * <tt>processWidget</tt>, then iterate over them separately for the layout. If you prefer this
	 * approach, you can simulate it by having <tt>layoutWidget</tt> do nothing but 'remember' each
	 * widget, then iterate over them in <tt>endContainerLayout</tt> (see the AdvancedLayout
	 * interface). However not all UI frameworks allow this approach, because they do not suport
	 * widgets being instantiated independent of a layout, nor moved between layouts (e.g. SWT)
	 * 
	 * @param widget
	 *            the widget to layout. Never null
	 * @param elementName
	 *            XML node name of the business field. Typically 'entity',
	 *            'property' or 'action'. Never null
	 * @param attributes
	 *            attributes of the widget to layout. Never null. This Map is
	 *            modifiable - changes will be passed to subsequent
	 *            WidgetProcessors and Layouts
	 * @param container
	 *            the container to add to. This is often the same as the given
	 *            Metawidget
	 * @param metawidget
	 *            the Metawidget to use to access additional services. Never
	 *            null
	 */

	// Note: we explored having layoutWidget return W (see SVN), and then having a CompositeLayout
	// class which could combine multiple Layouts such as a TabbedPaneLayout and a GridBagLayout.
	// This was problematic because:
	//
	// 1. Layouts tend to have side effects (ie. they add widgets to themselves) so it wasn't clear
	// what should happen if someone tries to combine, say, a GridBagLayout with a MigLayout.
	// 2. each Layout generally expects itself to be the 'end point' of the pipeline.
	// 3. returning W makes the Layout interface identical to the WidgetProcessor interface.
	//
	// We tried instead making TabbedPaneLayout into TabbedPaneProcessor (see SVN). This was also
	// problematic because:
	//
	// 1. Nested sections had to be handled as nested, partially-initalised Metawidgets which could
	// then use their chosen Layout (eg. GridBagLayout)
	// 2. Attributes for the components had to be attached somehow (ie. putClientProperty, or
	// wrapped in a Stub)
	// 3. elementNames for the components had to be attached somehow
	// 4. It 'felt' weird having a WidgetProcessor doing Layout stuff
	//
	// We finally settled on having a container W and a LayoutDecorator
	//
	void layoutWidget( W widget, String elementName,
			Map<String, String> attributes, C container, M metawidget );
}
