/*
 * Copyright (c) 2005, Filip Defoort, CirqueDigital.com
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * . Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.  
 * 
 * . Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * 
 * . Neither the name of the "OpenWFE" nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: LdapService.java 2391 2006-03-04 18:53:45Z jmettraux $
 */

//
// LdapService.java
//
// Filip Defoort  CirqueDigital.com
//

package openwfe.org.ldap;

import java.security.Principal;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

import javax.naming.Context;
import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.directory.InitialDirContext;
import javax.naming.directory.SearchControls;
import javax.naming.directory.SearchResult;

import openwfe.org.MapUtils;
import openwfe.org.AbstractService;
import openwfe.org.ApplicationContext;
import openwfe.org.ServiceException;

/**
 * This service encapsulates the interface with an LDAP server.
 * 
 * The idea is that other services (e.g. LdapParticipantMap) simply look up
 * this LdapService and use it.
 * 
 * @author filipdef@cirquedigital.com
 * @author john.mettraux@openwfe.org (parametrization)
 */
public class LdapService 
    extends AbstractService 
{

    private final static org.apache.log4j.Logger log = org.apache.log4j.Logger
    .getLogger(LdapService.class.getName());

    //
    // CONSTANTS and co

    /**
     * The host (name or IP) of the LDAP server.
     */
    public final static String P_HOST 
        = "host";

    /**
     * The port number of the LDAP server.
     */
    public final static String P_PORT 
        = "port";

    /**
     * The authentication method to use to connect to the LDAP server.
     */
    public final static String P_AUTH_METHOD 
        = "authentication-method";

    /**
     * Default value for this param is "com.sun.jndi.ldap.LdapCtxFactory".
     */
    public final static String P_INITIAL_CONTEXT_FACTORY 
        = "initial-context-factory";

    /**
     * Base DN for Users. E.g. if users are represented in LDAP as
     * uid=myuserid,ou=Users,dc=openwfe,dc=org then the user-base
     * is ou=Users,dc=openwfe=dc=org.
     *
     */
    public final static String P_USER_BASE = "user-base";

    /**
     * LDAP attribute used for userid, typically "uid".
     *
     */
    public final static String P_USER_ATTRIBUTE = "user-attribute";

    /**
     * LDAP object class for User. If this value != null, then 
     * users will be limited to user objects that implement the 
     * LDAP objClass specified here. If this value == null, all
     * users underneath P_USER_BASE will be returned.
     *
     */
    public final static String P_USER_OBJCLASS = "user-objclass";

    /**
     * Base DN for Groups. E.g. if groups are represented in LDAP as
     * cn=mygroup,ou=Groups,dc=openwfe,dc=org then the group-base
     * is ou=Groups,dc=openwfe=dc=org.
     *
     */
    public final static String P_GROUP_BASE = "group-base";

    /**
     * LDAP attribute used for groupid, typically "cn".
     */
    public final static String P_GROUP_ATTRIBUTE = "group-attribute";

    /**
     * LDAP object class for Groups. If this value != null, then 
     * groups will be limited to objects that implements the LDAP
     * objClass specified here. If this value == null, all groups
     * underneath P_GROUP_BASE will be returned.
     */
    public final static String P_GROUP_OBJCLASS = "group-objclass";

    /**
     * If this value != null, then its value will be removed from
     * the returned user and group ids. E.g. if a userid is 
     * uid=myuserid,ou=Users,dc=openwfe,dc=org and truncate-suffix
     * is ,dc=openwfe,dc=org then the returned userid will be
     * uid=myuserid,ou=Users.
     */
    public final static String P_TRUNCATE_SUFFIX = "truncate-suffix";

    //
    // FIELDS

    private Properties m_envProps;
    private String m_host;
    private String m_port;
    private String m_authenticationMethod;
    private String m_userBase;
    private String m_userAttr;
    private String m_userObjClass;
    private String m_groupBase;
    private String m_groupAttr;
    private String m_groupObjClass;
    private String m_truncateSuffix;
    private InitialDirContext m_dirCtx;

    //
    // CONSTRUCTORS

    public LdapService() {
        super();
    }

    public void init(final String serviceName,
            final ApplicationContext context, 
            final java.util.Map serviceParams)
    throws
    ServiceException
    {
        log.info("LdapService init()");
        
        super.init(serviceName, context, serviceParams);
        
        m_host = (String) serviceParams.get(P_HOST);
        m_port = (String) serviceParams.get(P_PORT);

        m_authenticationMethod = (String) serviceParams.get(P_AUTH_METHOD);
        
        m_envProps = new Properties();

        //m_envProps.put(Context.INITIAL_CONTEXT_FACTORY, 
        //        "com.sun.jndi.ldap.LdapCtxFactory");
        m_envProps.put
            (Context.INITIAL_CONTEXT_FACTORY, 
             MapUtils.getAsString
                (serviceParams, 
                 P_INITIAL_CONTEXT_FACTORY, 
                 "com.sun.jndi.ldap.LdapCtxFactory"));

        m_envProps.put(Context.PROVIDER_URL, 
                "ldap://" + m_host + ":" + m_port);
        m_envProps.put(Context.SECURITY_AUTHENTICATION, m_authenticationMethod);
        
        m_userBase = (String) serviceParams.get(P_USER_BASE);
        m_userAttr = (String) serviceParams.get(P_USER_ATTRIBUTE);
        m_userObjClass = (String) serviceParams.get(P_USER_OBJCLASS);
        m_groupBase = (String) serviceParams.get(P_GROUP_BASE);
        m_groupAttr = (String) serviceParams.get(P_GROUP_ATTRIBUTE);
        m_groupObjClass = (String) serviceParams.get(P_GROUP_OBJCLASS);
        
        m_truncateSuffix = (String) serviceParams.get(P_TRUNCATE_SUFFIX);
        
        if (log.isInfoEnabled()) {
            log.info("Using " + m_host + ":" + m_port + " for ldap.");
        }
        if (log.isDebugEnabled()) {
            log.debug("User base: " + m_userBase);
            log.debug("User attr: " + m_userAttr);
            log.debug("User objclass: " + m_userObjClass);
            log.debug("Group base: " + m_groupBase);
            log.debug("Group attr: " + m_groupAttr);
            log.debug("Group objclass: " + m_groupObjClass);
            log.debug("Strip suffix: " + m_truncateSuffix);
        }
        
        try {
            m_dirCtx = new InitialDirContext(m_envProps);
        } catch (NamingException e) {
            throw new ServiceException(e.getMessage(), e);
        }
        
    }

    //
    // METHODS

    /**
     * Strip the special common suffix from the name.
     * Do this for all names coming from LDAP and 
     * going to the GDI side.
     *
     * @param dn
     * @return
     */
    private String stripSuffix(final String dn) {
        if (m_truncateSuffix == null) {
            // no truncate hooked up, nothing to do
            return dn;
        } else if (dn.endsWith(m_truncateSuffix)) {
            // ends with suffix, remove it
            return dn.substring(0, dn.length() - m_truncateSuffix.length());
        } else {
            // doesn't end with suffix, leave it be
            return dn;
        }
    }

    public String getLdapUid(final Principal p) {
        
        return stripSuffix(m_userAttr + "=" + p.getName() + "," + m_userBase);
        
    }
    
    public String[] getGroups(final Principal p) throws NamingException {
        
        SearchControls constraints = new SearchControls();
        constraints.setSearchScope(SearchControls.ONELEVEL_SCOPE);
        
        String id = getLdapUid(p);
        int i = id.indexOf(",");
        if (i != -1) {
            id = id.substring(0, i);
        }
        i = id.indexOf("=");
        if (i != -1) {
            id = id.substring(i + 1);
        }
        
        NamingEnumeration results =
            m_dirCtx.search(m_groupBase, "memberUid" + "=" + id, constraints);
     
        List groups = new ArrayList();

        while (results != null && results.hasMoreElements()) {
            SearchResult si = (SearchResult) results.next();
            
            groups.add(stripSuffix(si.getName() + "," + m_groupBase));
            
        }
        
        return (String[]) groups.toArray(new String[0]);
    }
    
    public String[] getGroups() throws NamingException {
        SearchControls constraints = new SearchControls();
        constraints.setSearchScope(SearchControls.ONELEVEL_SCOPE);
        
        NamingEnumeration results;
        
        if (m_groupObjClass != null && !"".equals(m_groupObjClass)) {
            
            results = m_dirCtx.search(m_groupBase, 
                    "(&(objectClass=" + m_groupObjClass + ")(" + m_groupAttr + "=*))", 
                    constraints);
        } else {
            results = m_dirCtx.search(m_groupBase, 
                    m_groupAttr + "=*", 
                    constraints);
        }

        List groups = new ArrayList();
        
        while (results != null && results.hasMoreElements()) {
            SearchResult si = (SearchResult) results.next();
            
            groups.add(stripSuffix(si.getName() + "," + m_groupBase));
        }

        return (String[]) groups.toArray(new String[0]);
    }

    public String[] getUsers() throws NamingException {
        SearchControls constraints = new SearchControls();
        constraints.setSearchScope(SearchControls.ONELEVEL_SCOPE);
        
        NamingEnumeration results;
        
        if (m_userObjClass != null && !"".equals(m_userObjClass)) {
            
            results = m_dirCtx.search(m_userBase, 
                    "(&(objectClass=" + m_userObjClass + ")(" + m_userAttr + "=*))", 
                    constraints);
        } else {
            results = m_dirCtx.search(m_userBase, 
                    m_userAttr + "=*", 
                    constraints);
        }

        List users = new ArrayList();

        while (results != null && results.hasMoreElements()) {
            SearchResult si = (SearchResult) results.next();
            
            users.add(stripSuffix(si.getName() + "," + m_userBase));
            
        }

        return (String[]) users.toArray(new String[0]);
    }
}
