/*
 * Copyright (c) 2001-2006, John Mettraux, OpenWFE.org
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * . Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.  
 * 
 * . Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * 
 * . Neither the name of the "OpenWFE" nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: ChannelInputStream.java 2713 2006-06-01 14:38:45Z jmettraux $
 */

//
// ChannelInputStream.java
//
// john.mettraux@openwfe.org
//
// generated with 
// jtmpl 1.1.01 2004/05/19 (john.mettraux@openwfe.org)
//

package openwfe.org.net;

import java.nio.ByteBuffer;
import java.nio.channels.ReadableByteChannel;


/**
 * Wrapping a channel inside of an InputStream.
 * The initial intent of this method is to feed a SAXBuilder with a Channel.<br>
 * We had to implement this class in order to avoid 'non-blocking' problems
 * with Channels.newInputStream(ch); ...
 *
 * <p><font size=2>CVS Info :
 * <br>$Author: jmettraux $
 * <br>$Id: ChannelInputStream.java 2713 2006-06-01 14:38:45Z jmettraux $ </font>
 *
 * @author john.mettraux@openwfe.org
 */
public class ChannelInputStream

    extends java.io.InputStream

{

    private final static org.apache.log4j.Logger log = org.apache.log4j.Logger
        .getLogger(ChannelInputStream.class.getName());

    //
    // CONSTANTS & co

    /**
     * If the system property 'openwfe.org.net.ChannelInputStream.debug' is
     * set to true, this class will emit lots of debug output.
     */
    public final static String P_DEBUG
        = ChannelInputStream.class.getName()+".debug";

    private final static boolean DEBUG =
        (System.getProperty(P_DEBUG) != null &&
         System.getProperty(P_DEBUG).toLowerCase().equals("true"));

    static
    {
        log.info(")( emit extra debug info ? "+DEBUG);
    }

    //
    // FIELDS

    private ReadableByteChannel channel = null;

    private int bytesRead = 0;
    private int bytesToRead = -1;

    private int maxAttempt = 10;

    private int residualByte = -1;

    //
    // CONSTRUCTORS

    /**
     * This constructor wraps the given channel.
     */
    public ChannelInputStream (final ReadableByteChannel channel)
    {
        this.channel = channel;
    }

    /**
     * This constructor wraps the given channel, and it is also given
     * an indication on how many bytes are to be read.
     */
    public ChannelInputStream 
        (final ReadableByteChannel channel, 
         final int bytesToRead)
    {
        this.channel = channel;
        this.bytesToRead = bytesToRead;
    }

    public ChannelInputStream 
        (final ReadableByteChannel channel, 
         final int bytesToRead,
         final int maxAttempt)
    {
        this.channel = channel;
        this.bytesToRead = bytesToRead;
        this.maxAttempt = maxAttempt;
    }

    //
    // GETTERS and SETTERS

    public void setMaxAttempt (final int ma)
    {
        this.maxAttempt = ma;
    }

    public void setBytesToRead (final int btr)
    {
        this.bytesToRead = this.bytesRead + btr;
    }

    //
    // METHODS from InputStream

    /**
     * Reads and returns the next byte of the channel.
     */
    public int read ()
        throws java.io.IOException
    {
        if (DEBUG) log.info("read()");

        final byte[] bs = new byte[1];
        final int i = read(0, bs, 0, 1);

        if (i < 1) return i;

        return bs[0];
    }

    /**
     * Reads byte into a given byte array, returns the count of byte read
     * from the channel.
     */
    public int read (final byte[] ba)
        throws java.io.IOException
    {
        if (DEBUG) log.info("read(ba)");

        final ByteBuffer buffer = ByteBuffer.wrap(ba);

        final int readCount = this.channel.read(buffer);

        this.bytesRead += readCount;

        return readCount;
    }

    /**
     * Reads byte into a given byte array, returns the count of byte read
     * from the channel.
     */
    public int read (final byte[] ba, final int offset, final int length)
        throws java.io.IOException
    {
        if (DEBUG) log.info("read(ba, o, l) : (ba, "+offset+", "+length+")");

        return read(0, ba, offset, length);
    }

    /*
     * doing the job
     */
    private int read 
        (final int attempt, final byte[] ba, final int offset, final int length)
    throws 
        java.io.IOException
    {
        if (DEBUG) log.info
            ("read(a, ba, o, l) btr : "+this.bytesToRead+
             " / br : "+this.bytesRead);
        if (DEBUG) log.info
            ("read(a, ba, o, l) : ("+attempt+", ba, "+offset+", "+length+")");

        if (attempt > this.maxAttempt) return -1;

        final ByteBuffer buffer = ByteBuffer.wrap(ba, offset, length);

        final int readCount = this.channel.read(buffer);

        if (DEBUG) 
            log.info("read(a, ba, o, l) ("+attempt+") readCount : "+readCount);

        if (readCount == 0)
        {
            if (this.bytesToRead > 0 &&
                this.bytesRead >= this.bytesToRead)
            {
                log.debug("read(a, ba, o, l) done.");
                return -1;
            }

            if (attempt % 5 == 4) 
            {
                try
                {
                    Thread.sleep(28);
                }
                catch (final InterruptedException e)
                {
                    // do nada
                }
            }
            else
            {
                Thread.yield();
            }

            return read(attempt+1, ba, offset, length);
        }

        this.bytesRead += readCount;

        return readCount;
    }

    /**
     * Closes this input stream (and especially the underlying channel).
     */
    public void close ()
        throws java.io.IOException
    {
        //this.channel.close();
            //
            // The close responsability is not left to this class.
    }

    //
    // METHODS

    //
    // STATIC METHODS

    /**
     * A convenience (static) method : returns a BufferedReader built from
     * a ReadableByteChannel (of course, a ChannelInputStream is used
     * in the middle).
     */
    public static java.io.BufferedReader newBufferedReader
        (final ReadableByteChannel channel, final String charsetName)
    throws
        java.io.UnsupportedEncodingException
    {
        return new java.io.BufferedReader
            (newInputStreamReader(channel, charsetName));
    }

    /**
     * Returns a new InputStreamReader wrapping the given channel.
     */
    public static java.io.InputStreamReader newInputStreamReader
        (final ReadableByteChannel channel, final String charsetName)
    throws
        java.io.UnsupportedEncodingException
    {
        return new java.io.InputStreamReader
            (new ChannelInputStream(channel), charsetName);
    }

    /* *
     * Simply reads one byte from a ReadableByteChannel.
     * /
    public static int read (final ReadableByteChannel channel)
        throws java.io.IOException
    {
        //ByteBuffer bb = ByteBuffer.allocateDirect(1);
        ByteBuffer bb = ByteBuffer.allocate(1);

        int readCount = channel.read(bb);

        if (readCount < 1) return readCount;

        bb.flip();

        return bb.get();
    }
     */

}
