/**
 * PETALS - PETALS Services Platform. Copyright (c) 2005-2006 EBM WebSourcing,
 * http://www.ebmwebsourcing.com/
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version. This library is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * 
 * -------------------------------------------------------------------------
 * $Id: RemoteDeliveryChannelImpl.java,v 1.1 2007/06/11 13:34:07 lam Exp $
 * -------------------------------------------------------------------------
 */

package org.objectweb.petals.tools.rmi.server.remote.implementations;

import java.rmi.RemoteException;
import java.rmi.server.UnicastRemoteObject;
import java.util.Map;
import java.util.logging.Logger;

import javax.jbi.component.ComponentContext;
import javax.jbi.messaging.DeliveryChannel;
import javax.jbi.messaging.ExchangeStatus;
import javax.jbi.messaging.MessageExchange;
import javax.jbi.messaging.MessageExchangeFactory;
import javax.jbi.messaging.MessagingException;
import javax.jbi.servicedesc.ServiceEndpoint;
import javax.xml.namespace.QName;

import org.objectweb.petals.tools.rmi.server.remote.interfaces.RemoteDeliveryChannel;
import org.objectweb.petals.tools.rmi.server.remote.interfaces.RemoteMessageExchangeFactory;
import org.objectweb.petals.tools.rmi.server.util.Convert;

/**
 * The rmi implmentation of delivery channel.
 * 
 * @author nsalatge
 * 
 */
public class RemoteDeliveryChannelImpl extends UnicastRemoteObject implements RemoteDeliveryChannel {

    /**
     * the serialVersionUID.
     */
    private static final long serialVersionUID = -864958310638895099L;

    /**
     * internal error message.
     */
    private static final String INTERNAL_ERROR_MSG = "This message exchange is not present in the HashMap\n"
            + "Either the message exchange has been "
            + "created by another message exchange factory\n"
            + "or the message exchange is terminated "
            + "(It has already been deleted of the HashMap)";

    /**
     * the delivery channel.
     */
    private DeliveryChannel deliveryChannel;

    /**
     * the component context.
     */
    private ComponentContext componentContext;

    /**
     * HasMap containing all the messages exchanges.
     */
    private Map<String, MessageExchange> messages;

    /**
     * the component logger.
     */
    private Logger logger;

    /**
     * the constructor.
     * 
     * @param dl
     *            the delivery channel
     * @param c
     *            the component context
     * @param msgs
     *            the messages
     * @param log
     *            the logger
     * @throws RemoteException
     *             impossible to realize a remote access
     */
    public RemoteDeliveryChannelImpl(final DeliveryChannel dl, final ComponentContext c,
            final Map<String, MessageExchange> msgs, final Logger log) throws RemoteException {
        super();
        this.deliveryChannel = dl;
        this.messages = msgs;
        this.logger = log;
        this.componentContext = c;
    }

    /**
     * TODO accept check multi thread Blocking call used to service a
     * MessageExchange instance which has been initiated by another component.
     * This method supports concurrent invocation for multi-threaded
     * environments. This must be interruptable (Thread.interrupt()).
     * 
     * @return message exchange instance; must be non-null
     * @throws MessagingException
     *             failed to accept
     * @throws MessagingException
     *             when the blocked thread is interrupted
     * @throws RemoteException
     *             impossible to realize a remote access
     */
    public MessageExchange accept() throws MessagingException, RemoteException {
        MessageExchange fake = null;

        // realize the accept
        MessageExchange concrete = null;
        try {
            concrete = this.deliveryChannel.accept();
        } catch (MessagingException e) {
            throw Convert.convertMessagingException(e);
        }

        if (concrete != null) {
            // Convert concrete exchange to fake exchange
            fake = Convert.convertConcreteMessageExchangeToFakeMessageExchange(concrete, concrete
                    .getPattern());

            // Store the message
            this.checkAndStorageMessageExchange(fake, concrete);

            // Get the concrete message and delete of the hashmap if not
            // active
            this.getConcreteMessageOrRemoveIfNotActive(fake);

            this.logger.finest("Return the Accept() method, exchange Id : " + fake.getExchangeId());
        }
        return fake;
    }

    /**
     * TODO accept check multi thread Blocking call used to service a
     * MessageExchange instance which has been initiated by another component,
     * within the given timeout period. This method supports concurrent
     * invocation for multi-threaded environments.
     * 
     * @param timeoutMS
     *            timeout period, in milliseconds (0 means no timeout); must be
     *            greater than or equal to zero
     * @return message exchange instance, or <code>null</code> if timed out
     * @throws MessagingException
     *             failed to accept
     * @throws MessagingException
     *             when the blocked thread is interrupted
     * @throws RemoteException
     *             impossible to realize a remote access
     */
    public MessageExchange accept(final long timeoutMS) throws MessagingException, RemoteException {

        MessageExchange fake = null;

        // realize the accept
        MessageExchange concrete = null;
        try {
            concrete = this.deliveryChannel.accept(timeoutMS);
        } catch (MessagingException e) {
            throw Convert.convertMessagingException(e);
        }

        if (concrete != null) {
            // Convert concrete exchange to fake exchange
            fake = Convert.convertConcreteMessageExchangeToFakeMessageExchange(concrete, concrete
                    .getPattern());

            // Store the message
            this.checkAndStorageMessageExchange(fake, concrete);

            // Get the concrete message and delete of the hashmap if not
            // active
            this.getConcreteMessageOrRemoveIfNotActive(fake);

            this.logger.finest("Return the Accept(timeoutMS) method, exchange Id : "
                    + fake.getExchangeId());
        }

        return fake;
    }

    /**
     * TODO close pending messages, remove endpoint entries ? Closes the
     * delivery channel, halting all message traffic.
     * 
     * @throws MessagingException
     *             fatal error while closing channel
     * @throws RemoteException
     *             impossible to realize a remote access
     */
    public void close() throws MessagingException, RemoteException {
        throw new NoSuchMethodError("Not implemented.");
    }

    /**
     * Create a message exchange factory. This factory will create exchange
     * instances with all appropriate properties set to <code>null</code>.
     * 
     * @return a message exchange factory; must be non-null
     * @throws RemoteException
     *             impossible to realize a remote access
     */
    public RemoteMessageExchangeFactory createExchangeFactory() throws RemoteException {
        // create the concrete MessageExchangeFactory
        MessageExchangeFactory mef = this.deliveryChannel.createExchangeFactory();

        // Return the fake Message Exchange Factory
        return new RemoteMessageExchangeFactoryImpl(this.messages, mef);
    }

    /**
     * Create a message exchange factory for the given interface name.
     * 
     * @param interfaceName
     *            name of the interface for which all exchanges created by the
     *            returned factory will be set for; may be null
     * @return an exchange factory that will create exchanges for the the given
     *         interface; must be non-null
     * @throws RemoteException
     *             impossible to realize a remote access
     */
    public RemoteMessageExchangeFactory createExchangeFactory(final QName interfaceName)
            throws RemoteException {
        // create the concrete MessageExchangeFactory
        MessageExchangeFactory mef = this.deliveryChannel.createExchangeFactory(interfaceName);

        // Return the fake Message Exchange Factory
        return new RemoteMessageExchangeFactoryImpl(this.messages, mef);
    }

    /**
     * Create a message exchange factory for the given endpoint.
     * 
     * @param endpoint
     *            endpoint for which all exchanges created by the returned
     *            factory will be set for; may be null
     * @return an exchange factory that will create exchanges for the given
     *         endpoint; must be non-null
     * @throws RemoteException
     *             impossible to realize a remote access
     */
    public RemoteMessageExchangeFactory createExchangeFactory(final ServiceEndpoint endpoint)
            throws RemoteException {
        // create the concrete MessageExchangeFactory
        MessageExchangeFactory mef = this.deliveryChannel.createExchangeFactory(endpoint);

        // Return the fake Message Exchange Factory
        return new RemoteMessageExchangeFactoryImpl(this.messages, mef);
    }

    /**
     * Create a message exchange factory for the given service name.
     * 
     * @param serviceName
     *            name of the service for which all exchanges created by the
     *            returned factory will be set for; may be null
     * @return an exchange factory that will create exchanges for the the given
     *         service; must be non-null
     * @throws RemoteException
     *             impossible to realize a remote access
     */
    public RemoteMessageExchangeFactory createExchangeFactoryForService(final QName serviceName)
            throws RemoteException {
        // create the concrete MessageExchangeFactory
        MessageExchangeFactory mef = this.deliveryChannel.createExchangeFactory(serviceName);

        // Return the fake Message Exchange Factory
        return new RemoteMessageExchangeFactoryImpl(this.messages, mef);
    }

    /**
     * TODO send resolve jbi logical endpoint Routes a MessageExchange instance
     * through the Normalized Message Router to the appropriate servicing
     * component. This method supports concurrent invocation for multi-threaded
     * environments.
     * <p>
     * This is used not only to send the initial message in an exchange, but
     * also for the servicer to "return" the exchange with the appropriate
     * response (response, fault, or <code>ExchangeStatus</code>). In more
     * complex message exchange patterns, a single <code>MessageExchange</code>
     * can be sent back-and-forth via <code>send()</code> several times, but
     * always terminating with the MessageExchange instance being sent with a
     * terminal <code>ExchangeStatus</code>.
     * 
     * @param exchange
     *            message exchange to send; must be non-null
     * @throws MessagingException
     *             unable to send exchange
     * @throws RemoteException
     *             impossible to realize a remote access
     */
    public void send(final MessageExchange exchange) throws MessagingException, RemoteException {
        // Verification
        if (exchange == null) {
            throw new MessagingException("send: The MessageExchange is null");
        }

        if (!(exchange instanceof org.objectweb.petals.tools.rmi.common.serializable.MessageExchange)) {
            throw new MessagingException("send: Invalid object");
        }
        // Get the concrete message and delete of the hasmap
        MessageExchange concrete = this.getConcreteMessageOrRemoveIfNotActive(exchange);

        if (concrete == null) {
            throw new MessagingException(RemoteDeliveryChannelImpl.INTERNAL_ERROR_MSG);
        }

        // Set all value
        Convert.setFakeValueMEToConcreteValueME(this.componentContext,
                (org.objectweb.petals.tools.rmi.common.serializable.MessageExchange) exchange,
                concrete);

        // Send the concrete exchange
        try {
            this.deliveryChannel.send(concrete);
        } catch (MessagingException e) {
            // Delete the message exchange of the hashmap
            this.messages.remove(exchange.getExchangeId());
            throw Convert.convertMessagingException(e);
        } catch (RuntimeException e) {
            // Delete the message exchange of the hashmap
            this.messages.remove(exchange.getExchangeId());
            throw e;
        }

        // Get the concrete message and delete of the hashmap if not active
        this.getConcreteMessageOrRemoveIfNotActive(exchange);

        this.logger.finest("Return the send() method, exchange Id : " + exchange.getExchangeId());
    }

    /**
     * TODO sendSync check multithread The RMI interface of sendSync return a
     * message exchange. it is different of jbi interface because the message
     * exchange in input parameter is not a remote object, so, it cannot be
     * setted in the method. Routes a MessageExchange instance through the
     * Normalized Message Router to the appropriate servicing component, and
     * blocks until the exchange is returned (that is, the other participant
     * sends the message exchange back). This method supports concurrent
     * invocation for multi-threaded environments.
     * <p>
     * If the thread making this call is interrupted, the message exchange is
     * treated in the same fashion as a timed-out sendSync call; see
     * {@link RemoteDeliveryChannel#sendSync(MessageExchange, long)}.
     * <p>
     * Note that the "returned" message exchange (the instance returned by the
     * other participant in the exchange) is the same instance referred to by
     * the parameter <code>exchange</code>.
     * 
     * @param exchange
     *            message exchange to send; must be non-null
     * @return <code>the message exchange</code> if the exchange has been
     *         returned, or <code>null</code>
     * @throws MessagingException
     *             unable to send exchange, or no response is expected from the
     *             <code>send()</code> operation (i.e., the
     *             <code>MessageExchange</code> is being used to convey an
     *             <code>ExchangeStatus</code>)
     * @throws MessagingException
     *             when the blocked thread is interrupted
     * @throws RemoteException
     *             impossible to realize a remote access
     */
    public javax.jbi.messaging.MessageExchange sendSync(final MessageExchange exchange)
            throws MessagingException, RemoteException {

        // Verification
        if (exchange == null) {
            throw new MessagingException("sendSync: The MessageExchange is null");
        }

        if (!(exchange instanceof org.objectweb.petals.tools.rmi.common.serializable.MessageExchange)) {
            throw new MessagingException("sendSync: Invalid object");
        }
        MessageExchange result = null;

        // Get the concrete message and delete of the hashmap
        MessageExchange concrete = this.getConcreteMessageOrRemoveIfNotActive(exchange);

        if (concrete == null) {
            throw new MessagingException(RemoteDeliveryChannelImpl.INTERNAL_ERROR_MSG);
        }

        // Set all value
        Convert.setFakeValueMEToConcreteValueME(this.componentContext,
                (org.objectweb.petals.tools.rmi.common.serializable.MessageExchange) exchange,
                concrete);

        // Send the concrete exchange
        try {
            boolean res = this.deliveryChannel.sendSync(concrete);

            if (res) {
                // Set all value
                Convert
                        .setConcreteValueMEToFakeValueME(
                                this.componentContext,
                                concrete,
                                (org.objectweb.petals.tools.rmi.common.serializable.MessageExchange) exchange);

                result = exchange;
            } else {
                result = null;
            }
        } catch (MessagingException e) {
            // Delete the message exchange of the hashmap
            this.messages.remove(exchange.getExchangeId());
            throw Convert.convertMessagingException(e);
        } catch (RuntimeException e) {
            // Delete the message exchange of the hashmap
            this.messages.remove(exchange.getExchangeId());
            throw e;
        }

        // Get the concrete message and delete of the hashmap
        concrete = this.getConcreteMessageOrRemoveIfNotActive(exchange);

        this.logger.finest("Return the sendSync() method, exchange Id : "
                + exchange.getExchangeId());

        return result;
    }

    /**
     * TODO sendSync check multithread The RMI interface of sendSync return a
     * message exchange. it is different of jbi interface because the message
     * exchange in input parameter is not a remote object, so, it cannot be
     * setted in the method. Routes a MessageExchange instance through the
     * Normalized Message Router to the appropriate servicing component, and
     * blocks until the exchange is returned (that is, the other participant
     * sends the message exchange back), or the specified timeout interval
     * elapses. This method supports concurrent invocation for multi-threaded
     * environments.
     * <p>
     * If the thread making this call is interrupted while this method is
     * blocked, the message exchange is treated as if the time out occurred.
     * <p>
     * If this method returns false (indicating that timeout occurred), the
     * message exchange must be marked by the implementation as being in the
     * {@link ExchangeStatus#ERROR} state. Attempts by the service provider to
     * send a MessageExchange in such an ERROR state must result in the send (or
     * sendSync) method called ending abruptly with an appropriate
     * MessagingException.
     * <p>
     * Note that the "returned" message exchange (the instance returned by the
     * other participant in the exchange) is the same instance referred to by
     * the parameter <code>exchange</code>.
     * 
     * @param exchange
     *            message exchange to send; must be non-null
     * @param timeoutMS
     *            timeout period, in milliseconds (0 means no timeout); must be
     *            greater than or equal to zero
     * @return <code>the message exchange</code> if the exchange has been
     *         returned, or <code>null</code> if the method timed out while
     *         waiting
     * @throws MessagingException
     *             unable to send exchange, or no response is expected from the
     *             <code>send()</code> operation (i.e., the
     *             <code>MessageExchange</code> is being used to convey an
     *             <code>ExchangeStatus</code>)
     * @throws MessagingException
     *             when the blocked thread is interrupted
     * @throws RemoteException
     *             impossible to realize a remote access
     */
    public javax.jbi.messaging.MessageExchange sendSync(final MessageExchange exchange,
            final long timeoutMS) throws MessagingException, RemoteException {
        // Verification
        if (exchange == null) {
            throw new MessagingException("sendSync with timeout: The MessageExchange is null");
        }

        if (!(exchange instanceof org.objectweb.petals.tools.rmi.common.serializable.MessageExchange)) {
            throw new MessagingException("sendSync with timeout: Invalid object");
        }

        MessageExchange result = null;

        // Get the concrete message and delete of the hasmap
        MessageExchange concrete = this.getConcreteMessageOrRemoveIfNotActive(exchange);

        if (concrete == null) {
            throw new MessagingException(RemoteDeliveryChannelImpl.INTERNAL_ERROR_MSG);
        }

        // Set all value
        Convert.setFakeValueMEToConcreteValueME(this.componentContext,
                (org.objectweb.petals.tools.rmi.common.serializable.MessageExchange) exchange,
                concrete);

        // Send the concrete exchange
        try {
            boolean res = this.deliveryChannel.sendSync(concrete, timeoutMS);

            if (res) {
                // Set all value
                Convert
                        .setConcreteValueMEToFakeValueME(
                                this.componentContext,
                                concrete,
                                (org.objectweb.petals.tools.rmi.common.serializable.MessageExchange) exchange);

                result = exchange;
            } else {
                result = null;
            }
        } catch (MessagingException e) {
            // Delete the message exchange of the hashmap
            this.messages.remove(exchange.getExchangeId());
            throw Convert.convertMessagingException(e);
        } catch (RuntimeException e) {
            // Delete the message exchange of the hashmap
            this.messages.remove(exchange.getExchangeId());
            throw e;
        }

        // Get the concrete message and delete of the hashmap
        concrete = this.getConcreteMessageOrRemoveIfNotActive(exchange);

        this.logger.finest("Return the sendSync(timeoutMS) method, exchange Id : "
                + exchange.getExchangeId());

        return result;
    }

    /**
     * Check and storage the concrete message exchange if it is not active.
     * 
     * @param fake
     *            the fake message exchange
     * @param concrete
     *            the concrete message exchange
     */
    private void checkAndStorageMessageExchange(final MessageExchange fake,
            final MessageExchange concrete) {
        if (concrete.getStatus() == javax.jbi.messaging.ExchangeStatus.ACTIVE) {
            this.messages.put(fake.getExchangeId(), concrete);
        }
    }

    /*
     * Added methods
     */

    /**
     * Get the concrete message and remove it of the hashmap. if it not active
     * 
     * @param fake
     *            the fake message exchange
     * @return the concrete message exchange or null
     */
    private MessageExchange getConcreteMessageOrRemoveIfNotActive(final MessageExchange fake) {
        MessageExchange concrete = null;
        if (fake.getStatus() != javax.jbi.messaging.ExchangeStatus.ACTIVE) {
            // The message is not active. It is deleted of the hashmap
            concrete = this.messages.remove(fake.getExchangeId());
        } else {
            // The message is active. It is not deleted of the hashmap
            concrete = this.messages.get(fake.getExchangeId());
        }
        return concrete;
    }

}
