/**
 * PETALS - PETALS Services Platform.
 * Copyright (c) 2005-2006 EBM WebSourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * $Id: RemoteComponentContext.java,v 1.1 2007/06/11 13:34:07 lam Exp $
 * -------------------------------------------------------------------------
 */

package org.objectweb.petals.tools.rmi.server.remote.interfaces;


import java.rmi.Remote;
import java.rmi.RemoteException;

import javax.jbi.JBIException;
import javax.jbi.servicedesc.ServiceEndpoint;
import javax.xml.namespace.QName;

/**
 *  the Component context interface.
 * @author nsalatge
 *
 */
public interface RemoteComponentContext extends Remote  {

    /**
     * Activates the named endpoint with the NMR. Activation indicates to the
     * NMR that this component is ready to process requests sent to the named
     * endpoint.
     * <p>
     * Note that the JBI implementation may call this component's {@link
     * Component#getServiceDescription(ServiceEndpoint)} method before returning
     * from this method call; the component's implementation must be ready to
     * supply service description metadata before the result of this activation
     * call (a ServiceEndpoint) is known.
     *
     * @param interfaceName
     *            qualified name of the interface the endpoint exposes; must be
     *            non-null.
     * @param serviceName
     *            qualified name of the service the endpoint exposes; must be
     *            non-null.
     * @param endpointName
     *            the name of the endpoint to be activated; must be non-null and
     *            non-empty.
     * @return a reference to the activated endpoint; must be non-null.
     * @throws javax.jbi.JBIException
     *             if the endpoint cannot be activated.
     * @throws RemoteException
     *             impossible to realize a remote access
     */
    public ServiceEndpoint activateEndpoint(final QName interfaceName, final QName serviceName, final String endpointName)
    throws JBIException, RemoteException;

    /**
     * Deactivates the given endpoint with the NMR. Deactivation indicates to
     * the NMR that this component will no longer process requests sent to the
     * named endpoint.
     *
     * @param endpoint
     *            reference to the endpoint to be deactivated; must be non-null.
     * @throws javax.jbi.JBIException
     *             if the endpoint cannot be deactivated.
     * @throws RemoteException
     *             impossible to realize a remote access
     */
    void deactivateEndpoint(
            javax.jbi.servicedesc.ServiceEndpoint endpoint)
    throws javax.jbi.JBIException, RemoteException;

    /**
     * Deregisters the given external endpoint with the NMR. This indicates to
     * the NMR that the given external endpoint can no longer be used as a proxy
     * for external service consumers to access an internal service of the same
     * service name.
     *
     * @param externalEndpoint
     *            the external endpoint to be deregistered; must be non-null.
     * @throws javax.jbi.JBIException
     *             if the given external endpoint was not previously registered.
     * @throws RemoteException
     *             impossible to realize a remote access
     */
    void deregisterExternalEndpoint(
            javax.jbi.servicedesc.ServiceEndpoint externalEndpoint)
    throws javax.jbi.JBIException, RemoteException;

    /**
     * Get the unique component name of this component, as assigned by the
     * identification section of this component's installation descriptor.
     *
     * @throws RemoteException
     *             impossible to realize a remote access
     * @return the component name; must be non-null and non-empty.
     */
    java.lang.String getComponentName() throws RemoteException;

    /**
     * Get a channel for this component to use to communicate with the
     * Normalized Message Router. This channel must be used by the component to
     * send and receive message exchanges.
     *
     * @return the delivery channel for this component; must be non-null.
     * @throws javax.jbi.messaging.MessagingException
     *             if a channel has already been opened, but not yet closed.
     * @throws RemoteException
     *             impossible to realize a remote access
     */
    org.objectweb.petals.tools.rmi.server.remote.interfaces.RemoteDeliveryChannel
    getDeliveryChannel()
    throws javax.jbi.messaging.MessagingException, RemoteException;

    /**
     * Get the service endpoint for the named activated endpoint, if any.
     *
     * @param service
     *            qualified-name of the endpoint's service; must be non-null.
     * @param name
     *            name of the endpoint; must be non-null.
     * @return the named endpoint, or <code>null</code> if the named endpoint
     *         is not activated.
     * @throws RemoteException
     *             impossible to realize a remote access
     */
    javax.jbi.servicedesc.ServiceEndpoint getEndpoint(
            javax.xml.namespace.QName service, java.lang.String name)
    throws RemoteException;

    /**
     * TODO getEndpointDescriptor works only with implementation InternalEndpoint
     * Retrieve the service description metadata for the specified endpoint.
     * <p>
     * Note that the result can use either the WSDL 1.1 or WSDL 2.0 description
     * language.
     *
     * @param endpoint
     *            endpoint reference; must be non-null.
     * @return metadata describing endpoint, or <code>null</code> if metadata
     *         is unavailable.
     * @throws javax.jbi.JBIException
     *             invalid endpoint reference.
     * @throws RemoteException
     *             impossible to realize a remote access
     */
    org.w3c.dom.Document getEndpointDescriptor(
            javax.jbi.servicedesc.ServiceEndpoint endpoint)
    throws javax.jbi.JBIException, RemoteException;

    /**
     * Queries the NMR for active endpoints that implement the given interface.
     * This will return the endpoints for all services and endpoints that
     * implement the named interface (portType in WSDL 1.1). This method does
     * NOT include external endpoints (those registered using {@link
     * RemoteComponentContext#registerExternalEndpoint(ServiceEndpoint)}.
     *
     * @param interfaceName
     *            qualified name of interface/portType that is implemented by
     *            the endpoint; if <code>null</code> then all activated
     *            endpoints in the JBI environment must be returned.
     * @return ServiceEndpoint[] an array of available endpoints for the
     *         specified interface name; must be non-null; may be empty.
     * @throws RemoteException
     *             impossible to realize a remote access
     */
    javax.jbi.servicedesc.ServiceEndpoint[] getEndpoints(
            javax.xml.namespace.QName interfaceName) throws RemoteException;

    /**
     * Queries the NMR for active endpoints belonging to the given service. This
     * method does NOT include external endpoints (those registered using {@link
     * RemoteComponentContext#registerExternalEndpoint(ServiceEndpoint)}.
     *
     * @param serviceName
     *            qualified name of the service that the endpoints are part of;
     *            must be non-null.
     * @return ServiceEndpoint[] array of available endpoints for the specified
     *         an array of available endpoints for the specified service name;
     *         must be non-null; may be empty.
     * @throws RemoteException
     *             impossible to realize a remote access
     */
    javax.jbi.servicedesc.ServiceEndpoint[] getEndpointsForService(
            javax.xml.namespace.QName serviceName) throws RemoteException;

    /**
     * Queries the NMR for external endpoints that implement the given interface
     * name. This methods returns only registered external endpoints (see {@link
     * RemoteComponentContext#registerExternalEndpoint(ServiceEndpoint)}.
     *
     * @param interfaceName
     *            qualified name of interface implemented by the endpoints; must
     *            be non-null.
     * @return an array of available external endpoints for the specified
     *         interface name; must be non-null; may be empty.
     * @throws RemoteException
     *             impossible to realize a remote access
     */
    javax.jbi.servicedesc.ServiceEndpoint[] getExternalEndpoints(
            javax.xml.namespace.QName interfaceName) throws RemoteException;

    /**
     * Queries the NMR for external endpoints that are part of the given
     * service.
     *
     * @param serviceName
     *            qualified name of service that contains the endpoints; must be
     *            non-null.
     * @return an array of available external endpoints for the specified
     *         service name; must be non-null; may be empty.
     * @throws RemoteException
     *             impossible to realize a remote access
     */
    javax.jbi.servicedesc.ServiceEndpoint[] getExternalEndpointsForService(
            javax.xml.namespace.QName serviceName) throws RemoteException;

    /**
     * Get the installation root directory path for this component.
     * <p>
     * This method MUST return the file path formatted for the underlying
     * platform.
     *
     * @return the installation root directory path, in platform-specific form;
     *         must be non-null and non-empty.
     * @throws RemoteException
     *             impossible to realize a remote access
     */
    java.lang.String getInstallRoot() throws RemoteException;

    /**
     * TODO getLogger return a simple  Logger, not integrated to the platform
     * Get a logger instance from JBI. Loggers supplied by JBI are guaranteed to
     * have unique names such that they avoid name collisions with loggers from
     * other components created using this method. The suffix parameter allows
     * for the creation of subloggers as needed. The JBI specification says
     * nothing about the exact names to be used, only that they must be unique
     * across components and the JBI implementation itself.
     *
     * @param suffix
     *            for creating subloggers; use an empty string for the base
     *            component logger; must be non-null.
     * @param resourceBundleName
     *            name of <code>ResourceBundle</code> to be used for
     *            localizing messages for the logger. May be <code>null</code>
     *            if none of the messages require localization. The resource, if
     *            non-null, must be loadable using the component's class loader
     *            as the initiating loader.
     * @return a standard logger, named uniquely for this component (plus the
     *         given suffix, if applicable); must be non-null.
     * @throws javax.jbi.JBIException
     *             if the resourceBundleName has changed from a previous
     *             invocation by this component of this method with the same
     *             suffix.
     * @throws RemoteException
     *             impossible to realize a remote access
     */
    java.util.logging.Logger getLogger(java.lang.String suffix,
            java.lang.String resourceBundleName)
    throws javax.jbi.JBIException, RemoteException;

    /**
     * Get a reference to the MBeanNames creator for use in creating custom
     * MBean names.
     *
     * @return reference to the MBeanNames creator; must be non-null.
     * @throws RemoteException
     *             impossible to realize a remote access
     */
    javax.jbi.management.MBeanNames getMBeanNames() throws RemoteException;

    /**
     * Get the JMX MBean server used to register all MBeans in the JBI
     * environment.
     *
     * @return a reference to the MBean server; must be non-null.
     * @throws RemoteException
     *             impossible to realize a remote access
     */
    javax.management.MBeanServer getMBeanServer() throws RemoteException;

    /**
     * TODO getNamingContext not integrated
     * Get the JNDI naming context for this component. This context is a
     * standard JNDI <code>InitialContext</code> but its content will vary
     * based on the environment in which the JBI implementation is running.
     *
     * @return the JNDI naming context; must be non-null.
     * @throws RemoteException
     *             impossible to realize a remote access
     */
    javax.naming.InitialContext getNamingContext() throws RemoteException;

    /**
     * TODO getTransactionManager not integrated
     * Get the TransactionManager for this implementation. The instance returned
     * is an implementation of the standard JTA interface. If none is available,
     * this method returns <code>null</code>.
     * <p>
     * The object returned by this method is untyped, to allow this interface to
     * be compiled in environments that do not support JTA. If not null, the
     * object returned must be of type
     * <code>javax.transaction.TransactionManager</code>.
     * <p>
     * This downcast is necessary because JBI is used in environments that do
     * not support JTA (i.e., J2SE). Explicit use of JTA types would cause
     * compilation failures in such environments.
     *
     * @return A TransactionManager instance, or null if none is available in
     *         the execution environment.
     * @throws RemoteException
     *             impossible to realize a remote access
     */
    java.lang.Object getTransactionManager() throws RemoteException;

    /**
     * Get the root directory path for this component's private workspace.
     * <p>
     * This method MUST return the file path formatted for the underlying
     * platform.
     * <p>
     * The returned value must indicate a valid file path that the component may
     * use to write files to, and read files from.
     *
     * @return the private workspace root path, in platform-specific form; must
     *         be non-null and non-empty.
     * @throws RemoteException
     *             impossible to realize a remote access
     */
    java.lang.String getWorkspaceRoot() throws RemoteException;

    /**
     * Registers the given external endpoint with the NMR. This indicates to the
     * NMR that the given endpoint is used as a proxy for external service
     * consumers to access an internal service of the same service name (but a
     * different endpoint name).
     *
     * @param externalEndpoint
     *            the external endpoint to be registered, must be non-null.
     * @throws javax.jbi.JBIException
     *             if an external endpoint with the same name is already
     *             registered, by this or another component.
     * @throws RemoteException
     *             impossible to realize a remote access
     */
    void registerExternalEndpoint(
            javax.jbi.servicedesc.ServiceEndpoint externalEndpoint)
    throws javax.jbi.JBIException, RemoteException;

    /**
     * Resolve the given endpoint reference into a service endpoint. This is
     * called by the component when it has an EPR that it wants to resolve into
     * a service endpoint.
     * <p>
     * Note that the service endpoint returned refers to a dynamic endpoint; the
     * endpoint will exist only as long as this component retains a strong
     * reference to the object returned by this method. The endpoint may not be
     * included in the list of "activated" endpoints.
     *
     * @param epr
     *            endpoint reference as an XML fragment; must be non-null.
     * @return the service endpoint corresponding to the given endpoint
     *         reference; null if the reference cannot be resolved.
     * @throws RemoteException
     *             impossible to realize a remote access
     */
    javax.jbi.servicedesc.ServiceEndpoint resolveEndpointReference(
            org.w3c.dom.DocumentFragment epr) throws RemoteException;
}
