/**
 * 
 */
package org.richfaces.component;

import java.util.Map;

import javax.faces.component.UIComponent;
import javax.faces.component.UIComponentBase;
import javax.faces.context.FacesContext;
import javax.faces.el.ValueBinding;
import javax.faces.event.AbortProcessingException;
import javax.faces.event.FacesEvent;

import org.ajax4jsf.component.AjaxComponent;
import org.ajax4jsf.event.AjaxEvent;
import org.ajax4jsf.renderkit.AjaxRendererUtils;
import org.richfaces.component.events.TreeEvents;
import org.richfaces.event.DragEvent;
import org.richfaces.event.DragListener;
import org.richfaces.event.DropEvent;
import org.richfaces.event.DropListener;
import org.richfaces.event.NodeExpandedListener;
import org.richfaces.event.NodeSelectedListener;
import org.richfaces.event.TreeListenerEventsProducer;

/**
 * @author Nick Belaevski - nbelaevski@exadel.com created 22.11.2006 Component
 *         class providing concrete representation for tree node
 */

public abstract class UITreeNode extends UIComponentBase implements
TreeListenerEventsProducer, Draggable, Dropzone, AjaxComponent {

	private String dragType;
	private Object acceptedTypes;
	private String dragIndicator;
	
	public static final String COMPONENT_TYPE = "org.richfaces.TreeNode";

	public static final String COMPONENT_FAMILY = "org.richfaces.TreeNode";

	/**
	 * Attribute name to indicate if this tree node is a virtual
	 * {@link UITreeNode} providing default representation for node. Default
	 * tree node representation should use attributes of its parent
	 * {@link UITree} component
	 */
	protected final static String DEFAULT_NODE_FACE_ATTRIBUTE_NAME = "#defaultNodeFace";

	private final static String ICON_LEAF_FACET_NAME = "iconLeaf";
	private final static String ICON_FACET_NAME = "icon";
	private final static String ICON_EXPANDED_FACET_NAME = "iconExpanded";
	private final static String ICON_COLLAPSED_FACET_NAME = "iconCollapsed";
	
	public abstract String getType();

	public abstract void setType(String type);

	public abstract String getIcon();

	public abstract void setIcon(String icon);

	public abstract String getIconExpanded();

	public abstract void setIconExpanded(String icon);

	public abstract String getIconCollapsed();

	public abstract void setIconCollapsed(String icon);

	public abstract String getIconLeaf();

	public abstract void setIconLeaf(String icon);

	public abstract void setAjaxSubmitSelection(String ajaxSubmitSelection);

	public abstract String getAjaxSubmitSelection();

	public abstract void setHighlightedClass(String selectedClass);

	public abstract String getHighlightedClass();

	public abstract void setSelectedClass(String selectedClass);

	public abstract String getSelectedClass();

	public boolean hasAjaxSubmitSelection() {
		String ajaxSubmitSelection = getAjaxSubmitSelection();
		if ("inherit".equals(ajaxSubmitSelection) || 
				ajaxSubmitSelection == null || ajaxSubmitSelection.length() == 0) {
			return getUITree().isAjaxSubmitSelection();
		} else if ("true".equals(ajaxSubmitSelection)) {
			return true;
		} else if ("false".equals(ajaxSubmitSelection)) {
			return false;
		} else {
			throw new IllegalArgumentException(
			"Property \"ajaxSubmitSelection\" should be \"inherit\", \"true\" or \"false\".");
		}

	}


	public void broadcast(FacesEvent event) throws AbortProcessingException {
		super.broadcast(event);

		FacesContext context = getFacesContext();
		UITree tree = getUITree();

		TreeEvents.invokeListenerBindings(this, event, context);

		//TODO quick fix for UITree to invoke listeners
		if (tree != null) {
			if (event instanceof DragEvent || event instanceof DropEvent) {
				tree.broadcast(event);
			}
		}

		if (event instanceof AjaxEvent) {
			AjaxRendererUtils.addRegionsFromComponent(this, getFacesContext());
		}
	}

	public void addChangeExpandListener(NodeExpandedListener listener) {
		addFacesListener(listener);
	}

	public void addNodeSelectListener(NodeSelectedListener listener) {
		addFacesListener(listener);
	}

	public void removeChangeExpandListener(NodeExpandedListener listener) {
		removeFacesListener(listener);
	}

	public void removeNodeSelectListener(NodeSelectedListener listener) {
		removeFacesListener(listener);
	}

	public NodeExpandedListener[] getChangeExpandListeners() {
		return (NodeExpandedListener[]) getFacesListeners(NodeExpandedListener.class);
	}

	public NodeSelectedListener[] getNodeSelectListeners() {
		return (NodeSelectedListener[]) getFacesListeners(NodeSelectedListener.class);
	}


	/**
	 * Finds direct parent {@link UITree} component or throws
	 * 
	 * @return {@link UITree} instance
	 */
	public UITree getUITree() {
		UIComponent parent = getParent();
		while (parent != null) {
			if (parent instanceof UITree) {
				return (UITree) parent;
			} else {
				parent = parent.getParent();
			}
		}

		return null;
	}

	public void addDropListener(DropListener listener) {
		addFacesListener(listener);
	}

	public DropListener[] getDropListeners() {
		return (DropListener[]) getFacesListeners(DropListener.class);
	}

	public void removeDropListener(DropListener listener) {
		removeFacesListener(listener);
	}


	public void addDragListener(DragListener listener) {
		addFacesListener(listener);
	}

	public DragListener[] getDragListeners() {
		return (DragListener[]) getFacesListeners(DragListener.class);
	}

	public void removeDragListener(DragListener listener) {
		removeFacesListener(listener);
	}

	public Map getAttributes() {
		Map map = super.getAttributes();

		if (Boolean.TRUE.equals(map.get(DEFAULT_NODE_FACE_ATTRIBUTE_NAME))) {
			return getUITree().getAttributes();
		}

		return map;
	}

	public void setAcceptedTypes(Object types) {
		this.acceptedTypes = types;
	}

	/*
	 * List of drag types to be processd by the current drop zone.
	 * Getter for acceptedTypes
	 * @return acceptedTypes value from local variable or value bindings
	 */
	public Object getAcceptedTypes(  ){
		if (null != this.acceptedTypes)
		{
			return this.acceptedTypes;
		}
		ValueBinding vb = getValueBinding("acceptedTypes");
		if (null != vb){
			return (Object)vb.getValue(getFacesContext());
		} else {
			UITree tree = getUITree();
			if (tree != null) {
				return tree.getAcceptedTypes();
			}

			return null;
		}
	}

	public void setDragType(String dragType) {
		this.dragType = dragType;
	}

	/*
	 * Key of a drag object. It's used to define a necessity of processing the current dragged element on the drop zone side
	 * Getter for dragType
	 * @return dragType value from local variable or value bindings
	 */
	public String getDragType(  ){
		if (null != this.dragType)
		{
			return this.dragType;
		}
		ValueBinding vb = getValueBinding("dragType");
		if (null != vb){
			return (String)vb.getValue(getFacesContext());
		} else {
			UITree tree = getUITree();
			if (tree != null) {
				return tree.getDragType();
			}
			
			return null;
		}
	}

	private UIComponent getTreeFacet(String facetName) {
		UIComponent facet = getFacet(facetName);
		if (facet == null || !facet.isRendered()) {
			UIComponent parentFacet = getUITree().getFacet(facetName);
			if (facet == null || parentFacet != null && parentFacet.isRendered()) {
				facet = parentFacet;
			}
		}
		
		return facet;
	}
	
	public UIComponent getIconFacet() {
		return getTreeFacet(ICON_FACET_NAME);
	}

	public UIComponent getIconLeafFacet() {
		return getTreeFacet(ICON_LEAF_FACET_NAME);
	}
	
	public UIComponent getIconExpandedFacet() {
		return getTreeFacet(ICON_EXPANDED_FACET_NAME);
	}
	
	public UIComponent getIconCollapsedFacet() {
		return getTreeFacet(ICON_COLLAPSED_FACET_NAME);
	}

	public Object saveState(FacesContext context) {
		Object[] state = new Object[4];
		state[0] = super.saveState(context);
		state[1] = this.dragType;
		state[2] = this.acceptedTypes;
		state[3] = this.dragIndicator;
		
		return state;
	}

	public void restoreState(FacesContext context, Object state) {
		Object[] _state = (Object[]) state;
		super.restoreState(context, _state[0]);
		this.dragType = (String) _state[1];
		this.acceptedTypes = _state[2];
		this.dragIndicator = (String) _state[3];
	}
	
	public void setDragIndicator(String dragIndicator) {
		this.dragIndicator = dragIndicator;
	}
	
	protected String getLocalDragIndicator() {
		if (dragIndicator != null) {
			return dragIndicator;
		}
		
		ValueBinding vb = getValueBinding("dragIndicator");
		if (vb != null) {
			return (String) vb.getValue(getFacesContext());
		}
		
		return null;
	}
	
	public String getDragIndicator() {
		String localDragIndicator = getLocalDragIndicator();
		if (localDragIndicator == null) {
			UITree tree = getUITree();
			if (tree != null) {
				return tree.getDragIndicator();
			} else {
				return null;
			}
		} else {
			return localDragIndicator;
		}
	}
	
	public String getResolvedDragIndicator(FacesContext facesContext) {
		String indicatorId = getLocalDragIndicator();
		UITree tree = getUITree();
		if (tree != null) {
			if (indicatorId != null) {
				//tree is naming container
				UIComponent parent = tree.getParent();
				if (parent != null) {
					UIComponent indicator = parent.findComponent(indicatorId);
					if (indicator != null) {
						return indicator.getClientId(facesContext);
					}
				}
			} else {
				return tree.getResolvedDragIndicator(facesContext);
			}
		}
		
		return null;
	}
	
	protected String getDefaultOndragend() {
		//tag invokes read method on component creation
		//we shouldn't fail with NPE
		UITree tree = getUITree();
		if (tree == null) {
			return null;
		}
		return tree.getOndragend();
	}
	
	protected String getDefaultOndragenter() {
		UITree tree = getUITree();
		if (tree == null) {
			return null;
		}
		return tree.getOndragenter();
	}
	
	protected String getDefaultOndragexit() {
		UITree tree = getUITree();
		if (tree == null) {
			return null;
		}
		return tree.getOndragexit();
	}
	
	protected String getDefaultOndragstart() {
		UITree tree = getUITree();
		if (tree == null) {
			return null;
		}
		return tree.getOndragstart();
	}
	
	protected String getDefaultOndrop() {
		UITree tree = getUITree();
		if (tree == null) {
			return null;
		}
		return tree.getOndrop();
	}
	
	protected String getDefaultOndropend() {
		UITree tree = getUITree();
		if (tree == null) {
			return null;
		}
		return tree.getOndropend();
	}
}
