/*
 * Copyright 2005,2006 WSO2, Inc. http://www.wso2.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.wso2.ws.dataservice;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.io.Reader;
import java.io.StringWriter;
import java.net.MalformedURLException;
import java.net.URL;
import java.sql.Connection;
import java.sql.Date;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Properties;
import java.util.Set;

import javax.xml.namespace.QName;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;

import org.apache.axiom.om.OMAbstractFactory;
import org.apache.axiom.om.OMElement;
import org.apache.axiom.om.OMFactory;
import org.apache.axiom.om.OMNamespace;
import org.apache.axiom.om.OMNode;
import org.apache.axiom.om.impl.builder.StAXOMBuilder;
import org.apache.axiom.om.util.StAXUtils;
import org.apache.axis2.AxisFault;
import org.apache.axis2.context.MessageContext;
import org.apache.axis2.description.AxisOperation;
import org.apache.axis2.description.AxisService;
import org.apache.axis2.description.Parameter;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.poi.hssf.usermodel.HSSFCell;
import org.apache.poi.hssf.usermodel.HSSFRow;
import org.apache.poi.hssf.usermodel.HSSFSheet;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.poifs.filesystem.POIFSFileSystem;
import org.wso2.ws.dataservice.beans.CSVQuery;
import org.wso2.ws.dataservice.beans.ExcelQuery;

import au.com.bytecode.opencsv.CSVReader;

public class DBUtils {
	private static final Log log = LogFactory.getLog(DBUtils.class);

	private String configFilePath;

	public static OMNode toOM(Reader reader) throws XMLStreamException {
		XMLStreamReader xmlReader = StAXUtils.createXMLStreamReader(reader);
		OMFactory fac = OMAbstractFactory.getOMFactory();
		StAXOMBuilder staxOMBuilder = new StAXOMBuilder(fac, xmlReader);
		return staxOMBuilder.getDocumentElement();
	}

	public String getConfigFilePath() {
		return configFilePath;
	}

	public void setConfigFilePath(String config_file_parth) {
		this.configFilePath = config_file_parth;
	}

	private synchronized static OMElement processSQLQuery(OMElement operationElement, AxisService axisService,
			OMElement inputMessage) {
		HashMap inputs = new HashMap();
		for (Iterator iter = inputMessage.getChildElements(); iter.hasNext();) {
			OMElement element = (OMElement) iter.next();
			inputs.put(element.getLocalName(), element.getText());
		}

		OMElement callQueryElement =
			operationElement.getFirstChildWithName(new QName("call-query"));
		return getRDBMSResult(callQueryElement, axisService, inputs);
	}

	private static OMElement getRDBMSResult(OMElement callQueryElement, AxisService axisService,
			HashMap inputs) {
		OMElement resultElement = null;
		OMElement queryElement ;
		try {
			// Find the corresponding query element
			String href = callQueryElement.getAttributeValue(new QName("href"));
			if (href != null) {
				HashMap queries =
					(HashMap) axisService.getParameterValue(DBConstants.DB_QUERY_ELEMENTS);
				queryElement = (OMElement) queries.get(href);
			} else {
				queryElement = callQueryElement.getFirstChildWithName(new QName("query"));
			}

			HashMap params = new HashMap();
			// TODO : sumedha, we need a mechanism to specify parameter ordering
			// in the spec.
			// Right now order in which it appears in taken.
			HashMap paramOrder = new HashMap();
			Iterator iParams = queryElement.getChildrenWithName(new QName("param"));
			int position = 0;
			while (iParams.hasNext()) {
				OMElement param = (OMElement) iParams.next();
				String paramName = param.getAttributeValue(new QName("name"));
				String sqlType = param.getAttributeValue(new QName("sqlType"));
				params.put(paramName, sqlType);
				paramOrder.put(new Integer(position + 1), paramName);
				position++;
			}

			String query = queryElement.getFirstChildWithName(new QName("sql")).getText();
			// Check the type of SQL statement & get result
			// SELECT,INSERT,UPDATE,DELETE,CREATE all have 6 chars ;-)

			String firstPartOfSQL = query.substring(0, 6);

			if (firstPartOfSQL.equalsIgnoreCase("SELECT")) {
				resultElement =
					getSelectResult(queryElement, inputs, params, paramOrder, axisService, false);

			} else if (firstPartOfSQL.equalsIgnoreCase("INSERT")
					|| firstPartOfSQL.equalsIgnoreCase("UPDATE")
					|| firstPartOfSQL.equalsIgnoreCase("DELETE")
					|| firstPartOfSQL.equalsIgnoreCase("CREATE")) {
				resultElement = getSelectResult(queryElement, inputs, params, paramOrder, axisService, true);

			}else {
				//This can be a stored procedure call
				resultElement =
					getSelectResult(queryElement, inputs, params, paramOrder, axisService, false);
			}
		} catch (Exception e) {
			if(e instanceof AxisFault){
				return generateErrorResponse(axisService.getTargetNamespace(), (AxisFault)e);
			}
	    }
		return resultElement;
	}

	/*
	 * Generates an response containing fault element.
	 * Fault element contains two elements(code & detail) 
	 */
	private static OMElement generateErrorResponse(String targetNamespace,AxisFault e){
		//get stackTrace into a String
	    StringWriter sw = new StringWriter();
	    PrintWriter pw = new PrintWriter(sw);
	    e.printStackTrace(pw);

		
		OMFactory fac = OMAbstractFactory.getOMFactory();
		OMNamespace omNs = fac.createOMNamespace(targetNamespace, "data");
		
		OMElement resultElement = fac.createOMElement("data", omNs);
		//TODO : read from query/fault
		OMElement faultElement = fac.createOMElement("fault",null);		
		
		OMElement codeElement = fac.createOMElement("code", omNs);
		codeElement.setText(e.getMessage());
		OMElement detailElement = fac.createOMElement("detail", omNs);
		detailElement.setText(sw.toString());
		
		faultElement.addChild(codeElement);
		faultElement.addChild(detailElement);
		resultElement.addChild(faultElement);
		return resultElement;				
	}
	
//	private static void getDefaultValuesForResultsWrapper(String wrapperElementName,String rowElementName){
//	    if(wrapperElementName == null || wrapperElementName.trim().length() == 0){
//	        //default value
//	        wrapperElementName = "results";	            
//	    }
//        if(rowElementName == null || rowElementName.trim().length() == 0){
//            //default value
//            rowElementName = "row";             
//        }	    
//	}
	
	private static OMElement getSelectResult(OMElement queryElement, HashMap inputValues,
			HashMap params, HashMap paramOrder, AxisService axisService, boolean isDML)
	throws AxisFault {
		OMElement resultElement = null;
		Connection conn = null;
		String sqlQuery = queryElement.getFirstChildWithName(new QName("sql")).getText();
		try {
			Parameter dbConnectionParam = axisService.getParameter(DBConstants.DB_CONNECTION);  
			if(dbConnectionParam == null){
				throw new AxisFault("Database connection not found in Axis Configuration");
			}
			conn = (Connection)dbConnectionParam.getValue();
			conn.setAutoCommit(false);
			PreparedStatement preparedStatement =
				getProcessedPreparedStatement(inputValues, params, paramOrder, conn, sqlQuery);
			int responseCode = -1;
			ResultSet rs = null;
			if (isDML) {
				responseCode = preparedStatement.executeUpdate();
			} else {
				rs = preparedStatement.executeQuery();
			}

			OMElement result = queryElement.getFirstChildWithName(new QName("result"));
			String wrapperElementName = result.getAttributeValue(new QName("element"));
			String rowElementName = result.getAttributeValue(new QName("rowName"));
			
			//if wrapper element || row element is not set, set default values to them
	        if(wrapperElementName == null || wrapperElementName.trim().length() == 0){
	            //default value
	            wrapperElementName = "results";             
	        }
	        if(rowElementName == null || rowElementName.trim().length() == 0){
	            //default value
	            rowElementName = "row";             
	        } 
	        
	        //TODO : need to define a way to get values automatically
			//getDefaultValuesForResultsWrapper(wrapperElementName,rowElementName);

			// check ??
			String columnDefalut = result.getAttributeValue(new QName("columnDefault"));

			OMFactory fac = OMAbstractFactory.getOMFactory();
			OMNamespace omNs = fac.createOMNamespace(axisService.getTargetNamespace(), "data");
			resultElement = fac.createOMElement(wrapperElementName, omNs);

			if (isDML) {
				resultElement
				.setText("Your query executed successfully. Return code from the database was "
						+ responseCode);
			} else {
				//check if wrapper element should be qualified
				if ( !axisService.isElementFormDefault()) {
					omNs =fac.createOMNamespace("", "data"); 
				}
				while (rs.next()) {
					HashMap elementValues = new HashMap();
					int columnCount = rs.getMetaData().getColumnCount();

					OMElement row = fac.createOMElement(rowElementName, omNs);
					if (rowElementName == null) {
						row = resultElement;
					}
					for (int i = 1; i <= columnCount; i++) {
						String columnName = rs.getMetaData().getColumnLabel(i);
						String columnValue = rs.getString(columnName);
						elementValues.put(columnName, columnValue);
					}

					Iterator elements = result.getChildElements();
					boolean useAsParamToNextQuery = false;
					while (elements.hasNext()) {
						OMElement element = (OMElement) elements.next();
						if (element.getLocalName().equals("element")) {
							String displayTagName = element.getAttributeValue(new QName("name"));
							String resultSetFieldName =
								element.getAttributeValue(new QName("column"));

							// This means,the parameter is not part of the
							// resultset. i.e. it is being passed from user's
							// parameters.
							if (resultSetFieldName == null) {
								resultSetFieldName = element.getAttributeValue(new QName("param"));
								useAsParamToNextQuery = true;
							}
							String elementValue;

							if (useAsParamToNextQuery) {
								elementValue = (String) params.get(resultSetFieldName);
								elementValues.put(resultSetFieldName, elementValue);
							} else {
								elementValue = (String) elementValues.get(resultSetFieldName);
							}

							if (columnDefalut == null || columnDefalut.equals("element")) {
								OMElement rowElement = fac.createOMElement(displayTagName, omNs);
								rowElement.addChild(fac.createOMText(rowElement, elementValue));
								row.addChild(rowElement);
							} else if (columnDefalut.equals("attribute")) {
								row.addAttribute(displayTagName, elementValue, omNs);
							}
						} else if (element.getLocalName().equals("call-query")) {
							OMElement rowElement = getRDBMSResult(element, axisService, elementValues);
							row.addChild(rowElement);
						}
					}
					if (rowElementName != null) {
						resultElement.addChild(row);
					}
				}
			}

		} catch (SQLException e) {
		    log.error("Exception occurred while trying to execute the SQL statement : "+sqlQuery, e);
			throw new AxisFault("Exception occurred while trying to execute the SQL statement : "+sqlQuery, e);
		} 
		
		
//		finally {
//		try {
//		if (conn != null) {
//		conn.commit();
//		conn.close();
//		}
//		} catch (SQLException e) {
//		log.error(e.getMessage());
//		throw new AxisFault("Exception occurred while trying to commit.", e);
//		}
//		}
		return resultElement;
	}

	public static OMElement invoke(MessageContext msgContext) throws AxisFault{
		AxisOperation axisOp = msgContext.getAxisOperation();
		AxisService axisService = msgContext.getAxisService();
		OMElement inputMessage = msgContext.getEnvelope().getBody().getFirstElement();
		OMElement operationElement =
			(OMElement) (axisOp.getParameterValue(DBConstants.DB_OPERATION_ELEMENT));

		if(DBConstants.DATASOURCE_TYPE_CSV.equals((String)axisService.getParameterValue(DBConstants.DATASOURCE_TYPE))){
			return DBUtils.processCSVQuery(operationElement, axisService, inputMessage);
		}
		else if(DBConstants.DATASOURCE_TYPE_EXCEL.equals((String)axisService.getParameterValue(DBConstants.DATASOURCE_TYPE))){
			return processExcelQuery(operationElement, axisService, inputMessage);
		}
		else{//default is RDBMS (JNDI Data source falls under this too)
			return DBUtils.processSQLQuery(operationElement, axisService, inputMessage);
		}		
	}

	/*
	 * Check for CSV & Excel reader library availability
	 */

	private static boolean checkLibraryAvailability(String libraryName) throws AxisFault{
		try{
			if("POI".equals(libraryName)){
				Class.forName("org.apache.poi.poifs.filesystem.POIFSFileSystem");
			}else if("OpenCSV".equals(libraryName)){
				Class.forName("au.com.bytecode.opencsv.CSVReader");
			}
			return true;
			
		}catch(ClassNotFoundException e){
		    log.error("Library not found for : "+libraryName,e);
			throw new AxisFault("Library not found for : "+libraryName,e);
		}
	}
	private synchronized static OMElement processExcelQuery(OMElement operationElement, AxisService axisService,
			OMElement inputMessage) throws AxisFault{

		String excelFilePath = (String)axisService.getParameterValue(DBConstants.EXCEL_DATASOURCE);
		log.info("Using Excel file from : "+excelFilePath);		
		
		InputStream dataSourceInputStream = null;
		try 
		{
			//check for POI library
			checkLibraryAvailability("POI");
			
			POIFSFileSystem fs;	
			HSSFWorkbook wb;
			
			if(excelFilePath.startsWith("http://")){
				//This is a url file path
				URL url = new URL(excelFilePath);
				dataSourceInputStream = url.openStream();				
			}else{
				dataSourceInputStream = new FileInputStream(excelFilePath);
			}
			fs = new POIFSFileSystem(dataSourceInputStream);
			wb = new HSSFWorkbook(fs);

			OMElement callQueryElement =
				operationElement.getFirstChildWithName(new QName("call-query"));
			// Find the corresponding query element
			String href = callQueryElement.getAttributeValue(new QName("href"));
			OMElement queryElement;
			if (href != null) {
				HashMap queries =
					(HashMap) axisService.getParameterValue(DBConstants.DB_QUERY_ELEMENTS);
				queryElement = (OMElement) queries.get(href);
			} else {
				queryElement = callQueryElement.getFirstChildWithName(new QName("query"));
			}			
			return getExcelResult(wb,queryElement,axisService);

		}catch (FileNotFoundException e1) {
			log.error("Excel file not fould : "+excelFilePath, e1);
			throw new AxisFault("Excel file not fould : "+excelFilePath);
		}
		catch (IOException e) {
		    log.error("Error loading Excel file : "+excelFilePath, e);
		    throw new AxisFault("Error loading Excel file : "+excelFilePath);
		}finally{
			if(dataSourceInputStream != null){
				try {
					dataSourceInputStream.close();
				} catch (IOException e) {
					log.debug("Error occured while close InputStream for : "+excelFilePath, e);
				}
			}
		}
	}


	private static OMElement getExcelResult(HSSFWorkbook wb
			,OMElement queryElement
			, AxisService axisService)throws AxisFault {
		OMElement resultElement = null;

		//OMElement excelElement = queryElement.getFirstChildWithName(new QName("excel"));
		ExcelQuery excelQuery = new ExcelQuery(axisService,queryElement);

		OMElement result = queryElement.getFirstChildWithName(new QName("result"));
		String wrapperElementName = result.getAttributeValue(new QName("element"));
		String rowElementName = result.getAttributeValue(new QName("rowName"));
		String columnDefalut = result.getAttributeValue(new QName("columnDefault"));

		OMFactory fac = OMAbstractFactory.getOMFactory();
		OMNamespace omNs = fac.createOMNamespace(axisService.getTargetNamespace(), "data");
		resultElement = fac.createOMElement(wrapperElementName, omNs);

		if ( !axisService.isElementFormDefault()) {
			omNs = fac.createOMNamespace("", "data"); 
		}

		//Retrieve the sheet name, if user has set it in configuration file
		String sheetName = excelQuery.getWorkBookName();

		//this is used to skip header columns, the spread sheet
		int startReadingFromRow = excelQuery.getStartingRow();
		if(startReadingFromRow >= 0){
		    //rows start from 0
		    startReadingFromRow = startReadingFromRow -1;
		}
		int maxRowCount = excelQuery.getMaxRowCount();

		HSSFSheet sheet = wb.getSheet(sheetName);
		int rowCount  = sheet.getPhysicalNumberOfRows();

		//If hasHeaders is set to true, we need first row object in later stage.
		HSSFRow firstRow = null;
		if(excelQuery.hasHeaders()){
			//assumption : first row is the header row
			firstRow = sheet.getRow(0);					
		}
		
		int processedRowCount = 0;		
		for (int r = 0; r < rowCount; r++)
		{
			if( r >= startReadingFromRow)
			{
				if(processedRowCount == maxRowCount){
					break;
				}
				HSSFRow hssfRow = sheet.getRow(r);
				OMElement row = fac.createOMElement(rowElementName, omNs);
				if (rowElementName == null) {
					row = resultElement;
				}

				Iterator elements = result.getChildElements();
				while (elements.hasNext()) {
					OMElement element = (OMElement) elements.next();
					if (element.getLocalName().equals("element")) {
						String displayTagName = element.getAttributeValue(new QName("name"));
						String columnValue = element.getAttributeValue(new QName("column"));

						short a = 1;
						short columnNumber;
						if(excelQuery.hasHeaders()){
							//if hasHeaders is set to true, column Names should be specified
							//get the column number using specified name
							columnNumber = getExcelColumnNumber(columnValue,firstRow);
						}else{
							try{
								columnNumber = (short)(Short.valueOf(columnValue).shortValue()- a);	
							}catch(NumberFormatException e){
							    log.error("Column value for element : "+displayTagName +" should be a number.",e);
								throw new AxisFault("Column value for element : "+displayTagName +" should be a number.");
							}
														
						}
						
						HSSFCell hssfCell = hssfRow.getCell(columnNumber);
						String elementValue = "";
						if(hssfCell != null){
							if (HSSFCell.CELL_TYPE_STRING == hssfCell.getCellType()){
								elementValue = hssfCell.getRichStringCellValue().getString();
							}else if(HSSFCell.CELL_TYPE_BLANK == hssfCell.getCellType()){
								//do nothing
							}else if(HSSFCell.CELL_TYPE_BOOLEAN == hssfCell.getCellType()){
								elementValue = String.valueOf(hssfCell.getBooleanCellValue());
							}else if(HSSFCell.CELL_TYPE_FORMULA == hssfCell.getCellType()){
								elementValue = "{formula}";
							}else if(HSSFCell.CELL_TYPE_NUMERIC == hssfCell.getCellType()){
								elementValue = String.valueOf(hssfCell.getNumericCellValue());
							}
						}

						if (columnDefalut == null || columnDefalut.equals("element")) {
							OMElement rowElement = fac.createOMElement(displayTagName, omNs);
							rowElement.addChild(fac.createOMText(rowElement, elementValue));
							row.addChild(rowElement);
						} else if (columnDefalut.equals("attribute")) {
							row.addAttribute(displayTagName, elementValue, omNs);
						}
					}
				}
				if (rowElementName != null) {
					resultElement.addChild(row);
					processedRowCount++;
				}							
			}//end of if( k >= startReadingFromRow)
		}//for (int r = 0; r < rowCount; r++)
		return resultElement;
	}
	
	private static short getExcelColumnNumber(String columnName,HSSFRow headerRow)
	throws AxisFault{
		int noOfCells = headerRow.getPhysicalNumberOfCells();
		short columnNo = -1;
		for(int a = 0;a <noOfCells;a++){
			HSSFCell cell = headerRow.getCell((short)a);
			if(HSSFCell.CELL_TYPE_STRING == cell.getCellType()){
				if(columnName.equals(cell.getRichStringCellValue().getString())){
					columnNo = (short)a;
					break;
				}				
			}else if(HSSFCell.CELL_TYPE_NUMERIC == cell.getCellType()){
				try{
					double columnNameInDouble = Double.valueOf(columnName).doubleValue();
					if(columnNameInDouble == cell.getNumericCellValue()){
						columnNo = (short)a;
						break;
					}
				}catch(NumberFormatException e){
					log.error("Numeric value expected for Column Name : "+columnName, e);
					throw new AxisFault("Numeric value expected for Column Name : "+columnName, e);
				}				
			}
		}
		return columnNo;
	}

	private synchronized static OMElement processCSVQuery(
			OMElement operationElement, AxisService axisService,
			OMElement inputMessage) throws AxisFault{

		String csvFilePath = (String) axisService
				.getParameterValue(DBConstants.CSV_DATASOURCE);
		log.info("Using CSV file from : " + csvFilePath);

		CSVReader reader = null;
		InputStreamReader inputStreamReader = null;
		try {
			if (csvFilePath.startsWith("http://")) {
				// This is a url file path
				URL url = new URL(csvFilePath);
				inputStreamReader = new InputStreamReader(url.openStream());
				reader = new CSVReader(inputStreamReader);
			} else {
			    if(csvFilePath.startsWith("."+File.separator)
			            || csvFilePath.startsWith(".."+File.separator)){
			        //this is relative path
			        File csvFile = new File(csvFilePath);
			        csvFilePath = csvFile.getAbsolutePath();
			        log.info("relative file path reference found. Using "+csvFilePath+" as absolute path.");
			    }
				// local file
				reader = new CSVReader(new FileReader(csvFilePath));
			}

			OMElement callQueryElement = operationElement
					.getFirstChildWithName(new QName("call-query"));
			// Find the corresponding query element
			String href = callQueryElement.getAttributeValue(new QName("href"));
			OMElement queryElement;
			if (href != null) {
				HashMap queries = (HashMap) axisService
						.getParameterValue(DBConstants.DB_QUERY_ELEMENTS);
				queryElement = (OMElement) queries.get(href);
			} else {
				queryElement = callQueryElement
						.getFirstChildWithName(new QName("query"));
			}

			try {
				return getCSVResult(reader, queryElement, axisService);
			} catch (AxisFault e) {
			    throw e;
			    /*
				OMElement resultElement = null;
				OMFactory fac = OMAbstractFactory.getOMFactory();
				OMNamespace omNs = fac.createOMNamespace(axisService
						.getTargetNamespace(), "data");
				resultElement = fac.createOMElement("csvoutput", omNs);
				resultElement.setText(e.getMessage());
				return resultElement;
				*/
			}
		} catch (FileNotFoundException e) {
		    log.error("CSV file not found : "+csvFilePath,e);
			throw new AxisFault("CSV file not found : "+csvFilePath,e);
		} catch (MalformedURLException e) {
		    log.error("Incorrect URL : "+csvFilePath,e);
		    throw new AxisFault("Incorrect URL : "+csvFilePath,e);
		} 
		catch (IOException e) {
		    log.error("Error opening stream for : "+csvFilePath, e);
		    throw new AxisFault("Error opening stream for : "+csvFilePath, e);
		}
		finally{
			if(inputStreamReader != null){
				try {
					inputStreamReader.close();
				} catch (IOException e) {
					e.printStackTrace();
				}
			}		
		}
	}

	private static OMElement getCSVResult(CSVReader reader,OMElement queryElement, AxisService axisService)
	throws AxisFault {
		OMElement resultElement = null;
		//Extract CSV Query Parameters
		CSVQuery csvQuery = new CSVQuery(axisService);			

		//compute number of rows to read
		int start = csvQuery.getStartingRow();
		int end = csvQuery.getMaxRowCount();
		int noOfRowsToRead = 0;

		if(end == -1){
			noOfRowsToRead = -1;
		}else{
			noOfRowsToRead = end;
		}
		
		if(start >= 0){
		    start = start -1;
		}

		//Extract how result to be formatted
		OMElement result = queryElement.getFirstChildWithName(new QName("result"));
		String wrapperElementName = result.getAttributeValue(new QName("element"));
		String rowElementName = result.getAttributeValue(new QName("rowName"));
		String columnDefalut = result.getAttributeValue(new QName("columnDefault"));

		OMFactory fac = OMAbstractFactory.getOMFactory();
		OMNamespace omNs = fac.createOMNamespace(axisService.getTargetNamespace(), "data");
		resultElement = fac.createOMElement(wrapperElementName, omNs);

		if ( !axisService.isElementFormDefault()) {
			omNs = fac.createOMNamespace("", "data"); 
		}
		String [] nextLine;
		int[] columnNumbersToRead = null;

		//if hasHeader parameter is set, read column names
		if(csvQuery.hasHeaders()){
		//if csvQuery.hasHeaders() is true, what is read is column name
		//if not column number should be read
			try {
				nextLine = reader.readNext();
				columnNumbersToRead = new int[csvQuery.getColumnNames().length];
				//iterate through all column headers, if column name specified in the configuration
				//is found, add that column to columnNumbersToRead array.
				//This will be used when mapping columns to result later
				int index = 0;
				for (int b = 0; b < nextLine.length; b++) {
					//'realColumnValue' is the column name specified in CSV file
					String realColumnName = nextLine[b]; 
					if(csvQuery.isColumnNameMentioned(realColumnName)){
						columnNumbersToRead[index] = b;
						index++;
					}
				}				

			} catch (IOException e) {
			    log.error("Error reading header record",e);
				throw new AxisFault("Error reading header record",e);
			}
		}else{
			//if headers are not present, column headers must be specified as
			// numbers
			columnNumbersToRead = new int[csvQuery.getColumnNames().length];
			String colNo = "";
			for(int a = 0;a < csvQuery.getColumnNames().length;a++){
				try{
					colNo = csvQuery.getColumnNames()[a];
					columnNumbersToRead[a] =  Integer.valueOf(colNo).intValue();	
				}catch (NumberFormatException e){
				    log.error("When headers are set to false, you need to have " +
                            "numeric value for column.Incorrect value found : "+colNo,e);
					throw new AxisFault("When headers are set to false, you need to have " +
							"numeric value for column.Incorrect value found : "+colNo,e);
				}
			}
		}

		try {
			int processedRowCount = 0;
			while ((nextLine = reader.readNext()) != null) {
				//work-a-round for lines with spaces
				//Array size for this line is 1 (i.e no separators/commas)
				if(nextLine.length == 1){
					if(nextLine[0] == null || nextLine[0].trim().equals("")){
						continue;						
					}					
				}
				if(noOfRowsToRead != -1){
					if((processedRowCount+start) == (noOfRowsToRead+start)){
						break;
					}
				}
				if(start > (processedRowCount + 1)){
					processedRowCount++;
					continue;
				}				
				
				HashMap elementValues = new HashMap();
				int columnCount = csvQuery.getColumnNames().length;

				OMElement row = fac.createOMElement(rowElementName, omNs);
				if (rowElementName == null) {
					row = resultElement;
				}
				for (int i = 0; i < columnCount; i++) {
					int columnNo = columnNumbersToRead[i];	
					String columnValue = "";
					try{
						columnValue = nextLine[columnNo];						
					}catch(ArrayIndexOutOfBoundsException e){
						log.info("Reading row : "+(processedRowCount+1) +".No value found " +
								"for column no : "+columnNo+".Empty value will be printed.",e);
					}					
					elementValues.put(new Integer(columnNo), columnValue);
				}

				Iterator elements = result.getChildElements();
				while (elements.hasNext()) {
					OMElement element = (OMElement) elements.next();
					if (element.getLocalName().equals("element")) {
						String displayTagName = element.getAttributeValue(new QName("name"));
						String columnReference =
							element.getAttributeValue(new QName("column"));
						
						int columnRef = 0;
						try{
							columnRef = Integer.valueOf(columnReference).intValue();	
						}catch(NumberFormatException e){
						    log.error("Column value specified "+ columnReference
                                    + " for "+displayTagName+" should be a numeric value.",e);
							throw new AxisFault("Column value specified "+ columnReference
									+ " for "+displayTagName+" should be a numeric value.",e);
						}
						
						//in dbs file column number starts from 1, but inside code it starts from 0
						String elementValue = (String) elementValues.get(new Integer(columnRef - 1));

						if (columnDefalut == null || columnDefalut.equals("element")) {
							OMElement rowElement = fac.createOMElement(displayTagName, omNs);
							rowElement.addChild(fac.createOMText(rowElement, elementValue));
							row.addChild(rowElement);
						} else if (columnDefalut.equals("attribute")) {
							row.addAttribute(displayTagName, elementValue, omNs);
						}
					}
					//we cannot support this with single data source
					//else if (element.getLocalName().equals("call-query")) {
					//	OMElement rowElement = getResult(element, axisService, elementValues);
					//	row.addChild(rowElement);
					//}
				}
				if (rowElementName != null) {
					resultElement.addChild(row);
				}
				processedRowCount++;
			}
		}catch (IOException e) {
			log.error("Error reading CSV file.",e);
			throw new AxisFault("Error reading CSV file.",e);
		}
		return resultElement;
	}


	public static Connection createConnection(AxisService axisService) throws AxisFault {
		try{
			log.info("Creating database connection for "+axisService.getName());
			//Try to load the JDBC driver class. If class not found throw an error.
			Class.forName((String) axisService.getParameterValue(DBConstants.DRIVER)).newInstance();
			Connection conn ;
			Properties props = new Properties();
			props.put("user", axisService.getParameterValue(DBConstants.USER));
			props.put("password", axisService.getParameterValue(DBConstants.PASSWORD));

			conn = DriverManager.getConnection((String) axisService
					.getParameterValue(DBConstants.PROTOCOL), props);
			return conn;
		} catch (SQLException e) {
			log.error("Error occured while connecting to database",e);
			throw new AxisFault("Error occured while connecting to database",e);
			
		} catch (ClassNotFoundException e) {
			log.error("JDBC driver not available in classpath : "+e.getMessage());
			throw new AxisFault("JDBC driver not available in classpath : "+e.getMessage());
		}
		catch (InstantiationException e) {
			log.error("Error occurred during instantiating "+(String) axisService.getParameterValue(DBConstants.DRIVER), e);
			throw new AxisFault("Error occurred during instantiating "+(String) axisService.getParameterValue(DBConstants.DRIVER), e);
		} 
		catch (IllegalAccessException e) {
			log.error("Illegal Access error during loading "+(String) axisService.getParameterValue(DBConstants.DRIVER), e);
			throw new AxisFault("Illegal Access error during loading "+(String) axisService.getParameterValue(DBConstants.DRIVER), e);
		}
	}

	public static PreparedStatement getProcessedPreparedStatement(HashMap inputs, HashMap params,
			HashMap paramOrder, Connection conn, String sqlStatement) throws AxisFault {
	    
        String paramName = null;
        String sqlType = null;
        String value = null;

	    try{
	    
		PreparedStatement sqlQuery = conn.prepareStatement(sqlStatement);

		Set paramNames = params.keySet();
		Object pramNameArray[] = paramNames.toArray();

		for (int i = 0; i < pramNameArray.length; i++) {
			paramName = (String) paramOrder.get(new Integer(i + 1));
			sqlType = (String) params.get(paramName);
			value = (String) inputs.get(paramName);

			if (sqlType == null) {
				// Defaults to string
				sqlQuery.setString(i + 1, value);
			} else if (sqlType.equals("INTEGER")) {
				sqlQuery.setInt(i + 1, Integer.parseInt(value));
			} else if (sqlType.equals("STRING")) {
			    if(value == null || value.trim().length() == 0){
		            log.error("Empty value found for parameter : "+paramName);
		            throw new AxisFault("Empty value found for parameter : "+paramName);			        
			    }
				sqlQuery.setString(i + 1, value);
			} else if (sqlType.equals("DOUBLE")) {
				sqlQuery.setDouble(i + 1, Double.parseDouble(value));
			}else if(sqlType.equals("DATE")){
			    try{
			        sqlQuery.setDate(i+1, Date.valueOf(value));
			    }catch(IllegalArgumentException e){
		            log.error("Incorrect date format for parameter : "+paramName, e);
		            throw new AxisFault("Incorrect date format for parameter  : "
		                    +paramName+".Date should be in yyyy-mm-dd format.", e);			        
			    }			    
			}
			
		}
		return sqlQuery;
	    }
	    catch(NumberFormatException e){
	        log.error("Incorrect value found for parameter : "+paramName, e);
	        throw new AxisFault("Incorrect value found for parameter : "+paramName, e);
	    }catch(SQLException e){
	        log.error("Error occurred while preparing prepared statement for sql : "+sqlStatement, e);
	        throw new AxisFault("Error occurred while preparing prepared statement for sql : "+sqlStatement, e); 
	    }
	}
}
