/*
 * Decompiled with CFR 0.152.
 */
package org.wso2.carbon.deployment.engine.internal;

import java.io.File;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.wso2.carbon.deployment.engine.Artifact;
import org.wso2.carbon.deployment.engine.ArtifactType;
import org.wso2.carbon.deployment.engine.Deployer;
import org.wso2.carbon.deployment.engine.LifecycleEvent;
import org.wso2.carbon.deployment.engine.LifecycleListener;
import org.wso2.carbon.deployment.engine.config.DeploymentConfiguration;
import org.wso2.carbon.deployment.engine.exception.CarbonDeploymentException;
import org.wso2.carbon.deployment.engine.exception.DeployerRegistrationException;
import org.wso2.carbon.deployment.engine.exception.DeploymentEngineException;
import org.wso2.carbon.deployment.engine.internal.DataHolder;
import org.wso2.carbon.deployment.engine.internal.RepositoryScanner;
import org.wso2.carbon.deployment.engine.internal.SchedulerTask;

public class DeploymentEngine {
    private static final Logger logger = LoggerFactory.getLogger(DeploymentEngine.class);
    private RepositoryScanner repositoryScanner;
    private File serverRepositoryDirectory;
    private File runtimeRepositoryDirectory;
    private Map<ArtifactType, Deployer> deployerMap = new ConcurrentHashMap<ArtifactType, Deployer>();
    private List<LifecycleListener> lifecycleListeners = new ArrayList<LifecycleListener>();
    private Map<ArtifactType, ConcurrentHashMap<Object, Artifact>> deployedArtifacts = new ConcurrentHashMap<ArtifactType, ConcurrentHashMap<Object, Artifact>>();
    private Map<String, Artifact> faultyArtifacts = new ConcurrentHashMap<String, Artifact>();

    public DeploymentEngine() {
        this.repositoryScanner = new RepositoryScanner(this);
    }

    public void start(String serverRepositoryDir, String runtimeRepositoryDir) throws DeploymentEngineException {
        this.serverRepositoryDirectory = new File(serverRepositoryDir);
        this.runtimeRepositoryDirectory = new File(runtimeRepositoryDir);
        if (!this.serverRepositoryDirectory.exists()) {
            throw new DeploymentEngineException("Cannot find repository : " + this.serverRepositoryDirectory);
        }
        if (!this.runtimeRepositoryDirectory.exists()) {
            throw new DeploymentEngineException("Cannot find repository : " + this.runtimeRepositoryDirectory);
        }
        if (logger.isDebugEnabled()) {
            logger.debug("Starting carbon deployment engine for repository : " + serverRepositoryDir);
        }
        this.repositoryScanner.scan();
        this.startScheduler();
    }

    private void startScheduler() throws DeploymentEngineException {
        ScheduledExecutorService executorService = Executors.newScheduledThreadPool(5);
        SchedulerTask schedulerTask = new SchedulerTask(this.repositoryScanner);
        DeploymentConfiguration deploymentConfiguration = DataHolder.getInstance().getDeploymentConfiguration();
        int interval = 15;
        if (deploymentConfiguration != null) {
            interval = deploymentConfiguration.getUpdateInterval();
            logger.debug("Using the specified scheduler update interval of {}", (Object)interval);
        } else {
            logger.debug("Using the default deployment scheduler update interval of 15 seconds");
        }
        executorService.scheduleWithFixedDelay(schedulerTask, 0L, interval, TimeUnit.SECONDS);
    }

    public void registerDeployer(Deployer deployer) throws DeployerRegistrationException {
        if (deployer == null) {
            throw new DeployerRegistrationException("Failed to add Deployer : Deployer Class Name is null");
        }
        logger.debug("Registering deployer instance {} with deployment engine", (Object)deployer.getClass().getName());
        deployer.init();
        if (deployer.getLocation() == null) {
            throw new DeployerRegistrationException("Failed to add Deployer " + deployer.getClass().getName() + " : missing 'directory' attribute in deployer instance");
        }
        ArtifactType type = deployer.getArtifactType();
        if (type == null) {
            throw new DeployerRegistrationException("Artifact Type for Deployer : " + deployer + " is null");
        }
        Deployer existingDeployer = this.deployerMap.get(type);
        if (existingDeployer == null) {
            this.deployerMap.put(type, deployer);
        }
    }

    public void unregisterDeployer(Deployer deployer) throws DeploymentEngineException {
        logger.debug("Un-registering deployer instance {} from deployment engine", (Object)deployer.getClass().getName());
        ArtifactType type = deployer.getArtifactType();
        if (type == null) {
            throw new DeploymentEngineException("Artifact Type for Deployer : " + deployer + " is null");
        }
        Deployer existingDeployer = this.deployerMap.get(type);
        if (existingDeployer != null) {
            this.deployerMap.remove(type);
        }
    }

    public Deployer getDeployer(ArtifactType type) {
        return this.deployerMap.get(type);
    }

    public Map<ArtifactType, Deployer> getDeployers() {
        return this.deployerMap;
    }

    public void registerDeploymentLifecycleListener(LifecycleListener lifecycleListener) {
        logger.debug("Registering deployment listener instance {} with deployment engine", (Object)lifecycleListener.getClass().getName());
        this.lifecycleListeners.add(lifecycleListener);
    }

    public void unregisterDeploymentLifecycleListener(LifecycleListener lifecycleListener) {
        logger.debug("Un-registering deployment listener instance {} from deployment engine", (Object)lifecycleListener.getClass().getName());
        this.lifecycleListeners.remove(lifecycleListener);
    }

    public File getServerRepositoryDirectory() {
        return this.serverRepositoryDirectory;
    }

    public File geRuntimeRepositoryDirectory() {
        return this.runtimeRepositoryDirectory;
    }

    public Artifact getDeployedArtifact(ArtifactType type, Object artifactKey) {
        Artifact artifact = null;
        if (this.deployedArtifacts.get(type) != null) {
            artifact = this.deployedArtifacts.get(type).get(artifactKey);
        }
        return artifact;
    }

    public Map<ArtifactType, ConcurrentHashMap<Object, Artifact>> getDeployedArtifacts() {
        return this.deployedArtifacts;
    }

    public Map<String, Artifact> getFaultyArtifacts() {
        return this.faultyArtifacts;
    }

    public void deployArtifacts(List<Artifact> artifactsToDeploy) {
        artifactsToDeploy.forEach(artifactToDeploy -> {
            LifecycleEvent lifecycleEvent = new LifecycleEvent((Artifact)artifactToDeploy, new Date(), LifecycleEvent.STATE.BEFORE_START_EVENT);
            try {
                Object artifactKey;
                this.fireLifecycleEvent(lifecycleEvent);
                Deployer deployer = this.getDeployer(artifactToDeploy.getType());
                if (deployer != null) {
                    logger.debug("Deploying artifact {} using {} deployer", (Object)artifactToDeploy.getName(), (Object)deployer.getClass().getName());
                    artifactKey = deployer.deploy((Artifact)artifactToDeploy);
                    if (artifactKey == null) {
                        throw new CarbonDeploymentException("Deployed artifact key is null for : " + artifactToDeploy.getName());
                    }
                } else {
                    throw new CarbonDeploymentException("Deployer instance cannot be found for the type : " + artifactToDeploy.getType());
                }
                artifactToDeploy.setKey(artifactKey);
                this.addToDeployedArtifacts((Artifact)artifactToDeploy);
            }
            catch (CarbonDeploymentException e) {
                logger.error("Error while deploying artifacts", e);
                this.addToFaultyArtifacts((Artifact)artifactToDeploy);
                StringWriter sw = new StringWriter();
                e.printStackTrace(new PrintWriter(sw));
                lifecycleEvent.setTraceContent("Error while deploying artifact. \n" + sw.toString());
                lifecycleEvent.setDeploymentResult(LifecycleEvent.RESULT.FAILED);
            }
            lifecycleEvent.setState(LifecycleEvent.STATE.AFTER_START_EVENT);
            this.fireLifecycleEvent(lifecycleEvent);
        });
    }

    private void fireLifecycleEvent(LifecycleEvent lifecycleEvent) {
        logger.debug("Triggering lifecycle event {} for artifact - {}", (Object)lifecycleEvent.getState(), (Object)lifecycleEvent.getArtifact().getName());
        this.lifecycleListeners.forEach(lifecycleListener -> lifecycleListener.lifecycleEvent(lifecycleEvent));
    }

    public void updateArtifacts(List<Artifact> artifactsToUpdate) {
        artifactsToUpdate.forEach(artifactToUpdate -> {
            LifecycleEvent lifecycleEvent = new LifecycleEvent((Artifact)artifactToUpdate, new Date(), LifecycleEvent.STATE.BEFORE_UPDATE_EVENT);
            try {
                Object artifactKey;
                this.fireLifecycleEvent(lifecycleEvent);
                Deployer deployer = this.getDeployer(artifactToUpdate.getType());
                if (deployer != null) {
                    logger.debug("Updating artifact {} using {} deployer", (Object)artifactToUpdate.getName(), (Object)deployer.getClass().getName());
                    artifactKey = deployer.update((Artifact)artifactToUpdate);
                    if (artifactKey == null) {
                        throw new CarbonDeploymentException("Deployed artifact key is null for : " + artifactToUpdate.getName());
                    }
                } else {
                    throw new CarbonDeploymentException("Deployer instance cannot be found for the type : " + artifactToUpdate.getType());
                }
                artifactToUpdate.setKey(artifactKey);
                this.addToDeployedArtifacts((Artifact)artifactToUpdate);
            }
            catch (CarbonDeploymentException e) {
                logger.error("Error while updating artifact", e);
                this.addToFaultyArtifacts((Artifact)artifactToUpdate);
                StringWriter sw = new StringWriter();
                e.printStackTrace(new PrintWriter(sw));
                lifecycleEvent.setTraceContent("Error while updating artifact. \n" + sw.toString());
                lifecycleEvent.setDeploymentResult(LifecycleEvent.RESULT.FAILED);
            }
            lifecycleEvent.setState(LifecycleEvent.STATE.AFTER_UPDATE_EVENT);
            this.fireLifecycleEvent(lifecycleEvent);
        });
    }

    private void addToDeployedArtifacts(Artifact artifact) {
        ConcurrentHashMap<Object, Artifact> artifactMap = this.deployedArtifacts.get(artifact.getType());
        if (artifactMap == null) {
            artifactMap = new ConcurrentHashMap();
        }
        artifactMap.put(artifact.getKey(), artifact);
        this.deployedArtifacts.put(artifact.getType(), artifactMap);
        this.faultyArtifacts.remove(artifact.getPath());
    }

    private void addToFaultyArtifacts(Artifact artifact) {
        this.faultyArtifacts.put(artifact.getPath(), artifact);
        this.removeFromDeployedArtifacts(artifact);
    }

    public void undeployArtifacts(List<Artifact> artifactsToUndeploy) {
        artifactsToUndeploy.forEach(artifactToUnDeploy -> {
            LifecycleEvent lifecycleEvent = new LifecycleEvent((Artifact)artifactToUnDeploy, new Date(), LifecycleEvent.STATE.BEFORE_STOP_EVENT);
            try {
                this.fireLifecycleEvent(lifecycleEvent);
                Deployer deployer = this.getDeployer(artifactToUnDeploy.getType());
                if (deployer == null) {
                    throw new CarbonDeploymentException("Deployer instance cannot be found for the type : " + artifactToUnDeploy.getType());
                }
                logger.debug("Undeploying artifact {} using {} deployer", (Object)artifactToUnDeploy.getName(), (Object)deployer.getClass().getName());
                deployer.undeploy(artifactToUnDeploy.getKey());
                this.removeFromDeployedArtifacts((Artifact)artifactToUnDeploy);
            }
            catch (CarbonDeploymentException e) {
                logger.error("Error while undeploying artifacts", e);
                this.addToFaultyArtifacts((Artifact)artifactToUnDeploy);
                StringWriter sw = new StringWriter();
                e.printStackTrace(new PrintWriter(sw));
                lifecycleEvent.setTraceContent("Error while undeploying artifact. \n" + sw.toString());
                lifecycleEvent.setDeploymentResult(LifecycleEvent.RESULT.FAILED);
            }
            lifecycleEvent.setState(LifecycleEvent.STATE.AFTER_STOP_EVENT);
            this.fireLifecycleEvent(lifecycleEvent);
        });
    }

    private void removeFromDeployedArtifacts(Artifact artifact) {
        Map artifactMap = this.deployedArtifacts.get(artifact.getType());
        if (artifactMap != null && artifactMap.containsKey(artifact.getKey())) {
            artifactMap.remove(artifact.getKey());
            if (artifactMap.isEmpty()) {
                this.deployedArtifacts.remove(artifact.getType());
            }
        }
    }
}

