/*
* Copyright 2005,2006 WSO2, Inc. http://wso2.com
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
*
*/

package org.wso2.throttle;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 *
 *
 */

public abstract class Caller {

    /** Log for this class    */
    private static Log log = LogFactory.getLog(Caller.class.getName());
    /** next access time     */
    private long nextAccessTime = 0;
    /** frist access time    */
    private long firstAccessTime = 0;
    /** The nextTimeWindow - begining of next unit time period- end of current unit time period  */
    private long nextTimeWindow = 0;
    /** The count to keep track number of request   */
    private int count = 0;
    /** The Object for used to lock in synchronizing  */
    final Object lock = new Object();
    /** The Id of caller */
    private Object ID;

    public Caller(Object ID) {
        this.ID = ID;
    }

    /**
     *
     * @return  return Id of caller
     */
    public Object getID() {
        return ID;
    }

    /**
     * Init access
     *
     * @param configurationIPBased -The Configuration for this caller
     * @param throttleContext      -The Throttle that caller having pass
     * @param currentTime          -The system current time
     * @throws ThrottleException
     */
    public void initAccess(CallerConfiguration configurationIPBased, ThrottleContext throttleContext, long currentTime) throws ThrottleException {
        if (!(configurationIPBased.getMaximumRequestPerUnitTime() == 0)) {
            synchronized (lock) {
                this.firstAccessTime = currentTime;
                if (this.nextTimeWindow != 0) {
                    throttleContext.removeCaller(new Long(nextTimeWindow));
                }
                this.nextTimeWindow = currentTime + configurationIPBased.getUnitTimeInMiliSecond();
            }
            throttleContext.addCaller(this);
        }
    }

    /**
     * To verify access if unit time has already not overred
     *
     * @param configurationIPBased -The Configuration for this caller
     * @param throttleContext      -The Throttle that caller having pass
     * @param currentTime          -The system current time
     * @return boolean          -The boolean value which say access will allolw or not
     * @throws ThrottleException
     */
    public boolean canAccessIfUnitTimeNotOver(CallerConfiguration configurationIPBased, ThrottleContext throttleContext, long currentTime) throws ThrottleException {
        boolean canAcess = false;
        if (!(configurationIPBased.getMaximumRequestPerUnitTime() == 0)) {
            synchronized (lock) {
                if (count <= configurationIPBased.getMaximumRequestPerUnitTime() - 1) {
                    canAcess = true;
                    count++;
                    // can complete access

                } else {
                    //else , if caller has not already prohibit
                    if (this.nextAccessTime == 0) {
                        //and if there is no prohibit time  period in configurationIPBased
                        if (configurationIPBased.getProhibitTimePeriod() == 0) {
                            //prohibit access untill unit time period is over
                            this.nextAccessTime = this.firstAccessTime +
                                    configurationIPBased.getUnitTimeInMiliSecond();
                        } else {
                            //if there is a prohibit time period in configuartion ,then
                            //set it as prohibit period
                            this.nextAccessTime = currentTime +
                                    configurationIPBased.getProhibitTimePeriod();
                        }
                        log.debug("Maximum Number of requests are reached :IP-" + ID);

                    } else {
                        // else , if the caller has already prohabit and prohabit
                        // time period has already overed
                        if (this.nextAccessTime
                                <= currentTime) {
                            this.nextAccessTime = 0;
                            canAcess = true;
                            count = 1;
                            this.firstAccessTime = currentTime;
                            if (this.nextTimeWindow != 0) {
                                throttleContext.removeCaller(new Long(nextTimeWindow));
                            }
                            this.nextTimeWindow = currentTime + configurationIPBased.getUnitTimeInMiliSecond();
                        } else {
                            log.debug("Prohibit period is not yet over :IP- " + ID);

                        }
                    }
                }
            }
        }
        return canAcess;
    }

    /**
     * To verify access if unit time has already overred
     *
     * @param configurationIPBased -The Configuration for this caller
     * @param throttleContext      -The Throttle that caller having pass
     * @param currentTime          -The system current time
     * @return boolean          -The boolean value which say access will allolw or not
     * @throws ThrottleException
     */
    public boolean canAccessIfUnitTimeOver(CallerConfiguration configurationIPBased, ThrottleContext throttleContext, long currentTime) throws ThrottleException {
        boolean canAcess = false;
        // if number of access for a unit time is less than MAX and
        // if the unit time period (session time) has just overed
        if (!(configurationIPBased.getMaximumRequestPerUnitTime() == 0)) {
            synchronized (lock) {
                if (this.count <= configurationIPBased.getMaximumRequestPerUnitTime() - 1) {
                    if (this.nextTimeWindow != 0) {
                        throttleContext.removeCaller(new Long(nextTimeWindow));
                    }
                    canAcess = true; // this is bounus access
                    //next time callers can access as a new one
                } else {
                    // if number of access for a unit time has just been greater than MAX
                    // now same as a new session
                    // OR
                    //  if caller in prohabit session  and prohabit period has just overed
                    if ((this.nextAccessTime == 0) ||
                            (this.nextAccessTime <= currentTime)) {
                        this.nextAccessTime = 0;
                        canAcess = true;
                        count = 1;// can access the system   and this is same as fristAccess
                        this.firstAccessTime = currentTime;
                        if (this.nextTimeWindow != 0) {
                            throttleContext.removeCaller(new Long(nextTimeWindow));
                        }
                        this.nextTimeWindow = currentTime + configurationIPBased.getUnitTimeInMiliSecond();

                    } else {
                        // if  caller in prohabit session  and prohabit period has not  overed
                        log.debug("Even unit time has overed , CallerIP in prohibit state :IP -" + ID);

                    }
                }
            }
        }
        return canAcess;

    }

    /**
     * Check whether that caller can access or not
     *
     * @param throttleContext -The Configuration for this caller
     * @return boolean          -The boolean value which say access will allolw or not
     * @throws ThrottleException
     */
    public boolean canAccess(ThrottleContext throttleContext, long currentTime) throws ThrottleException {
        boolean canAcess = false;
        ThrottleConfiguration throttleConfiguration = throttleContext.getThrottleConfiguration();
        CallerConfiguration ipBasedconfiguration = throttleConfiguration.getCallerConfiguration(getID());
        if (ipBasedconfiguration == null) {
            return false;
        }
        if (ipBasedconfiguration.getMaximumRequestPerUnitTime() < 0 || ipBasedconfiguration.getUnitTimeInMiliSecond() <= 0 || ipBasedconfiguration.getProhibitTimePeriod() < 0)
        {
            throw new ThrottleException("Invalid Throttle Configuration");
        }

        if (!(ipBasedconfiguration.getMaximumRequestPerUnitTime() == 0)) {
            // if caller access first time in his new session
            synchronized (lock) {
                if (this.firstAccessTime == 0) {
                    initAccess(ipBasedconfiguration, throttleContext, currentTime);
                }
                // if unit time period (session time) is not over
                if (nextTimeWindow > currentTime) {
                    canAcess = canAccessIfUnitTimeNotOver(ipBasedconfiguration, throttleContext, currentTime);
                } else {
                    canAcess = canAccessIfUnitTimeOver(ipBasedconfiguration, throttleContext, currentTime);
                }
            }
        }
        return canAcess;

    }

    /**
     * To get next time window
     *
     * @return long value of next time window
     */
    public long getNextTimeWindow() {
        return nextTimeWindow;
    }

    /**
     * get type of throttle that this caller belong
     */
    public abstract int getType();

}
