/*
* Copyright 2005,2006 WSO2, Inc. http://wso2.com
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
*
*/

package org.wso2.throttle.impl.ipbase;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.wso2.throttle.*;
import org.wso2.throttle.factory.CallerFactory;

/**
 *
 *
 */

public class IPBaseAccessController implements AccessController {

    /** Log for this class    */
    private static Log log = LogFactory.getLog(IPBaseAccessController.class.getName());
    private static final String ACCESS_DENIED =
            "You cannot access this service since you have exceeded the allocated quota.";

    /**
     * To check wheather caller can access
     *
     * @param throttleContext - current states of throttle - RunTime Data
     * @param callerID
     * @return boolean
     * @throws ThrottleException
     */
    public boolean canAccess(ThrottleContext throttleContext,
                             Object callerID) throws ThrottleException {

        ThrottleConfiguration throttleConfigurationBean = throttleContext.getThrottleConfiguration();
        //Meta-data about caller
        String remoteIP = null;
        if (callerID instanceof String) {
            remoteIP = (String) callerID;
        } else {
            log.debug("Caller ID should instance of String for IPBASE Throttle");
        }
        if (remoteIP == null) {
            log.warn("Caller IP not found!");
            return false;
        }
        CallerConfiguration configuration =
                throttleContext.getThrottleConfiguration().getCallerConfiguration(remoteIP);
        if (configuration == null) {
            log.warn("Throttle configuration not be found for IP address " + remoteIP);
            return false;
        }
        if (configuration.getAccessState() == ThrottleConstants.ACCESS_DENIED) {
            log.info(ACCESS_DENIED);
            return false;
        } else if (configuration.getAccessState() == ThrottleConstants.ACCESS_ALLOWED) {
            return true;
        } else if (configuration.getAccessState() == ThrottleConstants.ACCESS_CONTROLLED) {
            Caller caller = throttleContext.getCaller(remoteIP);
            if (caller == null) {
                //if caller has not already registered ,then create new caller description and
                //set it in throttle
                Object correctedEPR =
                         throttleConfigurationBean.getConfigurationKeyOfCaller(remoteIP);
                if (correctedEPR != null) {
                    caller = CallerFactory.createCaller(ThrottleConstants.IP_BASE, correctedEPR);
                }
            }
            if (caller != null) {
                long currentTime = System.currentTimeMillis();
                if (!caller.canAccess(throttleContext, currentTime)) {
                    //if current caller cannot access , then perform cleaning
                    log.info(ACCESS_DENIED);
                    throttleContext.processCleanList(currentTime);
                    return false;
                }  else {
                    log.debug("Access  from IP address " + remoteIP + "is successful.");
                    return true;
                }
            } else {
                log.warn("Caller IP not found! " + remoteIP);
                return false;
            }
        }
        return true;
    }
}
