/*
* Copyright 2005,2006 WSO2, Inc. http://wso2.com
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
*
*/

package org.wso2.throttle.impl.ipbase;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.wso2.throttle.*;

import java.util.*;

/**
 *
 *
 */

public class IPBaseThrottleContext implements ThrottleContext {

    /** Log for this class    */
    private static Log log = LogFactory.getLog(IPBaseThrottleContext.class.getName());
    /** The callersMap that have registered for a particular throttle     */
    private TreeMap callersMap;
    /** For mapping epr to timeStamp   */
    private HashMap keyToTimeStampMap;
    /** The Time which next cleaning for this trottle will have to take place       */
    private long nextCleanTime;
    /** The Object for used to lock in synchronizing     */
    final Object lock = new Object();
    /** The configuration of a throtle     */
    private ThrottleConfiguration throttleConfiguration;

    /**
     * To create a ThrottleContext need to provide a ThrottleConfiguration
     *
     * @param throttleConfiguration
     */
    public IPBaseThrottleContext(ThrottleConfiguration throttleConfiguration) {
        this.keyToTimeStampMap = new HashMap();
        this.callersMap = new TreeMap();
        this.nextCleanTime = 0;
        this.throttleConfiguration = throttleConfiguration;
    }

    /**
     * To get the ThrottleConfiguration
     *
     * @return ThrottleConfiguration
     */
    public ThrottleConfiguration getThrottleConfiguration() {
        return throttleConfiguration;
    }

    /**
     * To get a CallerIP Object
     *
     * @param ID
     * @return CallerIP
     */
    public Caller getCaller(Object ID) {
        String keyOfConfiguration = (String) throttleConfiguration.getConfigurationKeyOfCaller(ID);
        if (keyOfConfiguration != null) {
            synchronized (lock) {
                Long timeKeyOfCallerContext = (Long) keyToTimeStampMap.get(keyOfConfiguration);
                if (timeKeyOfCallerContext != null) {
                    Object callerObject = callersMap.get(timeKeyOfCallerContext);
                    if (callerObject != null) {
                        if (callerObject instanceof Caller) {
                            return (Caller) callerObject;
                        } else if (callerObject instanceof LinkedList) {
                            LinkedList callersWithSameTimeStampList = (LinkedList) callerObject;
                            for (Iterator iterator = callersWithSameTimeStampList.iterator(); iterator.hasNext();)
                            {
                                Caller caller = (Caller) iterator.next();
                                if (keyOfConfiguration.equals(throttleConfiguration.getConfigurationKeyOfCaller(caller.getID())))
                                {
                                    return caller;
                                }
                            }

                        }
                    }
                }

            }
        }
        return null;
    }

    /**
     * To set the ThrottleConfiguration
     *
     * @param throttleConfiguration
     */
    public void setThrottleConfiguration(ThrottleConfiguration throttleConfiguration) {
        this.throttleConfiguration = throttleConfiguration;
    }

    /**
     * setting callerContext - put callersMap Against  time and put time Against epr
     *
     * @param caller
     */
    public void addCaller(Caller caller) {

        String keyOfConfiguration = (String) throttleConfiguration.getConfigurationKeyOfCaller(caller.getID());
        if (keyOfConfiguration != null) {
            setCaller(caller, keyOfConfiguration);
        }
    }

    /**
     * helper method to set a CallerIP Object
     *
     * @param caller
     * @param keyOfConfiguration
     */
    private void setCaller(Caller caller, String keyOfConfiguration) {
        Long time = new Long(caller.getNextTimeWindow());
        synchronized (lock) {
            if (!callersMap.containsKey(time)) {
                callersMap.put(time, caller);
            } else {
                //if there are callersMap with same timewindow ,then use linkedList to hold those
                LinkedList callersWithSameTimeStampList = null;
                Object callerObject = callersMap.get(time);
                if (callerObject != null) {
                    if (callerObject instanceof Caller) {
                        callersWithSameTimeStampList = new LinkedList();
                        callersWithSameTimeStampList.add(callerObject);
                        callersWithSameTimeStampList.add(caller);
                        callersMap.remove(time);
                    } else if (callerObject instanceof LinkedList) {
                        callersWithSameTimeStampList = (LinkedList) callerObject;
                        callersWithSameTimeStampList.add(caller);
                    }
                    if (callersWithSameTimeStampList != null) {
                        callersMap.put(time, callersWithSameTimeStampList);
                    }
                }
            }
            //set Time Vs key
            keyToTimeStampMap.put(keyOfConfiguration, time);

        }
    }

    /**
     * removing caller by providing ID of the caller
     *
     * @param ID
     */
    public void removeCaller(Object ID) {
        if (ID instanceof Long) {
            removeCaller((Long) ID);
        } else {
            String keyOfConfiguration = (String) throttleConfiguration.getConfigurationKeyOfCaller(ID);
            Long time;
            if (keyOfConfiguration != null) {
                time = (Long) keyToTimeStampMap.get(keyOfConfiguration);
                removeCaller(time);
            }

        }
    }

    /**
     * To remove Caller by providing time key
     *
     * @param key
     */
    private void removeCaller(Long key) {
        synchronized (lock) {
            callersMap.remove(key);
        }
    }

    /**
     * processing cleaning list- only process callerContexts which unit time already had overed
     *
     * @param currentTime - Current System Time
     * @throws ThrottleException
     */

    public void processCleanList(long currentTime) throws ThrottleException {
        synchronized (lock) {
            if (currentTime > nextCleanTime) {
                SortedMap sortedMap = callersMap.headMap(new Long(currentTime));
                if (sortedMap != null && sortedMap.size() > 0) {
                    for (Iterator callersIterator = sortedMap.values().iterator(); callersIterator.hasNext();)
                    {
                        Object callerObject = callersIterator.next();
                        if (callerObject != null) {
                            if (callerObject instanceof Caller)
                            { // In the case nextAccessTime is unique for the caller
                                Caller caller = ((Caller) callerObject);
                                caller.canAccessIfUnitTimeOver(this.getThrottleConfiguration().getCallerConfiguration(caller.getID()), this, currentTime);
                            } else if (callerObject instanceof LinkedList)
                            { //In the case nextAccessTime of multiple callers are same
                                LinkedList callersWithSameTimeStampList = (LinkedList) callerObject;
                                for (Iterator iterator = callersWithSameTimeStampList.iterator(); iterator.hasNext();)
                                {
                                    Caller caller = (Caller) iterator.next();
                                    if (caller != null) {
                                        caller.canAccessIfUnitTimeOver(this.getThrottleConfiguration().getCallerConfiguration(caller.getID()), this, currentTime);
                                    }
                                }
                            }
                        }
                    }
                }
                nextCleanTime = currentTime + ThrottleConstants.DEFAULT_THROTTLE_CLEAN_PERIOD;
            }
        }
    }

    public int getType() {
        return ThrottleConstants.IP_BASE;
    }
}
