/*
* Copyright 2005,2006 WSO2, Inc. http://wso2.com
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
*
*/

package org.wso2.throttle.module.handler;


import org.apache.axis2.AxisFault;
import org.apache.axis2.context.ConfigurationContext;
import org.apache.axis2.context.MessageContext;
import org.apache.axis2.description.AxisOperation;
import org.apache.axis2.description.AxisService;
import org.apache.axis2.description.Parameter;
import org.apache.axis2.handlers.AbstractHandler;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.wso2.throttle.*;
import org.wso2.throttle.factory.AccessControllerFactory;


public abstract class ThrottleHandler extends AbstractHandler {

    /** Log for this class   */
    private static Log log = LogFactory.getLog(ThrottleHandler.class.getName());
    /** @return int - indicate type of throttle   */
    protected abstract int getThrottleType();

    /**
     * Load to throttle metadata for a particular throttle
     *
     * @param messageContext - The messageContext
     * @param throttleType   - The type of throttle
     * @return IPBaseThrottleConfiguration     - The IPBaseThrottleConfiguration - load from AxisConfiguration
     * @throws AxisFault
     */
    public Throttle loadThrottle(MessageContext messageContext,
                                 int throttleType) throws AxisFault, ThrottleException {

        Throttle throttle = null;
        ConfigurationContext configContext = messageContext.getConfigurationContext();
        //the Parameter which hold throttle ipbase object
        Parameter throttleObjectParameter;
        // to get parameter from configuration context
        switch (throttleType) {
            case ThrottleConstants.GLOBAL_THROTTLE: {
                throttleObjectParameter =
                        configContext.getAxisConfiguration().
                                getParameter(ThrottleConstants.GLOBAL_IP_BASED_THROTTLE_KEY);
                break;
            }
            case ThrottleConstants.OPERATION_BASED_THROTTLE: {
                AxisOperation axisOperation = messageContext.getAxisOperation();
                if (axisOperation != null) {
                    String currentServiceName = ((AxisService) axisOperation.getParent()).getName();
                    throttleObjectParameter =
                            configContext.getAxisConfiguration().
                                    getParameter(currentServiceName +
                                            axisOperation.getName().getLocalPart());
                } else {
                    throw new AxisFault("Axis operation cannot be null");
                }
                break;
            }
            case ThrottleConstants.SERVICE_BASED_THROTTLE: {
                AxisService axisService = messageContext.getAxisService();
                if (axisService != null) {
                    throttleObjectParameter =
                            configContext.getAxisConfiguration().getParameter(axisService.getName());
                } else {
                    throw new AxisFault("Axis service cannot be null");
                }
                break;
            }
            default : {
                throw new ThrottleException("Unsupported Throttle type");
            }
        }
        //if there is a throttlebean
        if (throttleObjectParameter != null) {
            Object throttleObject = throttleObjectParameter.getValue();
            if (throttleObject instanceof Throttle) {
                throttle = (Throttle) throttleObject;
            } else {
                throw new ThrottleException("Incompatible object for IPBaseThrottleConfiguration");
            }
        }
        return throttle;
    }

    /**
     * process throttle with messagecontext
     *
     * @param throttle
     * @param messageContext
     * @throws ThrottleException
     */
    public void process(Throttle throttle,
                        MessageContext messageContext) throws ThrottleException, AxisFault {

        Object remoteIP = messageContext.getProperty(MessageContext.REMOTE_ADDR);
        if (remoteIP == null) {
            throw new AxisFault("Could not find the IP address of the caller - Currently only support caller-IP base access control");
        } else {
            ThrottleContext throttleContext =
                    throttle.getThrottleContext(ThrottleConstants.IP_BASED_THROTTLE_KEY);
            AccessController accessController = AccessControllerFactory.createAccessControler(ThrottleConstants.IP_BASE);
            if (!accessController.canAccess(throttleContext, remoteIP)) {
                throw new AxisFault("You cannot access this service since you have" +
                        " exceeded the allocated quota.");
            }
        }
    }

    public InvocationResponse invoke(MessageContext msgContext) throws AxisFault {
        //Load throttle
        try {
            Throttle throttle = loadThrottle(msgContext, getThrottleType());
            if (throttle != null) {
                process(throttle, msgContext);
            } else {
                log.warn("Cannot find throttle configuration.");
            }
        }
        catch (ThrottleException e) {
            log.error(e.getMessage());
            throw new AxisFault(e.getMessage());
        }
        return InvocationResponse.CONTINUE;
    }

}
