/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dlm.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * <b>[Custom snapshot policies that target instances only]</b> Information about pre and/or post scripts for a snapshot
 * lifecycle policy that targets instances. For more information, see <a
 * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/automate-app-consistent-backups.html"> Automating
 * application-consistent snapshots with pre and post scripts</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Script implements SdkPojo, Serializable, ToCopyableBuilder<Script.Builder, Script> {
    private static final SdkField<List<String>> STAGES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Stages")
            .getter(getter(Script::stagesAsStrings))
            .setter(setter(Builder::stagesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Stages").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> EXECUTION_HANDLER_SERVICE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ExecutionHandlerService").getter(getter(Script::executionHandlerServiceAsString))
            .setter(setter(Builder::executionHandlerService))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExecutionHandlerService").build())
            .build();

    private static final SdkField<String> EXECUTION_HANDLER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ExecutionHandler").getter(getter(Script::executionHandler)).setter(setter(Builder::executionHandler))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExecutionHandler").build()).build();

    private static final SdkField<Boolean> EXECUTE_OPERATION_ON_SCRIPT_FAILURE_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("ExecuteOperationOnScriptFailure")
            .getter(getter(Script::executeOperationOnScriptFailure))
            .setter(setter(Builder::executeOperationOnScriptFailure))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExecuteOperationOnScriptFailure")
                    .build()).build();

    private static final SdkField<Integer> EXECUTION_TIMEOUT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("ExecutionTimeout").getter(getter(Script::executionTimeout)).setter(setter(Builder::executionTimeout))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExecutionTimeout").build()).build();

    private static final SdkField<Integer> MAXIMUM_RETRY_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MaximumRetryCount").getter(getter(Script::maximumRetryCount)).setter(setter(Builder::maximumRetryCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaximumRetryCount").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STAGES_FIELD,
            EXECUTION_HANDLER_SERVICE_FIELD, EXECUTION_HANDLER_FIELD, EXECUTE_OPERATION_ON_SCRIPT_FAILURE_FIELD,
            EXECUTION_TIMEOUT_FIELD, MAXIMUM_RETRY_COUNT_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final List<String> stages;

    private final String executionHandlerService;

    private final String executionHandler;

    private final Boolean executeOperationOnScriptFailure;

    private final Integer executionTimeout;

    private final Integer maximumRetryCount;

    private Script(BuilderImpl builder) {
        this.stages = builder.stages;
        this.executionHandlerService = builder.executionHandlerService;
        this.executionHandler = builder.executionHandler;
        this.executeOperationOnScriptFailure = builder.executeOperationOnScriptFailure;
        this.executionTimeout = builder.executionTimeout;
        this.maximumRetryCount = builder.maximumRetryCount;
    }

    /**
     * <p>
     * Indicate which scripts Amazon Data Lifecycle Manager should run on target instances. Pre scripts run before
     * Amazon Data Lifecycle Manager initiates snapshot creation. Post scripts run after Amazon Data Lifecycle Manager
     * initiates snapshot creation.
     * </p>
     * <ul>
     * <li>
     * <p>
     * To run a pre script only, specify <code>PRE</code>. In this case, Amazon Data Lifecycle Manager calls the SSM
     * document with the <code>pre-script</code> parameter before initiating snapshot creation.
     * </p>
     * </li>
     * <li>
     * <p>
     * To run a post script only, specify <code>POST</code>. In this case, Amazon Data Lifecycle Manager calls the SSM
     * document with the <code>post-script</code> parameter after initiating snapshot creation.
     * </p>
     * </li>
     * <li>
     * <p>
     * To run both pre and post scripts, specify both <code>PRE</code> and <code>POST</code>. In this case, Amazon Data
     * Lifecycle Manager calls the SSM document with the <code>pre-script</code> parameter before initiating snapshot
     * creation, and then it calls the SSM document again with the <code>post-script</code> parameter after initiating
     * snapshot creation.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If you are automating VSS Backups, omit this parameter.
     * </p>
     * <p>
     * Default: PRE and POST
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasStages} method.
     * </p>
     * 
     * @return Indicate which scripts Amazon Data Lifecycle Manager should run on target instances. Pre scripts run
     *         before Amazon Data Lifecycle Manager initiates snapshot creation. Post scripts run after Amazon Data
     *         Lifecycle Manager initiates snapshot creation.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         To run a pre script only, specify <code>PRE</code>. In this case, Amazon Data Lifecycle Manager calls the
     *         SSM document with the <code>pre-script</code> parameter before initiating snapshot creation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To run a post script only, specify <code>POST</code>. In this case, Amazon Data Lifecycle Manager calls
     *         the SSM document with the <code>post-script</code> parameter after initiating snapshot creation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To run both pre and post scripts, specify both <code>PRE</code> and <code>POST</code>. In this case,
     *         Amazon Data Lifecycle Manager calls the SSM document with the <code>pre-script</code> parameter before
     *         initiating snapshot creation, and then it calls the SSM document again with the <code>post-script</code>
     *         parameter after initiating snapshot creation.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If you are automating VSS Backups, omit this parameter.
     *         </p>
     *         <p>
     *         Default: PRE and POST
     */
    public final List<StageValues> stages() {
        return StagesListCopier.copyStringToEnum(stages);
    }

    /**
     * For responses, this returns true if the service returned a value for the Stages property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasStages() {
        return stages != null && !(stages instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Indicate which scripts Amazon Data Lifecycle Manager should run on target instances. Pre scripts run before
     * Amazon Data Lifecycle Manager initiates snapshot creation. Post scripts run after Amazon Data Lifecycle Manager
     * initiates snapshot creation.
     * </p>
     * <ul>
     * <li>
     * <p>
     * To run a pre script only, specify <code>PRE</code>. In this case, Amazon Data Lifecycle Manager calls the SSM
     * document with the <code>pre-script</code> parameter before initiating snapshot creation.
     * </p>
     * </li>
     * <li>
     * <p>
     * To run a post script only, specify <code>POST</code>. In this case, Amazon Data Lifecycle Manager calls the SSM
     * document with the <code>post-script</code> parameter after initiating snapshot creation.
     * </p>
     * </li>
     * <li>
     * <p>
     * To run both pre and post scripts, specify both <code>PRE</code> and <code>POST</code>. In this case, Amazon Data
     * Lifecycle Manager calls the SSM document with the <code>pre-script</code> parameter before initiating snapshot
     * creation, and then it calls the SSM document again with the <code>post-script</code> parameter after initiating
     * snapshot creation.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If you are automating VSS Backups, omit this parameter.
     * </p>
     * <p>
     * Default: PRE and POST
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasStages} method.
     * </p>
     * 
     * @return Indicate which scripts Amazon Data Lifecycle Manager should run on target instances. Pre scripts run
     *         before Amazon Data Lifecycle Manager initiates snapshot creation. Post scripts run after Amazon Data
     *         Lifecycle Manager initiates snapshot creation.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         To run a pre script only, specify <code>PRE</code>. In this case, Amazon Data Lifecycle Manager calls the
     *         SSM document with the <code>pre-script</code> parameter before initiating snapshot creation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To run a post script only, specify <code>POST</code>. In this case, Amazon Data Lifecycle Manager calls
     *         the SSM document with the <code>post-script</code> parameter after initiating snapshot creation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To run both pre and post scripts, specify both <code>PRE</code> and <code>POST</code>. In this case,
     *         Amazon Data Lifecycle Manager calls the SSM document with the <code>pre-script</code> parameter before
     *         initiating snapshot creation, and then it calls the SSM document again with the <code>post-script</code>
     *         parameter after initiating snapshot creation.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If you are automating VSS Backups, omit this parameter.
     *         </p>
     *         <p>
     *         Default: PRE and POST
     */
    public final List<String> stagesAsStrings() {
        return stages;
    }

    /**
     * <p>
     * Indicates the service used to execute the pre and/or post scripts.
     * </p>
     * <ul>
     * <li>
     * <p>
     * If you are using custom SSM documents or automating application-consistent snapshots of SAP HANA workloads,
     * specify <code>AWS_SYSTEMS_MANAGER</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you are automating VSS Backups, omit this parameter.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Default: AWS_SYSTEMS_MANAGER
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #executionHandlerService} will return {@link ExecutionHandlerServiceValues#UNKNOWN_TO_SDK_VERSION}. The
     * raw value returned by the service is available from {@link #executionHandlerServiceAsString}.
     * </p>
     * 
     * @return Indicates the service used to execute the pre and/or post scripts.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         If you are using custom SSM documents or automating application-consistent snapshots of SAP HANA
     *         workloads, specify <code>AWS_SYSTEMS_MANAGER</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If you are automating VSS Backups, omit this parameter.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Default: AWS_SYSTEMS_MANAGER
     * @see ExecutionHandlerServiceValues
     */
    public final ExecutionHandlerServiceValues executionHandlerService() {
        return ExecutionHandlerServiceValues.fromValue(executionHandlerService);
    }

    /**
     * <p>
     * Indicates the service used to execute the pre and/or post scripts.
     * </p>
     * <ul>
     * <li>
     * <p>
     * If you are using custom SSM documents or automating application-consistent snapshots of SAP HANA workloads,
     * specify <code>AWS_SYSTEMS_MANAGER</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you are automating VSS Backups, omit this parameter.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Default: AWS_SYSTEMS_MANAGER
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #executionHandlerService} will return {@link ExecutionHandlerServiceValues#UNKNOWN_TO_SDK_VERSION}. The
     * raw value returned by the service is available from {@link #executionHandlerServiceAsString}.
     * </p>
     * 
     * @return Indicates the service used to execute the pre and/or post scripts.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         If you are using custom SSM documents or automating application-consistent snapshots of SAP HANA
     *         workloads, specify <code>AWS_SYSTEMS_MANAGER</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If you are automating VSS Backups, omit this parameter.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Default: AWS_SYSTEMS_MANAGER
     * @see ExecutionHandlerServiceValues
     */
    public final String executionHandlerServiceAsString() {
        return executionHandlerService;
    }

    /**
     * <p>
     * The SSM document that includes the pre and/or post scripts to run.
     * </p>
     * <ul>
     * <li>
     * <p>
     * If you are automating VSS backups, specify <code>AWS_VSS_BACKUP</code>. In this case, Amazon Data Lifecycle
     * Manager automatically uses the <code>AWSEC2-CreateVssSnapshot</code> SSM document.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you are automating application-consistent snapshots for SAP HANA workloads, specify
     * <code>AWSSystemsManagerSAP-CreateDLMSnapshotForSAPHANA</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you are using a custom SSM document that you own, specify either the name or ARN of the SSM document. If you
     * are using a custom SSM document that is shared with you, specify the ARN of the SSM document.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The SSM document that includes the pre and/or post scripts to run.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         If you are automating VSS backups, specify <code>AWS_VSS_BACKUP</code>. In this case, Amazon Data
     *         Lifecycle Manager automatically uses the <code>AWSEC2-CreateVssSnapshot</code> SSM document.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If you are automating application-consistent snapshots for SAP HANA workloads, specify
     *         <code>AWSSystemsManagerSAP-CreateDLMSnapshotForSAPHANA</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If you are using a custom SSM document that you own, specify either the name or ARN of the SSM document.
     *         If you are using a custom SSM document that is shared with you, specify the ARN of the SSM document.
     *         </p>
     *         </li>
     */
    public final String executionHandler() {
        return executionHandler;
    }

    /**
     * <p>
     * Indicates whether Amazon Data Lifecycle Manager should default to crash-consistent snapshots if the pre script
     * fails.
     * </p>
     * <ul>
     * <li>
     * <p>
     * To default to crash consistent snapshot if the pre script fails, specify <code>true</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * To skip the instance for snapshot creation if the pre script fails, specify <code>false</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * This parameter is supported only if you run a pre script. If you run a post script only, omit this parameter.
     * </p>
     * <p>
     * Default: true
     * </p>
     * 
     * @return Indicates whether Amazon Data Lifecycle Manager should default to crash-consistent snapshots if the pre
     *         script fails.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         To default to crash consistent snapshot if the pre script fails, specify <code>true</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To skip the instance for snapshot creation if the pre script fails, specify <code>false</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         This parameter is supported only if you run a pre script. If you run a post script only, omit this
     *         parameter.
     *         </p>
     *         <p>
     *         Default: true
     */
    public final Boolean executeOperationOnScriptFailure() {
        return executeOperationOnScriptFailure;
    }

    /**
     * <p>
     * Specifies a timeout period, in seconds, after which Amazon Data Lifecycle Manager fails the script run attempt if
     * it has not completed. If a script does not complete within its timeout period, Amazon Data Lifecycle Manager
     * fails the attempt. The timeout period applies to the pre and post scripts individually.
     * </p>
     * <p>
     * If you are automating VSS Backups, omit this parameter.
     * </p>
     * <p>
     * Default: 10
     * </p>
     * 
     * @return Specifies a timeout period, in seconds, after which Amazon Data Lifecycle Manager fails the script run
     *         attempt if it has not completed. If a script does not complete within its timeout period, Amazon Data
     *         Lifecycle Manager fails the attempt. The timeout period applies to the pre and post scripts individually.
     *         </p>
     *         <p>
     *         If you are automating VSS Backups, omit this parameter.
     *         </p>
     *         <p>
     *         Default: 10
     */
    public final Integer executionTimeout() {
        return executionTimeout;
    }

    /**
     * <p>
     * Specifies the number of times Amazon Data Lifecycle Manager should retry scripts that fail.
     * </p>
     * <ul>
     * <li>
     * <p>
     * If the pre script fails, Amazon Data Lifecycle Manager retries the entire snapshot creation process, including
     * running the pre and post scripts.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the post script fails, Amazon Data Lifecycle Manager retries the post script only; in this case, the pre
     * script will have completed and the snapshot might have been created.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If you do not want Amazon Data Lifecycle Manager to retry failed scripts, specify <code>0</code>.
     * </p>
     * <p>
     * Default: 0
     * </p>
     * 
     * @return Specifies the number of times Amazon Data Lifecycle Manager should retry scripts that fail.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         If the pre script fails, Amazon Data Lifecycle Manager retries the entire snapshot creation process,
     *         including running the pre and post scripts.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If the post script fails, Amazon Data Lifecycle Manager retries the post script only; in this case, the
     *         pre script will have completed and the snapshot might have been created.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If you do not want Amazon Data Lifecycle Manager to retry failed scripts, specify <code>0</code>.
     *         </p>
     *         <p>
     *         Default: 0
     */
    public final Integer maximumRetryCount() {
        return maximumRetryCount;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasStages() ? stagesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(executionHandlerServiceAsString());
        hashCode = 31 * hashCode + Objects.hashCode(executionHandler());
        hashCode = 31 * hashCode + Objects.hashCode(executeOperationOnScriptFailure());
        hashCode = 31 * hashCode + Objects.hashCode(executionTimeout());
        hashCode = 31 * hashCode + Objects.hashCode(maximumRetryCount());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Script)) {
            return false;
        }
        Script other = (Script) obj;
        return hasStages() == other.hasStages() && Objects.equals(stagesAsStrings(), other.stagesAsStrings())
                && Objects.equals(executionHandlerServiceAsString(), other.executionHandlerServiceAsString())
                && Objects.equals(executionHandler(), other.executionHandler())
                && Objects.equals(executeOperationOnScriptFailure(), other.executeOperationOnScriptFailure())
                && Objects.equals(executionTimeout(), other.executionTimeout())
                && Objects.equals(maximumRetryCount(), other.maximumRetryCount());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Script").add("Stages", hasStages() ? stagesAsStrings() : null)
                .add("ExecutionHandlerService", executionHandlerServiceAsString()).add("ExecutionHandler", executionHandler())
                .add("ExecuteOperationOnScriptFailure", executeOperationOnScriptFailure())
                .add("ExecutionTimeout", executionTimeout()).add("MaximumRetryCount", maximumRetryCount()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Stages":
            return Optional.ofNullable(clazz.cast(stagesAsStrings()));
        case "ExecutionHandlerService":
            return Optional.ofNullable(clazz.cast(executionHandlerServiceAsString()));
        case "ExecutionHandler":
            return Optional.ofNullable(clazz.cast(executionHandler()));
        case "ExecuteOperationOnScriptFailure":
            return Optional.ofNullable(clazz.cast(executeOperationOnScriptFailure()));
        case "ExecutionTimeout":
            return Optional.ofNullable(clazz.cast(executionTimeout()));
        case "MaximumRetryCount":
            return Optional.ofNullable(clazz.cast(maximumRetryCount()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Stages", STAGES_FIELD);
        map.put("ExecutionHandlerService", EXECUTION_HANDLER_SERVICE_FIELD);
        map.put("ExecutionHandler", EXECUTION_HANDLER_FIELD);
        map.put("ExecuteOperationOnScriptFailure", EXECUTE_OPERATION_ON_SCRIPT_FAILURE_FIELD);
        map.put("ExecutionTimeout", EXECUTION_TIMEOUT_FIELD);
        map.put("MaximumRetryCount", MAXIMUM_RETRY_COUNT_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Script, T> g) {
        return obj -> g.apply((Script) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Script> {
        /**
         * <p>
         * Indicate which scripts Amazon Data Lifecycle Manager should run on target instances. Pre scripts run before
         * Amazon Data Lifecycle Manager initiates snapshot creation. Post scripts run after Amazon Data Lifecycle
         * Manager initiates snapshot creation.
         * </p>
         * <ul>
         * <li>
         * <p>
         * To run a pre script only, specify <code>PRE</code>. In this case, Amazon Data Lifecycle Manager calls the SSM
         * document with the <code>pre-script</code> parameter before initiating snapshot creation.
         * </p>
         * </li>
         * <li>
         * <p>
         * To run a post script only, specify <code>POST</code>. In this case, Amazon Data Lifecycle Manager calls the
         * SSM document with the <code>post-script</code> parameter after initiating snapshot creation.
         * </p>
         * </li>
         * <li>
         * <p>
         * To run both pre and post scripts, specify both <code>PRE</code> and <code>POST</code>. In this case, Amazon
         * Data Lifecycle Manager calls the SSM document with the <code>pre-script</code> parameter before initiating
         * snapshot creation, and then it calls the SSM document again with the <code>post-script</code> parameter after
         * initiating snapshot creation.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If you are automating VSS Backups, omit this parameter.
         * </p>
         * <p>
         * Default: PRE and POST
         * </p>
         * 
         * @param stages
         *        Indicate which scripts Amazon Data Lifecycle Manager should run on target instances. Pre scripts run
         *        before Amazon Data Lifecycle Manager initiates snapshot creation. Post scripts run after Amazon Data
         *        Lifecycle Manager initiates snapshot creation.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        To run a pre script only, specify <code>PRE</code>. In this case, Amazon Data Lifecycle Manager calls
         *        the SSM document with the <code>pre-script</code> parameter before initiating snapshot creation.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To run a post script only, specify <code>POST</code>. In this case, Amazon Data Lifecycle Manager
         *        calls the SSM document with the <code>post-script</code> parameter after initiating snapshot creation.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To run both pre and post scripts, specify both <code>PRE</code> and <code>POST</code>. In this case,
         *        Amazon Data Lifecycle Manager calls the SSM document with the <code>pre-script</code> parameter before
         *        initiating snapshot creation, and then it calls the SSM document again with the
         *        <code>post-script</code> parameter after initiating snapshot creation.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If you are automating VSS Backups, omit this parameter.
         *        </p>
         *        <p>
         *        Default: PRE and POST
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stagesWithStrings(Collection<String> stages);

        /**
         * <p>
         * Indicate which scripts Amazon Data Lifecycle Manager should run on target instances. Pre scripts run before
         * Amazon Data Lifecycle Manager initiates snapshot creation. Post scripts run after Amazon Data Lifecycle
         * Manager initiates snapshot creation.
         * </p>
         * <ul>
         * <li>
         * <p>
         * To run a pre script only, specify <code>PRE</code>. In this case, Amazon Data Lifecycle Manager calls the SSM
         * document with the <code>pre-script</code> parameter before initiating snapshot creation.
         * </p>
         * </li>
         * <li>
         * <p>
         * To run a post script only, specify <code>POST</code>. In this case, Amazon Data Lifecycle Manager calls the
         * SSM document with the <code>post-script</code> parameter after initiating snapshot creation.
         * </p>
         * </li>
         * <li>
         * <p>
         * To run both pre and post scripts, specify both <code>PRE</code> and <code>POST</code>. In this case, Amazon
         * Data Lifecycle Manager calls the SSM document with the <code>pre-script</code> parameter before initiating
         * snapshot creation, and then it calls the SSM document again with the <code>post-script</code> parameter after
         * initiating snapshot creation.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If you are automating VSS Backups, omit this parameter.
         * </p>
         * <p>
         * Default: PRE and POST
         * </p>
         * 
         * @param stages
         *        Indicate which scripts Amazon Data Lifecycle Manager should run on target instances. Pre scripts run
         *        before Amazon Data Lifecycle Manager initiates snapshot creation. Post scripts run after Amazon Data
         *        Lifecycle Manager initiates snapshot creation.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        To run a pre script only, specify <code>PRE</code>. In this case, Amazon Data Lifecycle Manager calls
         *        the SSM document with the <code>pre-script</code> parameter before initiating snapshot creation.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To run a post script only, specify <code>POST</code>. In this case, Amazon Data Lifecycle Manager
         *        calls the SSM document with the <code>post-script</code> parameter after initiating snapshot creation.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To run both pre and post scripts, specify both <code>PRE</code> and <code>POST</code>. In this case,
         *        Amazon Data Lifecycle Manager calls the SSM document with the <code>pre-script</code> parameter before
         *        initiating snapshot creation, and then it calls the SSM document again with the
         *        <code>post-script</code> parameter after initiating snapshot creation.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If you are automating VSS Backups, omit this parameter.
         *        </p>
         *        <p>
         *        Default: PRE and POST
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stagesWithStrings(String... stages);

        /**
         * <p>
         * Indicate which scripts Amazon Data Lifecycle Manager should run on target instances. Pre scripts run before
         * Amazon Data Lifecycle Manager initiates snapshot creation. Post scripts run after Amazon Data Lifecycle
         * Manager initiates snapshot creation.
         * </p>
         * <ul>
         * <li>
         * <p>
         * To run a pre script only, specify <code>PRE</code>. In this case, Amazon Data Lifecycle Manager calls the SSM
         * document with the <code>pre-script</code> parameter before initiating snapshot creation.
         * </p>
         * </li>
         * <li>
         * <p>
         * To run a post script only, specify <code>POST</code>. In this case, Amazon Data Lifecycle Manager calls the
         * SSM document with the <code>post-script</code> parameter after initiating snapshot creation.
         * </p>
         * </li>
         * <li>
         * <p>
         * To run both pre and post scripts, specify both <code>PRE</code> and <code>POST</code>. In this case, Amazon
         * Data Lifecycle Manager calls the SSM document with the <code>pre-script</code> parameter before initiating
         * snapshot creation, and then it calls the SSM document again with the <code>post-script</code> parameter after
         * initiating snapshot creation.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If you are automating VSS Backups, omit this parameter.
         * </p>
         * <p>
         * Default: PRE and POST
         * </p>
         * 
         * @param stages
         *        Indicate which scripts Amazon Data Lifecycle Manager should run on target instances. Pre scripts run
         *        before Amazon Data Lifecycle Manager initiates snapshot creation. Post scripts run after Amazon Data
         *        Lifecycle Manager initiates snapshot creation.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        To run a pre script only, specify <code>PRE</code>. In this case, Amazon Data Lifecycle Manager calls
         *        the SSM document with the <code>pre-script</code> parameter before initiating snapshot creation.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To run a post script only, specify <code>POST</code>. In this case, Amazon Data Lifecycle Manager
         *        calls the SSM document with the <code>post-script</code> parameter after initiating snapshot creation.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To run both pre and post scripts, specify both <code>PRE</code> and <code>POST</code>. In this case,
         *        Amazon Data Lifecycle Manager calls the SSM document with the <code>pre-script</code> parameter before
         *        initiating snapshot creation, and then it calls the SSM document again with the
         *        <code>post-script</code> parameter after initiating snapshot creation.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If you are automating VSS Backups, omit this parameter.
         *        </p>
         *        <p>
         *        Default: PRE and POST
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stages(Collection<StageValues> stages);

        /**
         * <p>
         * Indicate which scripts Amazon Data Lifecycle Manager should run on target instances. Pre scripts run before
         * Amazon Data Lifecycle Manager initiates snapshot creation. Post scripts run after Amazon Data Lifecycle
         * Manager initiates snapshot creation.
         * </p>
         * <ul>
         * <li>
         * <p>
         * To run a pre script only, specify <code>PRE</code>. In this case, Amazon Data Lifecycle Manager calls the SSM
         * document with the <code>pre-script</code> parameter before initiating snapshot creation.
         * </p>
         * </li>
         * <li>
         * <p>
         * To run a post script only, specify <code>POST</code>. In this case, Amazon Data Lifecycle Manager calls the
         * SSM document with the <code>post-script</code> parameter after initiating snapshot creation.
         * </p>
         * </li>
         * <li>
         * <p>
         * To run both pre and post scripts, specify both <code>PRE</code> and <code>POST</code>. In this case, Amazon
         * Data Lifecycle Manager calls the SSM document with the <code>pre-script</code> parameter before initiating
         * snapshot creation, and then it calls the SSM document again with the <code>post-script</code> parameter after
         * initiating snapshot creation.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If you are automating VSS Backups, omit this parameter.
         * </p>
         * <p>
         * Default: PRE and POST
         * </p>
         * 
         * @param stages
         *        Indicate which scripts Amazon Data Lifecycle Manager should run on target instances. Pre scripts run
         *        before Amazon Data Lifecycle Manager initiates snapshot creation. Post scripts run after Amazon Data
         *        Lifecycle Manager initiates snapshot creation.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        To run a pre script only, specify <code>PRE</code>. In this case, Amazon Data Lifecycle Manager calls
         *        the SSM document with the <code>pre-script</code> parameter before initiating snapshot creation.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To run a post script only, specify <code>POST</code>. In this case, Amazon Data Lifecycle Manager
         *        calls the SSM document with the <code>post-script</code> parameter after initiating snapshot creation.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To run both pre and post scripts, specify both <code>PRE</code> and <code>POST</code>. In this case,
         *        Amazon Data Lifecycle Manager calls the SSM document with the <code>pre-script</code> parameter before
         *        initiating snapshot creation, and then it calls the SSM document again with the
         *        <code>post-script</code> parameter after initiating snapshot creation.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If you are automating VSS Backups, omit this parameter.
         *        </p>
         *        <p>
         *        Default: PRE and POST
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stages(StageValues... stages);

        /**
         * <p>
         * Indicates the service used to execute the pre and/or post scripts.
         * </p>
         * <ul>
         * <li>
         * <p>
         * If you are using custom SSM documents or automating application-consistent snapshots of SAP HANA workloads,
         * specify <code>AWS_SYSTEMS_MANAGER</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * If you are automating VSS Backups, omit this parameter.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Default: AWS_SYSTEMS_MANAGER
         * </p>
         * 
         * @param executionHandlerService
         *        Indicates the service used to execute the pre and/or post scripts.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        If you are using custom SSM documents or automating application-consistent snapshots of SAP HANA
         *        workloads, specify <code>AWS_SYSTEMS_MANAGER</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you are automating VSS Backups, omit this parameter.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Default: AWS_SYSTEMS_MANAGER
         * @see ExecutionHandlerServiceValues
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ExecutionHandlerServiceValues
         */
        Builder executionHandlerService(String executionHandlerService);

        /**
         * <p>
         * Indicates the service used to execute the pre and/or post scripts.
         * </p>
         * <ul>
         * <li>
         * <p>
         * If you are using custom SSM documents or automating application-consistent snapshots of SAP HANA workloads,
         * specify <code>AWS_SYSTEMS_MANAGER</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * If you are automating VSS Backups, omit this parameter.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Default: AWS_SYSTEMS_MANAGER
         * </p>
         * 
         * @param executionHandlerService
         *        Indicates the service used to execute the pre and/or post scripts.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        If you are using custom SSM documents or automating application-consistent snapshots of SAP HANA
         *        workloads, specify <code>AWS_SYSTEMS_MANAGER</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you are automating VSS Backups, omit this parameter.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Default: AWS_SYSTEMS_MANAGER
         * @see ExecutionHandlerServiceValues
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ExecutionHandlerServiceValues
         */
        Builder executionHandlerService(ExecutionHandlerServiceValues executionHandlerService);

        /**
         * <p>
         * The SSM document that includes the pre and/or post scripts to run.
         * </p>
         * <ul>
         * <li>
         * <p>
         * If you are automating VSS backups, specify <code>AWS_VSS_BACKUP</code>. In this case, Amazon Data Lifecycle
         * Manager automatically uses the <code>AWSEC2-CreateVssSnapshot</code> SSM document.
         * </p>
         * </li>
         * <li>
         * <p>
         * If you are automating application-consistent snapshots for SAP HANA workloads, specify
         * <code>AWSSystemsManagerSAP-CreateDLMSnapshotForSAPHANA</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * If you are using a custom SSM document that you own, specify either the name or ARN of the SSM document. If
         * you are using a custom SSM document that is shared with you, specify the ARN of the SSM document.
         * </p>
         * </li>
         * </ul>
         * 
         * @param executionHandler
         *        The SSM document that includes the pre and/or post scripts to run.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        If you are automating VSS backups, specify <code>AWS_VSS_BACKUP</code>. In this case, Amazon Data
         *        Lifecycle Manager automatically uses the <code>AWSEC2-CreateVssSnapshot</code> SSM document.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you are automating application-consistent snapshots for SAP HANA workloads, specify
         *        <code>AWSSystemsManagerSAP-CreateDLMSnapshotForSAPHANA</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you are using a custom SSM document that you own, specify either the name or ARN of the SSM
         *        document. If you are using a custom SSM document that is shared with you, specify the ARN of the SSM
         *        document.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder executionHandler(String executionHandler);

        /**
         * <p>
         * Indicates whether Amazon Data Lifecycle Manager should default to crash-consistent snapshots if the pre
         * script fails.
         * </p>
         * <ul>
         * <li>
         * <p>
         * To default to crash consistent snapshot if the pre script fails, specify <code>true</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * To skip the instance for snapshot creation if the pre script fails, specify <code>false</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * This parameter is supported only if you run a pre script. If you run a post script only, omit this parameter.
         * </p>
         * <p>
         * Default: true
         * </p>
         * 
         * @param executeOperationOnScriptFailure
         *        Indicates whether Amazon Data Lifecycle Manager should default to crash-consistent snapshots if the
         *        pre script fails.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        To default to crash consistent snapshot if the pre script fails, specify <code>true</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To skip the instance for snapshot creation if the pre script fails, specify <code>false</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        This parameter is supported only if you run a pre script. If you run a post script only, omit this
         *        parameter.
         *        </p>
         *        <p>
         *        Default: true
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder executeOperationOnScriptFailure(Boolean executeOperationOnScriptFailure);

        /**
         * <p>
         * Specifies a timeout period, in seconds, after which Amazon Data Lifecycle Manager fails the script run
         * attempt if it has not completed. If a script does not complete within its timeout period, Amazon Data
         * Lifecycle Manager fails the attempt. The timeout period applies to the pre and post scripts individually.
         * </p>
         * <p>
         * If you are automating VSS Backups, omit this parameter.
         * </p>
         * <p>
         * Default: 10
         * </p>
         * 
         * @param executionTimeout
         *        Specifies a timeout period, in seconds, after which Amazon Data Lifecycle Manager fails the script run
         *        attempt if it has not completed. If a script does not complete within its timeout period, Amazon Data
         *        Lifecycle Manager fails the attempt. The timeout period applies to the pre and post scripts
         *        individually. </p>
         *        <p>
         *        If you are automating VSS Backups, omit this parameter.
         *        </p>
         *        <p>
         *        Default: 10
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder executionTimeout(Integer executionTimeout);

        /**
         * <p>
         * Specifies the number of times Amazon Data Lifecycle Manager should retry scripts that fail.
         * </p>
         * <ul>
         * <li>
         * <p>
         * If the pre script fails, Amazon Data Lifecycle Manager retries the entire snapshot creation process,
         * including running the pre and post scripts.
         * </p>
         * </li>
         * <li>
         * <p>
         * If the post script fails, Amazon Data Lifecycle Manager retries the post script only; in this case, the pre
         * script will have completed and the snapshot might have been created.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If you do not want Amazon Data Lifecycle Manager to retry failed scripts, specify <code>0</code>.
         * </p>
         * <p>
         * Default: 0
         * </p>
         * 
         * @param maximumRetryCount
         *        Specifies the number of times Amazon Data Lifecycle Manager should retry scripts that fail.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        If the pre script fails, Amazon Data Lifecycle Manager retries the entire snapshot creation process,
         *        including running the pre and post scripts.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If the post script fails, Amazon Data Lifecycle Manager retries the post script only; in this case,
         *        the pre script will have completed and the snapshot might have been created.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If you do not want Amazon Data Lifecycle Manager to retry failed scripts, specify <code>0</code>.
         *        </p>
         *        <p>
         *        Default: 0
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maximumRetryCount(Integer maximumRetryCount);
    }

    static final class BuilderImpl implements Builder {
        private List<String> stages = DefaultSdkAutoConstructList.getInstance();

        private String executionHandlerService;

        private String executionHandler;

        private Boolean executeOperationOnScriptFailure;

        private Integer executionTimeout;

        private Integer maximumRetryCount;

        private BuilderImpl() {
        }

        private BuilderImpl(Script model) {
            stagesWithStrings(model.stages);
            executionHandlerService(model.executionHandlerService);
            executionHandler(model.executionHandler);
            executeOperationOnScriptFailure(model.executeOperationOnScriptFailure);
            executionTimeout(model.executionTimeout);
            maximumRetryCount(model.maximumRetryCount);
        }

        public final Collection<String> getStages() {
            if (stages instanceof SdkAutoConstructList) {
                return null;
            }
            return stages;
        }

        public final void setStages(Collection<String> stages) {
            this.stages = StagesListCopier.copy(stages);
        }

        @Override
        public final Builder stagesWithStrings(Collection<String> stages) {
            this.stages = StagesListCopier.copy(stages);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder stagesWithStrings(String... stages) {
            stagesWithStrings(Arrays.asList(stages));
            return this;
        }

        @Override
        public final Builder stages(Collection<StageValues> stages) {
            this.stages = StagesListCopier.copyEnumToString(stages);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder stages(StageValues... stages) {
            stages(Arrays.asList(stages));
            return this;
        }

        public final String getExecutionHandlerService() {
            return executionHandlerService;
        }

        public final void setExecutionHandlerService(String executionHandlerService) {
            this.executionHandlerService = executionHandlerService;
        }

        @Override
        public final Builder executionHandlerService(String executionHandlerService) {
            this.executionHandlerService = executionHandlerService;
            return this;
        }

        @Override
        public final Builder executionHandlerService(ExecutionHandlerServiceValues executionHandlerService) {
            this.executionHandlerService(executionHandlerService == null ? null : executionHandlerService.toString());
            return this;
        }

        public final String getExecutionHandler() {
            return executionHandler;
        }

        public final void setExecutionHandler(String executionHandler) {
            this.executionHandler = executionHandler;
        }

        @Override
        public final Builder executionHandler(String executionHandler) {
            this.executionHandler = executionHandler;
            return this;
        }

        public final Boolean getExecuteOperationOnScriptFailure() {
            return executeOperationOnScriptFailure;
        }

        public final void setExecuteOperationOnScriptFailure(Boolean executeOperationOnScriptFailure) {
            this.executeOperationOnScriptFailure = executeOperationOnScriptFailure;
        }

        @Override
        public final Builder executeOperationOnScriptFailure(Boolean executeOperationOnScriptFailure) {
            this.executeOperationOnScriptFailure = executeOperationOnScriptFailure;
            return this;
        }

        public final Integer getExecutionTimeout() {
            return executionTimeout;
        }

        public final void setExecutionTimeout(Integer executionTimeout) {
            this.executionTimeout = executionTimeout;
        }

        @Override
        public final Builder executionTimeout(Integer executionTimeout) {
            this.executionTimeout = executionTimeout;
            return this;
        }

        public final Integer getMaximumRetryCount() {
            return maximumRetryCount;
        }

        public final void setMaximumRetryCount(Integer maximumRetryCount) {
            this.maximumRetryCount = maximumRetryCount;
        }

        @Override
        public final Builder maximumRetryCount(Integer maximumRetryCount) {
            this.maximumRetryCount = maximumRetryCount;
            return this;
        }

        @Override
        public Script build() {
            return new Script(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
