/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.eks.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Request to update the configuration of the compute capability of your EKS Auto Mode cluster. For example, enable the
 * capability. For more information, see EKS Auto Mode compute capability in the <i>Amazon EKS User Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ComputeConfigRequest implements SdkPojo, Serializable,
        ToCopyableBuilder<ComputeConfigRequest.Builder, ComputeConfigRequest> {
    private static final SdkField<Boolean> ENABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("enabled").getter(getter(ComputeConfigRequest::enabled)).setter(setter(Builder::enabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("enabled").build()).build();

    private static final SdkField<List<String>> NODE_POOLS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("nodePools")
            .getter(getter(ComputeConfigRequest::nodePools))
            .setter(setter(Builder::nodePools))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("nodePools").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> NODE_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("nodeRoleArn").getter(getter(ComputeConfigRequest::nodeRoleArn)).setter(setter(Builder::nodeRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("nodeRoleArn").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ENABLED_FIELD,
            NODE_POOLS_FIELD, NODE_ROLE_ARN_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Boolean enabled;

    private final List<String> nodePools;

    private final String nodeRoleArn;

    private ComputeConfigRequest(BuilderImpl builder) {
        this.enabled = builder.enabled;
        this.nodePools = builder.nodePools;
        this.nodeRoleArn = builder.nodeRoleArn;
    }

    /**
     * <p>
     * Request to enable or disable the compute capability on your EKS Auto Mode cluster. If the compute capability is
     * enabled, EKS Auto Mode will create and delete EC2 Managed Instances in your Amazon Web Services account.
     * </p>
     * 
     * @return Request to enable or disable the compute capability on your EKS Auto Mode cluster. If the compute
     *         capability is enabled, EKS Auto Mode will create and delete EC2 Managed Instances in your Amazon Web
     *         Services account.
     */
    public final Boolean enabled() {
        return enabled;
    }

    /**
     * For responses, this returns true if the service returned a value for the NodePools property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasNodePools() {
        return nodePools != null && !(nodePools instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Configuration for node pools that defines the compute resources for your EKS Auto Mode cluster. For more
     * information, see EKS Auto Mode Node Pools in the <i>Amazon EKS User Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasNodePools} method.
     * </p>
     * 
     * @return Configuration for node pools that defines the compute resources for your EKS Auto Mode cluster. For more
     *         information, see EKS Auto Mode Node Pools in the <i>Amazon EKS User Guide</i>.
     */
    public final List<String> nodePools() {
        return nodePools;
    }

    /**
     * <p>
     * The ARN of the IAM Role EKS will assign to EC2 Managed Instances in your EKS Auto Mode cluster. This value cannot
     * be changed after the compute capability of EKS Auto Mode is enabled. For more information, see the IAM Reference
     * in the <i>Amazon EKS User Guide</i>.
     * </p>
     * 
     * @return The ARN of the IAM Role EKS will assign to EC2 Managed Instances in your EKS Auto Mode cluster. This
     *         value cannot be changed after the compute capability of EKS Auto Mode is enabled. For more information,
     *         see the IAM Reference in the <i>Amazon EKS User Guide</i>.
     */
    public final String nodeRoleArn() {
        return nodeRoleArn;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(enabled());
        hashCode = 31 * hashCode + Objects.hashCode(hasNodePools() ? nodePools() : null);
        hashCode = 31 * hashCode + Objects.hashCode(nodeRoleArn());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ComputeConfigRequest)) {
            return false;
        }
        ComputeConfigRequest other = (ComputeConfigRequest) obj;
        return Objects.equals(enabled(), other.enabled()) && hasNodePools() == other.hasNodePools()
                && Objects.equals(nodePools(), other.nodePools()) && Objects.equals(nodeRoleArn(), other.nodeRoleArn());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ComputeConfigRequest").add("Enabled", enabled())
                .add("NodePools", hasNodePools() ? nodePools() : null).add("NodeRoleArn", nodeRoleArn()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "enabled":
            return Optional.ofNullable(clazz.cast(enabled()));
        case "nodePools":
            return Optional.ofNullable(clazz.cast(nodePools()));
        case "nodeRoleArn":
            return Optional.ofNullable(clazz.cast(nodeRoleArn()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("enabled", ENABLED_FIELD);
        map.put("nodePools", NODE_POOLS_FIELD);
        map.put("nodeRoleArn", NODE_ROLE_ARN_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ComputeConfigRequest, T> g) {
        return obj -> g.apply((ComputeConfigRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ComputeConfigRequest> {
        /**
         * <p>
         * Request to enable or disable the compute capability on your EKS Auto Mode cluster. If the compute capability
         * is enabled, EKS Auto Mode will create and delete EC2 Managed Instances in your Amazon Web Services account.
         * </p>
         * 
         * @param enabled
         *        Request to enable or disable the compute capability on your EKS Auto Mode cluster. If the compute
         *        capability is enabled, EKS Auto Mode will create and delete EC2 Managed Instances in your Amazon Web
         *        Services account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enabled(Boolean enabled);

        /**
         * <p>
         * Configuration for node pools that defines the compute resources for your EKS Auto Mode cluster. For more
         * information, see EKS Auto Mode Node Pools in the <i>Amazon EKS User Guide</i>.
         * </p>
         * 
         * @param nodePools
         *        Configuration for node pools that defines the compute resources for your EKS Auto Mode cluster. For
         *        more information, see EKS Auto Mode Node Pools in the <i>Amazon EKS User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nodePools(Collection<String> nodePools);

        /**
         * <p>
         * Configuration for node pools that defines the compute resources for your EKS Auto Mode cluster. For more
         * information, see EKS Auto Mode Node Pools in the <i>Amazon EKS User Guide</i>.
         * </p>
         * 
         * @param nodePools
         *        Configuration for node pools that defines the compute resources for your EKS Auto Mode cluster. For
         *        more information, see EKS Auto Mode Node Pools in the <i>Amazon EKS User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nodePools(String... nodePools);

        /**
         * <p>
         * The ARN of the IAM Role EKS will assign to EC2 Managed Instances in your EKS Auto Mode cluster. This value
         * cannot be changed after the compute capability of EKS Auto Mode is enabled. For more information, see the IAM
         * Reference in the <i>Amazon EKS User Guide</i>.
         * </p>
         * 
         * @param nodeRoleArn
         *        The ARN of the IAM Role EKS will assign to EC2 Managed Instances in your EKS Auto Mode cluster. This
         *        value cannot be changed after the compute capability of EKS Auto Mode is enabled. For more
         *        information, see the IAM Reference in the <i>Amazon EKS User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nodeRoleArn(String nodeRoleArn);
    }

    static final class BuilderImpl implements Builder {
        private Boolean enabled;

        private List<String> nodePools = DefaultSdkAutoConstructList.getInstance();

        private String nodeRoleArn;

        private BuilderImpl() {
        }

        private BuilderImpl(ComputeConfigRequest model) {
            enabled(model.enabled);
            nodePools(model.nodePools);
            nodeRoleArn(model.nodeRoleArn);
        }

        public final Boolean getEnabled() {
            return enabled;
        }

        public final void setEnabled(Boolean enabled) {
            this.enabled = enabled;
        }

        @Override
        public final Builder enabled(Boolean enabled) {
            this.enabled = enabled;
            return this;
        }

        public final Collection<String> getNodePools() {
            if (nodePools instanceof SdkAutoConstructList) {
                return null;
            }
            return nodePools;
        }

        public final void setNodePools(Collection<String> nodePools) {
            this.nodePools = StringListCopier.copy(nodePools);
        }

        @Override
        public final Builder nodePools(Collection<String> nodePools) {
            this.nodePools = StringListCopier.copy(nodePools);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder nodePools(String... nodePools) {
            nodePools(Arrays.asList(nodePools));
            return this;
        }

        public final String getNodeRoleArn() {
            return nodeRoleArn;
        }

        public final void setNodeRoleArn(String nodeRoleArn) {
            this.nodeRoleArn = nodeRoleArn;
        }

        @Override
        public final Builder nodeRoleArn(String nodeRoleArn) {
            this.nodeRoleArn = nodeRoleArn;
            return this;
        }

        @Override
        public ComputeConfigRequest build() {
            return new ComputeConfigRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
