/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sfn.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes potential issues found during state machine validation. Rather than raise an exception, validation will
 * return a list of <b>diagnostic elements</b> containing diagnostic information.
 * </p>
 * <note>
 * <p>
 * The <a href="https://docs.aws.amazon.com/step-functions/latest/apireference/API_ValidateStateMachineDefinition.html">
 * ValidateStateMachineDefinitionlAPI</a> might add new diagnostics in the future, adjust diagnostic codes, or change
 * the message wording. Your automated processes should only rely on the value of the <b>result</b> field value (OK,
 * FAIL). Do <b>not</b> rely on the exact order, count, or wording of diagnostic messages.
 * </p>
 * </note>
 * <p>
 * <b>List of warning codes</b>
 * </p>
 * <dl>
 * <dt>NO_DOLLAR</dt>
 * <dd>
 * <p>
 * No <code>.$</code> on a field that appears to be a JSONPath or Intrinsic Function.
 * </p>
 * </dd>
 * <dt>NO_PATH</dt>
 * <dd>
 * <p>
 * Field value looks like a path, but field name does not end with 'Path'.
 * </p>
 * </dd>
 * <dt>PASS_RESULT_IS_STATIC</dt>
 * <dd>
 * <p>
 * Attempt to use a path in the result of a pass state.
 * </p>
 * </dd>
 * </dl>
 * <p>
 * <b>List of error codes</b>
 * </p>
 * <dl>
 * <dt>INVALID_JSON_DESCRIPTION</dt>
 * <dd>
 * <p>
 * JSON syntax problem found.
 * </p>
 * </dd>
 * <dt>MISSING_DESCRIPTION</dt>
 * <dd>
 * <p>
 * Received a null or empty workflow input.
 * </p>
 * </dd>
 * <dt>SCHEMA_VALIDATION_FAILED</dt>
 * <dd>
 * <p>
 * Schema validation reported errors.
 * </p>
 * </dd>
 * <dt>INVALID_RESOURCE</dt>
 * <dd>
 * <p>
 * The value of a Task-state resource field is invalid.
 * </p>
 * </dd>
 * <dt>MISSING_END_STATE</dt>
 * <dd>
 * <p>
 * The workflow does not have a terminal state.
 * </p>
 * </dd>
 * <dt>DUPLICATE_STATE_NAME</dt>
 * <dd>
 * <p>
 * The same state name appears more than once.
 * </p>
 * </dd>
 * <dt>INVALID_STATE_NAME</dt>
 * <dd>
 * <p>
 * The state name does not follow the naming convention.
 * </p>
 * </dd>
 * <dt>STATE_MACHINE_NAME_EMPTY</dt>
 * <dd>
 * <p>
 * The state machine name has not been specified.
 * </p>
 * </dd>
 * <dt>STATE_MACHINE_NAME_INVALID</dt>
 * <dd>
 * <p>
 * The state machine name does not follow the naming convention.
 * </p>
 * </dd>
 * <dt>STATE_MACHINE_NAME_TOO_LONG</dt>
 * <dd>
 * <p>
 * The state name exceeds the allowed length.
 * </p>
 * </dd>
 * <dt>STATE_MACHINE_NAME_ALREADY_EXISTS</dt>
 * <dd>
 * <p>
 * The state name already exists.
 * </p>
 * </dd>
 * <dt>DUPLICATE_LABEL_NAME</dt>
 * <dd>
 * <p>
 * A label name appears more than once.
 * </p>
 * </dd>
 * <dt>INVALID_LABEL_NAME</dt>
 * <dd>
 * <p>
 * You have provided an invalid label name.
 * </p>
 * </dd>
 * <dt>MISSING_TRANSITION_TARGET</dt>
 * <dd>
 * <p>
 * The value of "Next" field doesn't match a known state name.
 * </p>
 * </dd>
 * <dt>TOO_DEEPLY_NESTED</dt>
 * <dd>
 * <p>
 * The states are too deeply nested.
 * </p>
 * </dd>
 * </dl>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ValidateStateMachineDefinitionDiagnostic implements SdkPojo, Serializable,
        ToCopyableBuilder<ValidateStateMachineDefinitionDiagnostic.Builder, ValidateStateMachineDefinitionDiagnostic> {
    private static final SdkField<String> SEVERITY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("severity").getter(getter(ValidateStateMachineDefinitionDiagnostic::severityAsString))
            .setter(setter(Builder::severity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("severity").build()).build();

    private static final SdkField<String> CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("code")
            .getter(getter(ValidateStateMachineDefinitionDiagnostic::code)).setter(setter(Builder::code))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("code").build()).build();

    private static final SdkField<String> MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("message")
            .getter(getter(ValidateStateMachineDefinitionDiagnostic::message)).setter(setter(Builder::message))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("message").build()).build();

    private static final SdkField<String> LOCATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("location").getter(getter(ValidateStateMachineDefinitionDiagnostic::location))
            .setter(setter(Builder::location))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("location").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SEVERITY_FIELD, CODE_FIELD,
            MESSAGE_FIELD, LOCATION_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("severity", SEVERITY_FIELD);
                    put("code", CODE_FIELD);
                    put("message", MESSAGE_FIELD);
                    put("location", LOCATION_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final String severity;

    private final String code;

    private final String message;

    private final String location;

    private ValidateStateMachineDefinitionDiagnostic(BuilderImpl builder) {
        this.severity = builder.severity;
        this.code = builder.code;
        this.message = builder.message;
        this.location = builder.location;
    }

    /**
     * <p>
     * A value of <code>ERROR</code> means that you cannot create or update a state machine with this definition.
     * </p>
     * <p>
     * <code>WARNING</code> level diagnostics alert you to potential issues, but they will not prevent you from creating
     * or updating your state machine.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #severity} will
     * return {@link ValidateStateMachineDefinitionSeverity#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #severityAsString}.
     * </p>
     * 
     * @return A value of <code>ERROR</code> means that you cannot create or update a state machine with this
     *         definition.</p>
     *         <p>
     *         <code>WARNING</code> level diagnostics alert you to potential issues, but they will not prevent you from
     *         creating or updating your state machine.
     * @see ValidateStateMachineDefinitionSeverity
     */
    public final ValidateStateMachineDefinitionSeverity severity() {
        return ValidateStateMachineDefinitionSeverity.fromValue(severity);
    }

    /**
     * <p>
     * A value of <code>ERROR</code> means that you cannot create or update a state machine with this definition.
     * </p>
     * <p>
     * <code>WARNING</code> level diagnostics alert you to potential issues, but they will not prevent you from creating
     * or updating your state machine.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #severity} will
     * return {@link ValidateStateMachineDefinitionSeverity#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #severityAsString}.
     * </p>
     * 
     * @return A value of <code>ERROR</code> means that you cannot create or update a state machine with this
     *         definition.</p>
     *         <p>
     *         <code>WARNING</code> level diagnostics alert you to potential issues, but they will not prevent you from
     *         creating or updating your state machine.
     * @see ValidateStateMachineDefinitionSeverity
     */
    public final String severityAsString() {
        return severity;
    }

    /**
     * <p>
     * Identifying code for the diagnostic.
     * </p>
     * 
     * @return Identifying code for the diagnostic.
     */
    public final String code() {
        return code;
    }

    /**
     * <p>
     * Message describing the diagnostic condition.
     * </p>
     * 
     * @return Message describing the diagnostic condition.
     */
    public final String message() {
        return message;
    }

    /**
     * <p>
     * Location of the issue in the state machine, if available.
     * </p>
     * <p>
     * For errors specific to a field, the location could be in the format:
     * <code>/States/&lt;StateName&gt;/&lt;FieldName&gt;</code>, for example: <code>/States/FailState/ErrorPath</code>.
     * </p>
     * 
     * @return Location of the issue in the state machine, if available.</p>
     *         <p>
     *         For errors specific to a field, the location could be in the format:
     *         <code>/States/&lt;StateName&gt;/&lt;FieldName&gt;</code>, for example:
     *         <code>/States/FailState/ErrorPath</code>.
     */
    public final String location() {
        return location;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(severityAsString());
        hashCode = 31 * hashCode + Objects.hashCode(code());
        hashCode = 31 * hashCode + Objects.hashCode(message());
        hashCode = 31 * hashCode + Objects.hashCode(location());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ValidateStateMachineDefinitionDiagnostic)) {
            return false;
        }
        ValidateStateMachineDefinitionDiagnostic other = (ValidateStateMachineDefinitionDiagnostic) obj;
        return Objects.equals(severityAsString(), other.severityAsString()) && Objects.equals(code(), other.code())
                && Objects.equals(message(), other.message()) && Objects.equals(location(), other.location());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ValidateStateMachineDefinitionDiagnostic").add("Severity", severityAsString())
                .add("Code", code() == null ? null : "*** Sensitive Data Redacted ***")
                .add("Message", message() == null ? null : "*** Sensitive Data Redacted ***")
                .add("Location", location() == null ? null : "*** Sensitive Data Redacted ***").build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "severity":
            return Optional.ofNullable(clazz.cast(severityAsString()));
        case "code":
            return Optional.ofNullable(clazz.cast(code()));
        case "message":
            return Optional.ofNullable(clazz.cast(message()));
        case "location":
            return Optional.ofNullable(clazz.cast(location()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<ValidateStateMachineDefinitionDiagnostic, T> g) {
        return obj -> g.apply((ValidateStateMachineDefinitionDiagnostic) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ValidateStateMachineDefinitionDiagnostic> {
        /**
         * <p>
         * A value of <code>ERROR</code> means that you cannot create or update a state machine with this definition.
         * </p>
         * <p>
         * <code>WARNING</code> level diagnostics alert you to potential issues, but they will not prevent you from
         * creating or updating your state machine.
         * </p>
         * 
         * @param severity
         *        A value of <code>ERROR</code> means that you cannot create or update a state machine with this
         *        definition.</p>
         *        <p>
         *        <code>WARNING</code> level diagnostics alert you to potential issues, but they will not prevent you
         *        from creating or updating your state machine.
         * @see ValidateStateMachineDefinitionSeverity
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ValidateStateMachineDefinitionSeverity
         */
        Builder severity(String severity);

        /**
         * <p>
         * A value of <code>ERROR</code> means that you cannot create or update a state machine with this definition.
         * </p>
         * <p>
         * <code>WARNING</code> level diagnostics alert you to potential issues, but they will not prevent you from
         * creating or updating your state machine.
         * </p>
         * 
         * @param severity
         *        A value of <code>ERROR</code> means that you cannot create or update a state machine with this
         *        definition.</p>
         *        <p>
         *        <code>WARNING</code> level diagnostics alert you to potential issues, but they will not prevent you
         *        from creating or updating your state machine.
         * @see ValidateStateMachineDefinitionSeverity
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ValidateStateMachineDefinitionSeverity
         */
        Builder severity(ValidateStateMachineDefinitionSeverity severity);

        /**
         * <p>
         * Identifying code for the diagnostic.
         * </p>
         * 
         * @param code
         *        Identifying code for the diagnostic.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder code(String code);

        /**
         * <p>
         * Message describing the diagnostic condition.
         * </p>
         * 
         * @param message
         *        Message describing the diagnostic condition.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder message(String message);

        /**
         * <p>
         * Location of the issue in the state machine, if available.
         * </p>
         * <p>
         * For errors specific to a field, the location could be in the format:
         * <code>/States/&lt;StateName&gt;/&lt;FieldName&gt;</code>, for example:
         * <code>/States/FailState/ErrorPath</code>.
         * </p>
         * 
         * @param location
         *        Location of the issue in the state machine, if available.</p>
         *        <p>
         *        For errors specific to a field, the location could be in the format:
         *        <code>/States/&lt;StateName&gt;/&lt;FieldName&gt;</code>, for example:
         *        <code>/States/FailState/ErrorPath</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder location(String location);
    }

    static final class BuilderImpl implements Builder {
        private String severity;

        private String code;

        private String message;

        private String location;

        private BuilderImpl() {
        }

        private BuilderImpl(ValidateStateMachineDefinitionDiagnostic model) {
            severity(model.severity);
            code(model.code);
            message(model.message);
            location(model.location);
        }

        public final String getSeverity() {
            return severity;
        }

        public final void setSeverity(String severity) {
            this.severity = severity;
        }

        @Override
        public final Builder severity(String severity) {
            this.severity = severity;
            return this;
        }

        @Override
        public final Builder severity(ValidateStateMachineDefinitionSeverity severity) {
            this.severity(severity == null ? null : severity.toString());
            return this;
        }

        public final String getCode() {
            return code;
        }

        public final void setCode(String code) {
            this.code = code;
        }

        @Override
        public final Builder code(String code) {
            this.code = code;
            return this;
        }

        public final String getMessage() {
            return message;
        }

        public final void setMessage(String message) {
            this.message = message;
        }

        @Override
        public final Builder message(String message) {
            this.message = message;
            return this;
        }

        public final String getLocation() {
            return location;
        }

        public final void setLocation(String location) {
            this.location = location;
        }

        @Override
        public final Builder location(String location) {
            this.location = location;
            return this;
        }

        @Override
        public ValidateStateMachineDefinitionDiagnostic build() {
            return new ValidateStateMachineDefinitionDiagnostic(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
